package pangea.hiagent.workpanel;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import pangea.hiagent.dto.WorkPanelEvent;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;

/**
 * 工作面板数据收集器实现
 * 负责采集Agent执行过程中的各类数据
 */
@Slf4j
@Component
public class WorkPanelDataCollector implements IWorkPanelDataCollector {
    
    /**
     * 事件列表（线程安全）
     */
    private final List<WorkPanelEvent> events = new CopyOnWriteArrayList<>();
    
    /**
     * 事件订阅者列表（线程安全）
     */
    private final List<Consumer<WorkPanelEvent>> subscribers = new CopyOnWriteArrayList<>();
    
    /**
     * 最大事件数量，防止内存溢出
     */
    private static final int MAX_EVENTS = 1000;
    
    /**
     * 根据状态确定事件类型
     */
    private String getEventTypeFromStatus(String status) {
        if (status == null) {
            return "tool_result";
        }
        
        switch (status.toLowerCase()) {
            case "success":
                return "tool_result";
            case "error":
            case "failure":
                return "tool_error";
            default:
                return "tool_result";
        }
    }
    
    @Override
    public void recordThinking(String content, String thinkingType) {
        try {
            // 过滤掉过于简短的内容，避免记录过多无关信息
            if (content == null || content.trim().length() < 1) {
                return;
            }
            
            WorkPanelEvent event = WorkPanelEvent.builder()
                    .eventType("thinking")
                    .timestamp(System.currentTimeMillis())
                    .content(content)
                    .thinkingType(thinkingType != null ? thinkingType : "reasoning")
                    .build();
            
            addEvent(event);
            log.debug("已记录思考过程: 类型={}, 内容={}", thinkingType, content);
        } catch (Exception e) {
            log.error("记录思考过程时发生错误: content={}, type={}", content, thinkingType, e);
        }
    }
    
    @Override
    public void recordToolCallStart(String toolName, String toolAction, Object input) {
        try {
            long currentTime = System.currentTimeMillis();
            WorkPanelEvent event = WorkPanelEvent.builder()
                    .eventType("tool_call")
                    .timestamp(currentTime)
                    .toolName(toolName != null ? toolName : "未知工具")
                    .toolAction(toolAction != null ? toolAction : "未知操作")
                    .toolInput(convertToMap(input))
                    .toolStatus("pending")  // 声明初始状态为pending
                    .build();
            
            addEvent(event);
            
            // 添加更详细的日志输出
            String formattedTime = new java.text.SimpleDateFormat("HH:mm:ss").format(new java.util.Date(currentTime));
            log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n📥 输入: {}\n📊 状态: 处理中", 
                     toolName != null ? toolName : "未知工具", 
                     formattedTime,
                     convertToJsonString(input));
            
            log.debug("已记录工具调用开始: 工具={}, 方法={}, 状态=pending", toolName, toolAction);
        } catch (Exception e) {
            log.error("记录工具调用开始时发生错误: toolName={}, toolAction={}", toolName, toolAction, e);
        }
    }
    
    @Override
    public void recordToolCallComplete(String toolName, Object output, String status) {
        recordToolCallComplete(toolName, output, status, null);
    }
    
    /**
     * 记录工具调用完成（带执行时间）
     */
    public void recordToolCallComplete(String toolName, Object output, String status, Long executionTime) {
        try {
            long currentTime = System.currentTimeMillis();
            // 查找最近的该工具的pending事件并更新它
            WorkPanelEvent lastToolCall = getLastPendingToolCall(toolName);
            if (lastToolCall != null) {
                // 保存原有的toolInput
                Map<String, Object> originalToolInput = lastToolCall.getToolInput();
                
                // 更新现有事件
                lastToolCall.setEventType(getEventTypeFromStatus(status)); // 更新事件类型
                lastToolCall.setToolOutput(output);
                lastToolCall.setToolStatus(status != null ? status : "unknown");
                // 如果有执行时间，设置执行时间
                if (executionTime != null) {
                    lastToolCall.setExecutionTime(executionTime);
                }
                // 更新时间戳
                lastToolCall.setTimestamp(currentTime);
                // 确保toolInput字段不会丢失
                if (originalToolInput != null) {
                    lastToolCall.setToolInput(originalToolInput);
                }
                
                // 重新发布更新后的事件
                notifySubscribers(lastToolCall);
            } else {
                // 如果没有对应的pending事件，创建一个新事件
                // 此时需要去回退查找toolInput，会避免数据丢失
                Map<String, Object> fallbackToolInput = null;
                WorkPanelEvent lastAnyToolCall = getLastPendingToolCallAny();
                if (lastAnyToolCall != null && lastAnyToolCall.getToolInput() != null) {
                    fallbackToolInput = lastAnyToolCall.getToolInput();
                    log.warn("[没有找到对应的pending事件] 工具={}, 已从上一个工具事件回退toolInput", toolName);
                } else {
                    log.warn("[没有找到对应的pending事件] 工具={}, 也没有任何可回退的pending事件", toolName);
                }
                
                // 此步骤是待录容错机制的水滴水滴，立骨待宋殊水信息
                WorkPanelEvent event = WorkPanelEvent.builder()
                        .eventType(getEventTypeFromStatus(status))
                        .timestamp(currentTime)
                        .toolName(toolName != null ? toolName : "未知工具")
                        .toolOutput(output)
                        .toolStatus(status != null ? status : "unknown")
                        .toolInput(fallbackToolInput)  // 设置回退的toolInput，可以是null
                        .executionTime(executionTime)
                        .build();
                
                addEvent(event);
            }
            
            // 添加更详细的日志输出
            String formattedTime = new java.text.SimpleDateFormat("HH:mm:ss").format(new java.util.Date(currentTime));
            String statusText = getStatusText(status);
            
            if ("success".equals(status)) {
                log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n✅ 状态: 成功\n📤 输出: {}{}", 
                         toolName != null ? toolName : "未知工具",
                         formattedTime,
                         convertToJsonString(output),
                         executionTime != null ? "\n⏱️ 耗时: " + executionTime + "ms" : "");
            } else if ("failure".equals(status) || "error".equals(status)) {
                log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n❌ 状态: 失败\n💬 错误: {}{}", 
                         toolName != null ? toolName : "未知工具",
                         formattedTime,
                         convertToJsonString(output),
                         executionTime != null ? "\n⏱️ 耗时: " + executionTime + "ms" : "");
            } else {
                log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n📊 状态: {}\n📤 输出: {}{}", 
                         toolName != null ? toolName : "未知工具",
                         formattedTime,
                         statusText,
                         convertToJsonString(output),
                         executionTime != null ? "\n⏱️ 耗时: " + executionTime + "ms" : "");
            }
            
            log.debug("已记录工具调用完成: 工具={}, 状态={}, 执行时间={}ms", toolName, status, executionTime);
        } catch (Exception e) {
            log.error("记录工具调用完成时发生错误: toolName={}, status={}", toolName, status, e);
        }
    }
    
    @Override
    public void recordToolCallError(String toolName, String errorMessage) {
        try {
            WorkPanelEvent event = WorkPanelEvent.builder()
                    .eventType("tool_error")
                    .timestamp(System.currentTimeMillis())
                    .toolName(toolName)
                    .content(errorMessage)
                    .toolStatus("failure")
                    .build();
            
            addEvent(event);
            log.debug("已记录工具调用错误: 工具={}, 错误={}", toolName, errorMessage);
        } catch (Exception e) {
            log.error("记录工具调用错误时发生错误: toolName={}", toolName, e);
        }
    }
    
    @Override
    public void recordLog(String message, String level) {
        try {
            // 过滤掉空消息
            if (message == null || message.trim().isEmpty()) {
                return;
            }
            
            WorkPanelEvent event = WorkPanelEvent.builder()
                    .eventType("log")
                    .timestamp(System.currentTimeMillis())
                    .content(message)
                    .logLevel(level != null ? level : "info")
                    .build();
            
            addEvent(event);
        } catch (Exception e) {
            log.error("记录日志时发生错误: message={}, level={}", message, level, e);
        }
    }
    
    @Override
    public void recordEmbed(String url, String type, String title, String htmlContent) {
        try {
            WorkPanelEvent event = WorkPanelEvent.builder()
                    .eventType("embed")
                    .timestamp(System.currentTimeMillis())
                    .embedUrl(url)
                    .embedType(type)
                    .embedTitle(title)
                    .embedHtmlContent(htmlContent)
                    .build();
            
            addEvent(event);
            log.debug("已记录embed事件: title={}, type={}, hasContent={}", title, type, htmlContent != null && !htmlContent.isEmpty());
        } catch (Exception e) {
            log.error("recordEmbed时发生错误: title={}, type={}", title, type, e);
        }
    }
    
    @Override
    public List<WorkPanelEvent> getEvents() {
        return new ArrayList<>(events);
    }
    
    @Override
    public void subscribe(Consumer<WorkPanelEvent> consumer) {
        if (consumer != null) {
            subscribers.add(consumer);
            log.debug("已添加事件订阅者，当前订阅者数量: {}", subscribers.size());
        }
    }
    
    @Override
    public void clear() {
        try {
            events.clear();
            log.debug("已清空工作面板事件");
        } catch (Exception e) {
            log.error("清空事件时发生错误", e);
        }
    }
    
    @Override
    public WorkPanelEvent getLastToolCall() {
        // 从后往前查找最后一个工具调用事件
        for (int i = events.size() - 1; i >= 0; i--) {
            WorkPanelEvent event = events.get(i);
            if ("tool_call".equals(event.getEventType()) || "tool_result".equals(event.getEventType())) {
                return event;
            }
        }
        return null;
    }
    
    /**
     * 查找最近的该工具的pending事件
     */
    private WorkPanelEvent getLastPendingToolCall(String toolName) {
        // 从后往前查找最近的该工具的pending事件
        for (int i = events.size() - 1; i >= 0; i--) {
            WorkPanelEvent event = events.get(i);
            if ("tool_call".equals(event.getEventType()) && 
                toolName.equals(event.getToolName()) && 
                "pending".equals(event.getToolStatus())) {
                return event;
            }
        }
        return null;
    }
    
    /**
     * 查找最近的任何工具的pending事件（容错机制）
     */
    private WorkPanelEvent getLastPendingToolCallAny() {
        // 从后往前查找最近的任何pending事件
        for (int i = events.size() - 1; i >= 0; i--) {
            WorkPanelEvent event = events.get(i);
            if ("tool_call".equals(event.getEventType()) && 
                "pending".equals(event.getToolStatus())) {
                return event;
            }
        }
        return null;
    }
    
    /**
     * 添加事件到列表，并通知所有订阅者（隔离异常避免一个订阅者异常影响其他订阅者）
     */
    private void addEvent(WorkPanelEvent event) {
        try {
            // 控制事件数量，防止内存溢出
            if (events.size() >= MAX_EVENTS) {
                events.remove(0); // 移除最老的事件
            }
            
            events.add(event);
            
            // 通知所有订阅者
            notifySubscribers(event);
        } catch (Exception e) {
            // 即使在addEvent方法中也增加异常保护，防止影响主流程
            log.debug("添加事件失败: {}", e.getMessage());
        }
    }
    
    /**
     * 通知所有订阅者（隔离异常避免一个订阅者异常影响其他订阅者）
     */
    private void notifySubscribers(WorkPanelEvent event) {
        try {
            // 通知所有订阅者（使用隔离异常处理，确保一个订阅者异常不影响其他订阅者）
            if (!subscribers.isEmpty()) {
                for (Consumer<WorkPanelEvent> subscriber : subscribers) {
                    try {
                        if (subscriber != null) {
                            subscriber.accept(event);
                        }
                    } catch (Exception e) {
                        // 异常降级为debug日志，避免过度日志记录
                        // 异常通常由于SSE连接已断开导致，这是正常情况
                        if (e instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
                            log.debug("订阅者处理事件失败：异步请求不可用（客户端已断开连接）");
                        } else if (e instanceof java.io.IOException) {
                            log.debug("订阅者处理事件失败：客户端连接已断开");
                        } else if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                            log.debug("订阅者处理事件失败：响应已提交");
                        } else {
                            // 其他异常也降级为debug，避免日志污染
                            log.debug("订阅者处理事件失败: {}", e.getMessage());
                        }
                    }
                }
            }
        } catch (Exception e) {
            // 即使在addEvent方法中也增加异常保护，防止影响主流程
            log.debug("添加事件失败: {}", e.getMessage());
        }
    }
    
    /**
     * 将对象转换为Map（用于工具输入参数）
     */
    private Map<String, Object> convertToMap(Object input) {
        if (input == null) {
            return new HashMap<>();
        }
        
        if (input instanceof Map) {
            // 安全地转换Map类型，确保键为String类型
            Map<?, ?> rawMap = (Map<?, ?>) input;
            Map<String, Object> resultMap = new HashMap<>();
            for (Map.Entry<?, ?> entry : rawMap.entrySet()) {
                // 将键转换为String类型
                String key = entry.getKey() != null ? entry.getKey().toString() : "null";
                resultMap.put(key, entry.getValue());
            }
            return resultMap;
        }
        
        // 简单对象转换为Map
        Map<String, Object> result = new HashMap<>();
        result.put("value", input);
        return result;
    }
    /**
     * 将对象转换为JSON字符串
     */
    private String convertToJsonString(Object obj) {
        try {
            if (obj == null) {
                return "null";
            }
            
            if (obj instanceof String) {
                return (String) obj;
            }
            
            // 使用Jackson ObjectMapper进行序列化
            com.fasterxml.jackson.databind.ObjectMapper mapper = new com.fasterxml.jackson.databind.ObjectMapper();
            return mapper.writeValueAsString(obj);
        } catch (Exception e) {
            if (obj != null) {
                return obj.toString();
            } else {
                return "null";
            }
        }
    }
    
    /**
     * 获取状态文本
     */
    private String getStatusText(String status) {
        if (status == null) {
            return "未知";
        }
        
        switch (status.toLowerCase()) {
            case "success": return "成功";
            case "pending": return "处理中";
            case "error": return "错误";
            case "failure": return "失败";
            default: return status;
        }
    }
    
}