package pangea.hiagent.core;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.UserMessage;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.chat.model.Generation;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.ai.chat.prompt.SystemPromptTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.react.ReactCallback;
import pangea.hiagent.react.ReactExecutor;
import pangea.hiagent.service.AgentService;
import pangea.hiagent.service.AgentToolManager;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.function.Consumer;

/**
 * 基于Spring AI ChatClient的ReAct Service类
 * 负责实现ReAct Agent的核心逻辑，使用Spring AI的标准工具调用机制
 * 集成工作面板数据收集器，捕获思考、行动、观察、结束等步骤
 */
@Slf4j
@Service
public class ReActService {
    
    /**
     * Token消费者接口，支持完成回调
     */
    public interface TokenConsumerWithCompletion extends Consumer<String> {
        /**
         * 当流式处理完成时调用
         * @param fullContent 完整的内容
         */
        default void onComplete(String fullContent) {
            // 默认实现为空
        }
    }
    
    // 添加默认构造函数以支持测试
    public ReActService() {
    }    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private RagService ragService;
    
    // 通过AgentToolManager管理工具，不再直接注入所有@Component工具类
    
    @Autowired
    private pangea.hiagent.workpanel.IWorkPanelDataCollector workPanelCollector;    
    @Autowired
    private MemoryService memoryService;
    
    @Autowired
    private ReactCallback defaultReactCallback;
    
    @Autowired
    private ReactExecutor defaultReactExecutor;
    
    @Autowired
    private AgentToolManager agentToolManager;
    
    /**
     * 处理用户请求的主方法（同步方式）
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param userId 用户ID（可选）
     * @return 处理结果
     */
    public String processRequest(Agent agent, String userMessage, String userId) {
        log.info("开始处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent.getId(), userMessage);
        
        try {
            // 为每个用户-Agent组合创建唯一的会话ID
            String sessionId = memoryService.generateSessionId(agent, userId);
            
            // 添加用户消息到ChatMemory
            memoryService.addUserMessageToMemory(sessionId, userMessage);
            
            // 检查是否启用RAG并尝试RAG增强
            String ragResponse = tryRagEnhancement(agent, userMessage);
            if (ragResponse != null) {
                // 触发最终答案回调
                if (defaultReactCallback != null) {
                    defaultReactCallback.onFinalAnswer(ragResponse);
                }
                return ragResponse;
            }
            
            // 准备执行环境
            ChatClient client = ChatClient.builder(agentService.getChatModelForAgent(agent)).build();
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            // 添加自定义回调到ReAct执行器
            if (defaultReactExecutor != null && defaultReactCallback != null) {
                defaultReactExecutor.addReactCallback(defaultReactCallback);
            }
            
            // 使用ReAct执行器执行流程，传递Agent对象以支持记忆功能
            String finalAnswer = defaultReactExecutor.executeWithAgent(client, userMessage, tools, agent);
            
            // 将助理回复添加到ChatMemory
            memoryService.addAssistantMessageToMemory(sessionId, finalAnswer);
            
            return finalAnswer;
        } catch (Exception e) {
            // 检查是否是401 Unauthorized错误
            if (isUnauthorizedError(e)) {
                log.error("LLM返回401未授权错误: {}", e.getMessage());
                return " 请配置API密钥";
            } else {
                log.error("处理ReAct请求时发生错误", e);
                return "处理请求时发生错误: " + e.getMessage();
            }
        }
    }
    
    /**
     * 处理用户请求的主方法（同步方式）- 默认不传递userId
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @return 处理结果
     */
    public String processRequest(Agent agent, String userMessage) {
        return processRequest(agent, userMessage, null);
    }
    
    // 移除了prepareChatClient、prepareTools、addReactCallbackIfNeeded和executeReactProcess方法，
    // 因为它们只是简单地封装了其他方法调用，可以直接内联到processRequest方法中以提高效率
    
    /**
     * 尝试RAG增强
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @return RAG增强结果，如果没有启用或没有结果则返回null
     */
    private String tryRagEnhancement(Agent agent, String userMessage) {
        if (Boolean.TRUE.equals(agent.getEnableRag())) {
            log.info("Agent启用RAG功能，尝试RAG增强");
            try {
                String ragResponse = ragService.ragQa(agent, userMessage);
                
                // 如果RAG有结果，直接返回
                if (ragResponse != null && !ragResponse.isEmpty()) {
                    log.info("RAG增强返回结果");
                    return saveRagResponseToMemory(agent, ragResponse);
                }
            } catch (Exception e) {
                log.error("RAG增强过程中发生错误", e);
                // RAG失败不应该影响正常的ReAct流程，继续执行
            }
        }
        return null;
    }
    
    /**
     * 保存响应到ChatMemory
     * @param agent Agent对象
     * @param response 响应内容
     * @return 响应内容
     */
    private String saveResponseToMemory(Agent agent, String response) {
        try {
            String sessionId = memoryService.generateSessionId(agent);
            // 将响应添加到ChatMemory
            memoryService.addAssistantMessageToMemory(sessionId, response);
            return response;
        } catch (Exception e) {
            log.error("保存响应到内存时发生错误", e);
            return response; // 即使保存失败，也要返回原始响应
        }
    }
    
    /**
     * 保存RAG响应到ChatMemory
     * @param agent Agent对象
     * @param ragResponse RAG响应内容
     * @return RAG响应内容
     */
    private String saveRagResponseToMemory(Agent agent, String ragResponse) {
        try {
            String sessionId = memoryService.generateSessionId(agent);
            // 将RAG响应添加到ChatMemory
            memoryService.addAssistantMessageToMemory(sessionId, "[RAG增强] " + ragResponse);
            return ragResponse;
        } catch (Exception e) {
            log.error("保存RAG响应到内存时发生错误", e);
            return ragResponse; // 即使保存失败，也要返回原始响应
        }
    }
    
    /**
     * 判断异常是否为401未授权错误
     * @param e 异常对象
     * @return 是否为401错误
     */
    private boolean isUnauthorizedError(Throwable e) {
        if (e == null) {
            return false;
        }
        
        // 检查异常消息中是否包含401 Unauthorized
        String message = e.getMessage();
        if (message != null && (message.contains("401 Unauthorized") || message.contains("Unauthorized"))) {
            return true;
        }
        
        // 递归检查cause
        return isUnauthorizedError(e.getCause());
    }
    
    /**
     * 初始化工作面板
     */
    private void initializeWorkPanel(Agent agent) {
        if (workPanelCollector != null) {
            try {
                workPanelCollector.clear();
                workPanelCollector.recordLog("开始处理Agent请求: " + agent.getName(), "info");
            } catch (Exception e) {
                log.error("初始化工作面板时发生错误", e);
            }
        }
    }
    
    /**
     * 为流式处理设置工作面板数据收集器
     */
    public void setWorkPanelEventSubscriber(java.util.function.Consumer<pangea.hiagent.dto.WorkPanelEvent> consumer) {
        // 订阅工作面板事件，用于实时推送
        if (workPanelCollector != null && consumer != null) {
            workPanelCollector.subscribe(consumer);
        }
    }    
    /**
     * 流式处理用户请求
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param tokenConsumer token处理回调函数
     * @param userId 用户ID（可选）
     */
    public void processRequestStream(Agent agent, String userMessage, java.util.function.Consumer<String> tokenConsumer, String userId) {
        log.info("开始流式处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent.getId(), userMessage);
        
        try {
            // 为每个用户-Agent组合创建唯一的会话ID
            String sessionId = memoryService.generateSessionId(agent, userId);
            
            // 添加用户消息到ChatMemory
            memoryService.addUserMessageToMemory(sessionId, userMessage);
            
            // 检查是否启用RAG并尝试RAG增强
            String ragResponse = tryRagEnhancement(agent, userMessage);
            if (ragResponse != null) {
                // 触发最终答案回调
                if (defaultReactCallback != null) {
                    defaultReactCallback.onFinalAnswer(ragResponse);
                }
                // 对于流式处理，我们需要将RAG响应作为token发送
                if (tokenConsumer != null) {
                    tokenConsumer.accept(ragResponse);
                    // 发送完成信号
                    if (tokenConsumer instanceof TokenConsumerWithCompletion) {
                        ((TokenConsumerWithCompletion) tokenConsumer).onComplete(ragResponse);
                    }
                }
                return;
            }
            
            // 准备执行环境
            ChatClient client = ChatClient.builder(agentService.getChatModelForAgent(agent)).build();
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            // 添加自定义回调到ReAct执行器
            if (defaultReactExecutor != null && defaultReactCallback != null) {
                defaultReactExecutor.addReactCallback(defaultReactCallback);
            }
            
            // 使用ReAct执行器流式执行流程，传递Agent对象以支持记忆功能
            defaultReactExecutor.executeStreamWithAgent(client, userMessage, tools, tokenConsumer, agent);
        } catch (Exception e) {
            // 检查是否是401 Unauthorized错误
            if (isUnauthorizedError(e)) {
                log.error("LLM返回401未授权错误: {}", e.getMessage());
                if (tokenConsumer != null) {
                    tokenConsumer.accept(" 请配置API密钥");
                }
            } else {
                log.error("流式处理ReAct请求时发生错误", e);
                if (tokenConsumer != null) {
                    tokenConsumer.accept("处理请求时发生错误: " + e.getMessage());
                }
            }
        }
    }
    
    /**
     * 流式处理用户请求（默认不传递userId）
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param tokenConsumer token处理回调函数
     */
    public void processRequestStream(Agent agent, String userMessage, java.util.function.Consumer<String> tokenConsumer) {
        processRequestStream(agent, userMessage, tokenConsumer, null);
    }
    
    /**
     * 获取工作面板数据收集器
     */
    public pangea.hiagent.workpanel.IWorkPanelDataCollector getWorkPanelCollector() {
        return workPanelCollector;
    }
}