package pangea.hiagent.agent.sse;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.web.dto.WorkPanelEvent;
import pangea.hiagent.workpanel.event.EventService;

import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import java.util.function.Consumer;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * SSE连接协调器
 * 专门负责协调SSE连接的创建、管理和销毁过程
 */
@Slf4j
@Component
public class SseConnectionCoordinator {
    
    private final UserSseService unifiedSseService;
    private final EventService eventService;
    
    public SseConnectionCoordinator(
            UserSseService unifiedSseService,
            EventService eventService) {
        this.unifiedSseService = unifiedSseService;
        this.eventService = eventService;
    }
    
    /**
     * 创建并注册SSE连接
     * 
     * @param userId 用户ID
     * @return SSE Emitter
     */
    public SseEmitter createAndRegisterConnection(String userId) {
        log.debug("开始为用户 {} 创建SSE连接", userId);
        
        // 创建 SSE emitter
        SseEmitter emitter = unifiedSseService.createEmitter();
        log.debug("SSE Emitter创建成功");
        
        // 注册用户的SSE连接
        unifiedSseService.registerSession(userId, emitter);
        log.debug("用户 {} 的SSE连接注册成功", userId);
        
        // 注册 emitter 回调
        unifiedSseService.registerCallbacks(emitter, userId);
        log.debug("SSE Emitter回调注册成功");
        
        // 启动心跳机制
        unifiedSseService.startHeartbeat(emitter, new AtomicBoolean(false));
        log.debug("心跳机制启动成功");
        
        log.info("用户 {} 的SSE连接创建和注册完成", userId);
        return emitter;
    }
    
    /**
     * 订阅工作面板事件
     * 
     * @param userId 用户ID
     * @param workPanelEventConsumer 工作面板事件消费者
     * @param emitter SSE Emitter
     */
    public void subscribeToWorkPanelEvents(String userId, Consumer<WorkPanelEvent> workPanelEventConsumer, SseEmitter emitter) {
        log.debug("开始为用户 {} 订阅工作面板事件", userId);
        
        // 发送连接成功事件
        try {
            WorkPanelEvent connectedEvent = WorkPanelEvent.builder()
                .type("observation")
                .title("连接成功")
                .timestamp(System.currentTimeMillis())
                .build();
            eventService.sendWorkPanelEvent(emitter, connectedEvent);
            log.debug("已发送连接成功事件");
        } catch (Exception e) {
            log.error("发送连接成功事件失败", e);
        }
        
        log.info("用户 {} 的工作面板事件订阅完成", userId);
    }
}