// 事件去重服务
export class EventDeduplicationService {
  // 用于跟踪已添加的事件的Set，防止重复
  private eventHashSet: Set<string> = new Set();

  /**
   * 检查是否为重复事件
   * @param event 当前事件
   * @returns 是否为重复事件
   */
  isDuplicateEvent(event: any): boolean {
    // 对于某些关键事件类型，我们允许重复显示（如错误事件）
    const criticalEventTypes = ['error', 'result'];
    if (criticalEventTypes.includes(event.type)) {
      return false;
    }

    // 生成事件的唯一标识符
    const eventHash = this.generateEventHash(event);
    if (this.eventHashSet.has(eventHash)) {
      return true;
    }

    // 将事件哈希添加到Set中
    this.eventHashSet.add(eventHash);

    // 限制Set大小以避免内存泄漏
    if (this.eventHashSet.size > 1000) {
      // 删除最早的100个条目
      const iterator = this.eventHashSet.values();
      for (let i = 0; i < 100; i++) {
        const value = iterator.next();
        if (!value.done) {
          this.eventHashSet.delete(value.value);
        }
      }
    }

    return false;
  }

  /**
   * 生成事件的唯一标识符
   * @param event 事件对象
   * @returns 事件哈希值
   */
  private generateEventHash(event: any): string {
    // 确保有时间戳
    const timestamp = event.timestamp || Date.now();
    
    // 对于工具事件，使用类型+工具名+工具操作+时间戳作为标识
    if (event.type && event.type.startsWith('tool_') && event.toolName && event.toolAction) {
      return `${event.type}-${event.toolName}-${event.toolAction}-${timestamp}`;
    }
    
    // 对于嵌入事件，使用URL+时间戳作为标识
    if (event.type === 'embed' && event.embedUrl) {
      return `embed-${event.embedUrl}-${timestamp}`;
    }
    
    // 对于其他事件，使用类型+标题+时间戳作为标识
    return `${event.type}-${event.title || ''}-${timestamp}`;
  }

  /**
   * 清除事件哈希集合
   */
  clearEventHashSet(): void {
    this.eventHashSet.clear();
  }
}