package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.openai.OpenAiChatModel;
import org.springframework.ai.openai.api.OpenAiApi;
import org.springframework.ai.openai.OpenAiChatOptions;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * OpenAI模型适配器
 * 实现OpenAI模型的创建和配置
 */
@Slf4j
@Component
public class OpenAiModelAdapter implements ModelAdapter {
    
    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            throw new IllegalArgumentException("无效的OpenAI配置");
        }
        
        try {
            // 使用Builder模式创建OpenAiApi实例
            OpenAiApi openAiApi = OpenAiApi.builder()
                    .apiKey(config.getApiKey())
                    .baseUrl(config.getBaseUrl() != null && !config.getBaseUrl().isEmpty() ? 
                        config.getBaseUrl() : "https://api.openai.com/v1")
                    .build();
                    
            // 使用Builder模式创建OpenAiChatModel实例
            return OpenAiChatModel.builder()
                    .openAiApi(openAiApi)
                    .defaultOptions(OpenAiChatOptions.builder()
                            .model(config.getModelName())
                            .temperature(config.getTemperature())
                            .maxTokens(config.getMaxTokens())
                            .build())
                    .build();
                    
        } catch (Exception e) {
            log.error("创建OpenAI ChatModel失败: {}", e.getMessage(), e);
            throw new RuntimeException("创建OpenAI ChatModel失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public String getProviderName() {
        return "openai";
    }
    
    @Override
    public boolean validateConfig(LlmConfig config) {
        return config != null &&
            config.getEnabled() &&
            config.getApiKey() != null &&
            !config.getApiKey().isEmpty();
    }
}