package pangea.hiagent.websocket;

import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.core.PlaywrightManager;

import java.util.concurrent.ConcurrentMap;

/**
 * DOM同步服务
 * 负责获取页面DOM、监听DOM变化、发送DOM更新
 */
@Slf4j
public class DomSyncService {
    // Playwright核心实例
    private Browser browser;
    private Page page;
    private BrowserContext context;
    
    // Playwright管理器引用
    private PlaywrightManager playwrightManager;
    
    // 当前会话的用户ID
    private String currentUserId;
    
    // 统计信息
    private final ConcurrentMap<String, Long> messageCounters;
    
    public DomSyncService(ConcurrentMap<String, Long> messageCounters) {
        this.messageCounters = messageCounters;
    }
    
    /**
     * 设置PlaywrightManager
     */
    public void setPlaywrightManager(PlaywrightManager playwrightManager) {
        this.playwrightManager = playwrightManager;
    }
    
    /**
     * 为指定用户初始化浏览器上下文和页面
     */
    public synchronized boolean initUserBrowserContext(String userId) {
        // 校验用户ID的有效性
        if (userId == null || userId.isEmpty()) {
            log.warn("用户ID为null或空字符串，跳过浏览器上下文初始化");
            return false;
        }
        
        // 检查Playwright管理器是否已设置
        if (playwrightManager == null) {
            log.warn("Playwright管理器未设置，无法初始化浏览器上下文和页面");
            return false;
        }
        
        try {
            log.debug("开始为用户 {} 初始化浏览器上下文", userId);
            
            // 保存当前用户ID
            this.currentUserId = userId;
            
            // 从Playwright管理器获取共享的浏览器实例
            browser = playwrightManager.getBrowser();
            
            // 从Playwright管理器获取用户专用的浏览器上下文
            context = playwrightManager.getUserContext(currentUserId);
            log.debug("用户 {} 的浏览器上下文创建成功", currentUserId);
            
            // 创建新页面
            page = context.newPage();
            log.debug("用户 {} 的页面创建成功", currentUserId);
            
            // 设置默认超时时间
            page.setDefaultTimeout(10000);
            log.debug("用户 {} 的页面超时时间设置完成", currentUserId);
            
            log.info("用户 {} 的浏览器上下文和页面初始化成功", currentUserId);
            return true;
        } catch (Exception e) {
            log.error("用户 {} 的浏览器上下文初始化失败", userId, e);
            // 清理可能部分初始化的对象
            cleanupResources();
            this.currentUserId = null;
            return false;
        }
    }
    
    /**
     * 清理资源
     */
    private void cleanupResources() {
        try {
            if (page != null && !page.isClosed()) {
                page.close();
            }
        } catch (Exception e) {
            log.warn("关闭页面失败", e);
        } finally {
            page = null;
        }
        
        try {
            if (context != null) {
                context.close();
            }
        } catch (Exception e) {
            log.warn("关闭浏览器上下文失败", e);
        } finally {
            context = null;
        }
        
        browser = null;
        currentUserId = null;
    }
    
    /**
     * 检查Playwright实例是否有效
     */
    public boolean isPlaywrightInstanceValid() {
        try {
            boolean browserValid = browser != null && browser.isConnected();
            boolean contextValid = context != null && !context.pages().isEmpty();
            boolean pageValid = page != null && !page.isClosed();
            return browserValid && contextValid && pageValid;
        } catch (Exception e) {
            log.warn("检查Playwright实例状态失败", e);
            return false;
        }
    }
    
    /**
     * 初始化页面监听事件（核心：捕获DOM变化）
     */
    public boolean initPageListener(BinaryMessageSender messageSender) {
        // 检查page是否已初始化
        if (!isPlaywrightInstanceValid()) {
            log.warn("Playwright实例未正确初始化，跳过页面监听器初始化");
            return false;
        }
        
        try {
            // 初始化统计计数器
            messageCounters.put("domChanges", 0L);
            messageCounters.put("websocketMessages", 0L);
            messageCounters.put("errors", 0L);
            
            // 1. 页面加载完成后，推送完整的DOM（初始化）
            page.onLoad(page -> {
                incrementCounter("pageLoads");
                // 发送完整的DOM内容到客户端
                sendFullDomToClientsWithRetry(messageSender);
            });

            // 2. 监听DOM变化（使用MutationObserver），推送增量更新
            // 针对iframe特点优化：只监听body区域的变化
            page.evaluate("() => {\n" +
                    "  // 创建MutationObserver监听DOM变化\n" +
                    "  const observer = new MutationObserver((mutations) => {\n" +
                    "    // 将变化的DOM节点转为字符串，发送给Playwright\n" +
                    "    const changes = mutations.map(mutation => ({\n" +
                    "      type: mutation.type,\n" +
                    "      target: mutation.target.outerHTML || mutation.target.textContent,\n" +
                    "      addedNodes: Array.from(mutation.addedNodes).map(node => node.outerHTML || node.textContent || ''),\n" +
                    "      removedNodes: Array.from(mutation.removedNodes).map(node => node.outerHTML || node.textContent || ''),\n" +
                    "      attributeName: mutation.attributeName,\n" +
                    "      oldValue: mutation.oldValue\n" +
                    "    }));\n" +
                    "    // 调用Playwright的暴露函数，传递DOM变化数据\n" +
                    "    window.domChanged(JSON.stringify(changes));\n" +
                    "  });\n" +
                    "  // 配置监听：监听body节点的添加/删除、属性变化、子节点变化\n" +
                    "  observer.observe(document.body, {\n" +
                    "    childList: true,\n" +
                    "    attributes: true,\n" +
                    "    subtree: true,\n" +
                    "    characterData: true,\n" +
                    "    attributeOldValue: true\n" +
                    "  });\n" +
                    "}");

            // 3. 暴露Playwright函数，接收前端的DOM变化数据
            page.exposeFunction("domChanged", args -> {
                try {
                    if (args.length > 0 && args[0] instanceof String) {
                        String changes = (String) args[0];
                        if (changes != null && !changes.isEmpty()) {
                            incrementCounter("domChanges");
                            sendIncrementalDomToClients(changes, messageSender);
                        }
                    }
                } catch (Exception e) {
                    String errorMsg = "处理DOM变化失败: " + e.getMessage();
                    log.error(errorMsg, e);
                    incrementCounter("errors");
                    messageSender.sendErrorToClients(errorMsg);
                }
                return null;
            });

            // 4. 监听页面导航事件，导航后重新初始化
            page.onFrameNavigated(frame -> {
                incrementCounter("navigations");
                // 异步处理导航完成后的DOM发送，避免阻塞
                java.util.concurrent.CompletableFuture.runAsync(() -> {
                    try {
                        // 使用更宽松的等待条件，避免NETWORKIDLE可能出现的问题
                        page.waitForLoadState(LoadState.DOMCONTENTLOADED);
                        // 等待一小段时间确保关键资源加载完成
                        try {
                            Thread.sleep(500);
                        } catch (InterruptedException ignored) {
                        }
                        // 发送更新后的DOM内容到客户端
                        sendFullDomToClientsWithRetry(messageSender);
                    } catch (Exception e) {
                        String errorMsg = "页面加载状态等待失败: " + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        incrementCounter("errors");
                        messageSender.sendErrorToClients(errorMsg);
                    }
                });
            });        
            // 5. 监听页面错误事件
            page.onPageError(error -> {
                try {
                    String errorMsg = "页面错误: " + error;
                    System.err.println(errorMsg);
                    incrementCounter("errors");
                    
                    // 对特定错误进行特殊处理
                    if (errorMsg.contains("sso.hisense.com") && errorMsg.contains("setRequestHeader")) {
                        log.warn("检测到海信SSO系统的JavaScript错误，这不会影响系统核心功能: {}", errorMsg);
                        // 不向客户端发送此类错误，避免干扰用户体验
                    } else {
                        messageSender.sendErrorToClients(errorMsg);
                    }
                } catch (Exception e) {
                    log.error("处理页面错误事件失败", e);
                }
            });
            
            // 6. 监听控制台错误信息
            page.onConsoleMessage(msg -> {
                if ("error".equals(msg.type().toString().toLowerCase())) {
                    String errorMsg = "控制台错误: " + msg.text();
                    log.error("页面控制台错误: {}", errorMsg);
                    
                    // 对特定错误进行特殊处理
                    if (errorMsg.contains("sso.hisense.com") && errorMsg.contains("setRequestHeader")) {
                        log.warn("检测到海信SSO系统的控制台JavaScript错误，这不会影响系统核心功能");
                        // 不向客户端发送此类错误，避免干扰用户体验
                    }
                }
            });
            
            log.info("页面监听器初始化成功");
            return true;
        } catch (Exception e) {
            log.error("页面监听器初始化失败", e);
            return false;
        }
    }
    
    /**
     * 推送完整的DOM给所有客户端（带重试机制）
     */
    public void sendFullDomToClientsWithRetry(BinaryMessageSender messageSender) {
        sendFullDomToClientsWithRetry(messageSender, 3, 500); // 默认重试3次，每次间隔500毫秒
    }
    
    /**
     * 推送完整的DOM给所有客户端（带重试机制）
     * 使用二进制协议传输大消息
     */
    public void sendFullDomToClientsWithRetry(BinaryMessageSender messageSender, int maxRetries, long retryDelay) {
        Exception lastException = null;
        String pageUrl = "unknown";
        
        for (int i = 0; i < maxRetries; i++) {
            try {
                // 验证page实例是否有效
                if (!isPlaywrightInstanceValid()) {
                    String errorMsg = "第" + (i + 1) + "次尝试：Playwright实例无效";
                    log.error(errorMsg);
                    lastException = new Exception(errorMsg);
                    continue;
                }
                
                if (page.isClosed()) {
                    String errorMsg = "第" + (i + 1) + "次尝试：Page实例已关闭";
                    log.error(errorMsg);
                    lastException = new Exception(errorMsg);
                    continue;
                }
                
                // 获取当前页面URL用于日志记录
                pageUrl = getCurrentPageUrl();
                log.debug("第{}次尝试获取DOM，当前URL: {}", i + 1, pageUrl);
                
                // 1. 获取页面完整DOM（包含所有节点）
                String fullDom = page.content();
                
                if (fullDom == null || fullDom.trim().isEmpty()) {
                    String errorMsg = "第" + (i + 1) + "次尝试：获取到的DOM为空或只有空白";
                    log.warn(errorMsg);
                    lastException = new Exception(errorMsg);
                    
                    // 如果不是最后一次重试，则等待一段时间再重试
                    if (i < maxRetries - 1) {
                        try {
                            log.debug("等待{}ms后进行第{}次重试", retryDelay, i + 2);
                            Thread.sleep(retryDelay);
                        } catch (InterruptedException ie) {
                            Thread.currentThread().interrupt();
                            String interruptMsg = "获取完整DOM被中断：" + ie.getMessage();
                            log.error(interruptMsg, ie);
                            messageSender.sendErrorToClients(interruptMsg);
                            return;
                        }
                    }
                    continue;
                }
                
                log.debug("第{}次尝试成功获取DOM，长度: {} 字节", i + 1, fullDom.length());

                // 使用二进制协议发送DOM数据
                messageSender.sendBinaryMessage(fullDom);
                log.debug("第{}次尝试：DOM推送成功", i + 1);
                
                // 成功发送，直接返回
                return;
            } catch (Exception e) {
                lastException = e;
                log.warn("第{}次获取完整DOM失败: {}", i + 1, e.getMessage(), e);
                
                // 如果不是最后一次重试，则等待一段时间再重试
                if (i < maxRetries - 1) {
                    try {
                        log.debug("等待{}ms后进行第{}次重试", retryDelay, i + 2);
                        Thread.sleep(retryDelay);
                    } catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        String interruptMsg = "获取完整DOM被中断：" + ie.getMessage();
                        log.error(interruptMsg, ie);
                        messageSender.sendErrorToClients(interruptMsg);
                        return;
                    }
                }
            }
        }
        
        // 所有重试都失败了
        if (lastException != null) {
            String errorMsg = "获取完整DOM失败（已重试" + maxRetries + "次），最后错误: " + lastException.getMessage() + "，当前URL: " + pageUrl;
            log.error(errorMsg, lastException);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 推送增量DOM变化给所有客户端（DOM更新时调用）
     * 使用二进制协议传输
     */
    public void sendIncrementalDomToClients(String changes, BinaryMessageSender messageSender) {
        try {
            // 使用二进制协议发送增量更新
            messageSender.sendBinaryMessage(changes);
        } catch (Exception e) {
            log.error("推送增量 DOM 失败", e);
            messageSender.sendErrorToClients("推送增量 DOM 失败：" + e.getMessage());
        }
    }
    
    /**
     * 增加计数器
     */
    private void incrementCounter(String counterName) {
        messageCounters.merge(counterName, 1L, Long::sum);
    }
    
    /**
     * 安全地获取当前页面URL
     */
    private String getCurrentPageUrl() {
        try {
            return page != null ? page.url() : "";
        } catch (Exception e) {
            log.warn("获取当前页面URL失败", e);
            return "";
        }
    }
    
    /**
     * 检查页面是否有效
     */
    public boolean isPageValid() {
        return page != null && !page.isClosed();
    }
    
    /**
     * 释放用户资源
     */
    public void releaseUserResources(String userId, PlaywrightManager playwrightManager) {
        try {
            if (page != null && !page.isClosed()) {
                page.close();
            }
        } catch (Exception e) {
            log.warn("关闭页面失败", e);
        }
        
        try {
            if (context != null) {
                context.close();
                // 通知Playwright管理器释放用户上下文
                if (playwrightManager != null) {
                    playwrightManager.releaseUserContext(userId);
                }
            }
        } catch (Exception e) {
            log.warn("关闭浏览器上下文失败", e);
        }
    }
    
    /**
     * 销毁资源
     */
    public void destroy(String userId, PlaywrightManager playwrightManager) {
        try {
            // 关闭Playwright资源
            if (page != null && !page.isClosed()) {
                page.close();
            }
        } catch (Exception e) {
            log.warn("关闭页面失败", e);
        }
        
        try {
            if (context != null) {
                context.close();
                // 通知Playwright管理器释放用户上下文
                if (playwrightManager != null && userId != null) {
                    playwrightManager.releaseUserContext(userId);
                }
            }
        } catch (Exception e) {
            log.warn("关闭浏览器上下文失败", e);
        }
    }
    
    /**
     * 获取当前用户ID
     */
    public String getCurrentUserId() {
        return currentUserId;
    }
    
    /**
     * 获取当前页面对象
     */
    public Page getCurrentPage() {
        return page;
    }
}