package pangea.hiagent.websocket;

import com.alibaba.fastjson2.JSON;
import com.microsoft.playwright.Locator;
import com.microsoft.playwright.Page;
import com.microsoft.playwright.options.LoadState;
import com.microsoft.playwright.options.WaitForSelectorState;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.ConcurrentMap;

/**
 * 指令处理器
 * 负责处理客户端发送的各种指令
 */
@Slf4j
public class CommandProcessor {
    // 指令频率限制
    private static final int MAX_COMMANDS_PER_SECOND = 10; // 每秒最大指令数
    
    private final ConcurrentMap<String, Long> lastCommandTimes;
    private final ConcurrentMap<String, Integer> commandCounts;
    private final ConcurrentMap<String, Long> messageCounters;
    
    public CommandProcessor(ConcurrentMap<String, Long> lastCommandTimes, 
                           ConcurrentMap<String, Integer> commandCounts,
                           ConcurrentMap<String, Long> messageCounters) {
        this.lastCommandTimes = lastCommandTimes;
        this.commandCounts = commandCounts;
        this.messageCounters = messageCounters;
    }
    
    /**
     * 处理客户端发送的指令（如导航、点击、输入）
     */
    public void processCommand(String payload, Page page, BinaryMessageSender messageSender, 
                              StatisticsService statisticsService, String userId) throws Exception {
        // 记录接收到的消息
        log.debug("收到WebSocket消息: {}", payload);
        incrementCounter("websocketMessages");
        
        if (userId == null || userId.isEmpty()) {
            // 如果没有有效的用户ID，使用默认值
            userId = "unknown-user";
            log.warn("指令处理缺少有效的用户认证信息，使用默认用户ID: {}", userId);
        }
        
        // 指令频率限制
        long currentTime = System.currentTimeMillis();
        Long lastTime = lastCommandTimes.getOrDefault(userId, 0L);
        Integer commandCount = commandCounts.getOrDefault(userId, 0);
        
        // 检查是否超过每秒最大指令数
        if (currentTime - lastTime < 1000) {
            // 同一秒内
            if (commandCount >= MAX_COMMANDS_PER_SECOND) {
                String errorMsg = "指令执行过于频繁，请稍后再试";
                log.warn("用户 {} {}", userId, errorMsg);
                messageSender.sendErrorToClients(errorMsg);
                return;
            }
            commandCounts.put(userId, commandCount + 1);
        } else {
            // 新的一秒
            lastCommandTimes.put(userId, currentTime);
            commandCounts.put(userId, 1);
        }
        
        try {
            // 检查Playwright实例是否有效
            if (!isPlaywrightInstanceValid(page)) {
                String errorMsg = "Playwright实例未初始化或已关闭，请刷新页面重试";
                log.warn("用户 {} {}", userId, errorMsg);
                messageSender.sendErrorToClients(errorMsg);
                return;
            }
            
            // 指令格式：指令类型:参数（如navigate:https://www.baidu.com）
            // 特殊处理navigate命令，避免URL中的冒号导致分割错误
            String command;
            String param;
            
            int firstColonIndex = payload.indexOf(':');
            if (firstColonIndex != -1) {
                command = payload.substring(0, firstColonIndex);
                param = payload.substring(firstColonIndex + 1);
            } else {
                command = payload;
                param = "";
            }
            
            // 处理用户指令
            log.debug("处理指令: {}, 参数: {} (用户: {})", command, param, userId);
            
            switch (command) {
                case "navigate":
                    handleNavigateCommand(param, page, messageSender);
                    break;
                case "stats":
                    handleStatsCommand(statisticsService, messageSender);
                    break;
                case "reset-stats":
                    handleResetStatsCommand(statisticsService, messageSender);
                    break;
                case "click":
                    handleClickCommand(param, page, messageSender);
                    break;
                case "dblclick":
                    handleDblClickCommand(param, page, messageSender);
                    break;
                case "hover":
                    handleHoverCommand(param, page, messageSender);
                    break;
                case "focus":
                    handleFocusCommand(param, page, messageSender);
                    break;
                case "blur":
                    handleBlurCommand(param, page, messageSender);
                    break;
                case "select":
                    handleSelectCommand(param, page, messageSender);
                    break;
                case "keydown":
                    handleKeyDownCommand(param, page, messageSender);
                    break;
                case "keyup":
                    handleKeyUpCommand(param, page, messageSender);
                    break;
                case "keypress":
                    handleKeyPressCommand(param, page, messageSender);
                    break;
                case "scroll":
                    handleScrollCommand(param, page, messageSender);
                    break;
                case "type":
                    handleTypeCommand(param, page, messageSender);
                    break;
                case "input":
                    handleInputCommand(param, page, messageSender);
                    break;
                default:
                    messageSender.sendErrorToClients("未知指令：" + command);
            }
        } catch (Exception e) {
            log.error("指令执行失败", e);
            messageSender.sendErrorToClients("指令执行失败：" + e.getMessage());
        }
    }
    
    /**
     * 处理导航指令
     */
    private void handleNavigateCommand(String param, Page page, BinaryMessageSender messageSender) {
        // 导航到指定URL
        if (param == null || param.trim().isEmpty()) {
            messageSender.sendErrorToClients("导航URL不能为空");
            return;
        }
        
        // 验证URL格式
        if (!isValidUrl(param)) {
            messageSender.sendErrorToClients("无效的URL格式: " + param);
            return;
        }
        
        try {
            page.navigate(param);
            
            // 异步处理页面加载完成后的DOM发送，避免阻塞WebSocket
            java.util.concurrent.CompletableFuture.runAsync(() -> {
                try {
                    // 等待页面加载状态：DOMCONTENTLOADED确保DOM可用
                    log.debug("等待页面DOM加载: {}", param);
                    page.waitForLoadState(LoadState.DOMCONTENTLOADED);
                    
                    // 额外等待500ms，确保关键脚本执行完成
                    Thread.sleep(500);
                    log.debug("页面DOM加载完成: {}", param);
                    
                    // 发送完整DOM到前端
                    String fullDom = page.content();
                    messageSender.sendBinaryMessage(fullDom);
                } catch (Exception e) {
                    String errorMsg = "页面加载或DOM获取失败: " + e.getMessage();
                    log.error(errorMsg, e);
                    // 重要：必须将错误发送给前端
                    messageSender.sendErrorToClients(errorMsg);
                }
            });
        } catch (Exception e) {
            String errorMsg = "导航命令执行失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理统计信息指令
     */
    private void handleStatsCommand(StatisticsService statisticsService, BinaryMessageSender messageSender) {
        try {
            String stats = statisticsService.getStatisticsSummary();
            // 使用二进制协议发送统计信息
            messageSender.broadcastMessage(stats);
        } catch (Exception e) {
            String errorMsg = "获取统计信息失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理重置统计信息指令
     */
    private void handleResetStatsCommand(StatisticsService statisticsService, BinaryMessageSender messageSender) {
        try {
            statisticsService.resetAllCounters();
            messageSender.sendErrorToClients("统计信息已重置");
        } catch (Exception e) {
            String errorMsg = "重置统计信息失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理点击指令
     */
    private void handleClickCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.locator(param).click();
        } catch (Exception e) {
            String errorMsg = "点击元素失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理双击指令
     */
    private void handleDblClickCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.locator(param).dblclick();
        } catch (Exception e) {
            String errorMsg = "双击元素失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理悬停指令
     */
    private void handleHoverCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            // 增强hover操作：先等待元素可见，再执行hover
            Locator locator = page.locator(param);
            
            // 等待元素可见，最多等待10秒
            locator.waitFor(new Locator.WaitForOptions()
                    .setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE)
                    .setTimeout(10000));
            
            // 执行hover操作
            locator.hover();
        } catch (Exception e) {
            String errorMsg = "悬停元素失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理聚焦指令
     */
    private void handleFocusCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.locator(param).focus();
        } catch (Exception e) {
            String errorMsg = "聚焦元素失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理失去焦点指令
     */
    private void handleBlurCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.locator(param).blur();
        } catch (Exception e) {
            String errorMsg = "失去焦点失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理选择指令
     */
    private void handleSelectCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            String[] selectParts = param.split(":", 2);
            if (selectParts.length == 2) {
                page.locator(selectParts[0]).selectOption(selectParts[1]);
            }
        } catch (Exception e) {
            String errorMsg = "选择选项失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理键盘按下指令
     */
    private void handleKeyDownCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.keyboard().down(param);
        } catch (Exception e) {
            String errorMsg = "键盘按下失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理键盘释放指令
     */
    private void handleKeyUpCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.keyboard().up(param);
        } catch (Exception e) {
            String errorMsg = "键盘释放失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理按键事件指令
     */
    private void handleKeyPressCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            page.keyboard().press(param);
        } catch (Exception e) {
            String errorMsg = "按键事件失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理滚动指令
     */
    private void handleScrollCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            int scrollY = Integer.parseInt(param);
            page.evaluate("window.scrollTo(0, " + scrollY + ")");
        } catch (Exception e) {
            String errorMsg = "滚动页面失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理输入指令
     */
    private void handleTypeCommand(String param, Page page, BinaryMessageSender messageSender) {
        try {
            // 使用indexOf和substring替代split，避免内容中的冒号导致分割错误
            int colonIndex = param.indexOf(':');
            if (colonIndex != -1 && colonIndex < param.length() - 1) {
                String selector = param.substring(0, colonIndex);
                String content = param.substring(colonIndex + 1);
                Locator inputLocator = page.locator(selector);
                inputLocator.fill(content);
            }
        } catch (Exception e) {
            String errorMsg = "输入内容失败：" + e.getMessage();
            log.error(errorMsg, e);
            messageSender.sendErrorToClients(errorMsg);
        }
    }
    
    /**
     * 处理输入指令（别名）
     */
    private void handleInputCommand(String param, Page page, BinaryMessageSender messageSender) {
        handleTypeCommand(param, page, messageSender);
    }
    
    /**
     * 验证URL格式是否有效
     */
    private boolean isValidUrl(String url) {
        if (url == null || url.trim().isEmpty()) {
            return false;
        }
        
        // 检查是否以http://或https://开头
        if (!url.toLowerCase().startsWith("http://") && !url.toLowerCase().startsWith("https://")) {
            return false;
        }
        
        // 简单的URL格式检查
        try {
            new java.net.URL(url);
            return true;
        } catch (Exception e) {
            return false;
        }
    }
    
    /**
     * 检查Playwright实例是否有效
     */
    private boolean isPlaywrightInstanceValid(Page page) {
        try {
            boolean pageValid = page != null && !page.isClosed();
            return pageValid;
        } catch (Exception e) {
            log.warn("检查Playwright实例状态失败", e);
            return false;
        }
    }
    
    /**
     * 增加计数器
     */
    private void incrementCounter(String counterName) {
        messageCounters.merge(counterName, 1L, Long::sum);
    }
}