package pangea.hiagent.utils;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 异步用户上下文使用示例
 * 展示如何在异步任务中正确获取用户认证信息
 */
@Slf4j
@Component
public class AsyncUserContextUsageExample {
    
    // 示例线程池
    private final ExecutorService executorService = Executors.newFixedThreadPool(10);
    
    /**
     * 方式一：使用SecurityContextHolder的InheritableThreadLocal策略（推荐）
     * 适用于父子线程关系明确的场景
     */
    public void executeTaskWithInheritableThreadLocal() {
        // 在主线程中获取用户ID（正常情况下可以获取到）
        String userId = UserUtils.getCurrentUserId();
        log.info("主线程中获取到用户ID: {}", userId);
        
        // 提交异步任务，由于使用了InheritableThreadLocal策略，子线程可以继承父线程的SecurityContext
        CompletableFuture.runAsync(() -> {
            // 在异步线程中获取用户ID
            String asyncUserId = UserUtils.getCurrentUserId();
            log.info("异步线程中获取到用户ID: {}", asyncUserId);
            
            // 执行业务逻辑
            performBusinessLogic(asyncUserId);
        }, executorService);
    }
    
    /**
     * 方式二：使用UserContextPropagationUtil手动传播用户上下文
     * 适用于复杂的异步场景或需要更精确控制的场景
     */
    public void executeTaskWithManualPropagation() {
        // 在主线程中获取用户ID
        String userId = UserUtils.getCurrentUserId();
        log.info("主线程中获取到用户ID: {}", userId);
        
        // 提交异步任务，手动传播用户上下文
        CompletableFuture.runAsync(AsyncUserContextDecorator.wrapWithContext(() -> {
            // 在异步线程中获取用户ID
            String asyncUserId = UserUtils.getCurrentUserId();
            log.info("异步线程中获取到用户ID: {}", asyncUserId);
            
            // 执行业务逻辑
            performBusinessLogic(asyncUserId);
        }), executorService);
    }
    
    /**
     * 方式三：使用专门的异步环境获取方法
     * 适用于无法通过线程上下文传播获取用户信息的场景
     */
    public void executeTaskWithDirectTokenParsing() {
        // 在主线程中获取用户ID
        String userId = UserUtils.getCurrentUserId();
        log.info("主线程中获取到用户ID: {}", userId);
        
        // 提交异步任务，直接解析请求中的token获取用户ID
        CompletableFuture.runAsync(() -> {
            // 在异步线程中通过直接解析token获取用户ID
            String asyncUserId = UserUtils.getCurrentUserIdInAsync();
            log.info("异步线程中通过直接解析token获取到用户ID: {}", asyncUserId);
            
            // 执行业务逻辑
            performBusinessLogic(asyncUserId);
        }, executorService);
    }
    
    /**
     * 执行业务逻辑示例
     * @param userId 用户ID
     */
    private void performBusinessLogic(String userId) {
        if (userId != null) {
            log.info("为用户 {} 执行业务逻辑", userId);
            // 这里执行具体的业务逻辑
        } else {
            log.warn("未获取到用户ID，执行匿名用户逻辑");
            // 这里执行匿名用户的业务逻辑
        }
    }
    
    /**
     * 清理资源
     */
    public void shutdown() {
        executorService.shutdown();
    }
}