package pangea.hiagent.web.service;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.model.AgentToolRelation;
import pangea.hiagent.web.repository.AgentToolRelationRepository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Agent和Tool关系服务类
 * 负责管理Agent和Tool之间的多对多关系
 * 采用多对多关系表存储，禁止使用JSON字段
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class AgentToolRelationService {
    
    private final AgentToolRelationRepository agentToolRelationRepository;
    
    /**
     * 为Agent设置工具关联关系（完全替换）
     * 先删除所有旧关联，再添加新关联
     * 
     * @param agentId Agent ID
     * @param toolIds 工具ID列表
     */
    @Transactional(rollbackFor = Exception.class)
    public void setAgentTools(String agentId, List<String> toolIds) {
        log.info("开始为Agent设置工具关联，Agent ID: {}, 新工具数量: {}", agentId, toolIds != null ? toolIds.size() : 0);
        
        try {
            // 参数验证
            if (agentId == null || agentId.trim().isEmpty()) {
                log.warn("Agent ID为空");
                throw new IllegalArgumentException("Agent ID不能为空");
            }
            
            // 获取原有关联
            List<String> oldToolIds = agentToolRelationRepository.getToolIdsByAgentId(agentId);
            log.debug("Agent {} 原有工具数量: {}", agentId, oldToolIds != null ? oldToolIds.size() : 0);
            
            // 规范化新工具ID列表：去除空值和重复
            Set<String> newToolIds = new HashSet<>();
            if (toolIds != null) {
                newToolIds.addAll(
                    toolIds.stream()
                        .filter(id -> id != null && !id.trim().isEmpty())
                        .map(String::trim)
                        .collect(Collectors.toSet())
                );
            }
            
            log.debug("规范化后的新工具ID数量: {}", newToolIds.size());
            
            // 计算需要删除和新增的关联
            Set<String> oldToolIdSet = new HashSet<>(oldToolIds != null ? oldToolIds : new ArrayList<>());
            Set<String> toDelete = new HashSet<>(oldToolIdSet);
            toDelete.removeAll(newToolIds);
            Set<String> toAdd = new HashSet<>(newToolIds);
            toAdd.removeAll(oldToolIdSet);
            
            // 执行删除操作
            if (!toDelete.isEmpty()) {
                log.debug("Agent {} 删除工具关联数: {}", agentId, toDelete.size());
                for (String toolId : toDelete) {
                    agentToolRelationRepository.delete(
                        new LambdaQueryWrapper<AgentToolRelation>()
                            .eq(AgentToolRelation::getAgentId, agentId)
                            .eq(AgentToolRelation::getToolId, toolId)
                    );
                }
            }
            
            // 执行新增操作
            if (!toAdd.isEmpty()) {
                log.debug("Agent {} 新增工具关联数: {}", agentId, toAdd.size());
                // MyBatis Plus的BaseMapper没有insertBatch方法，使用循环插入
                for (String toolId : toAdd) {
                    AgentToolRelation relation = new AgentToolRelation();
                    relation.setAgentId(agentId);
                    relation.setToolId(toolId);
                    // id由BaseEntity的@TableId(type = IdType.ASSIGN_UUID)自动生成
                    agentToolRelationRepository.insert(relation);
                }
            }
            
            log.info("Agent {} 的工具关联关系设置完成。删除: {}, 新增: {}", agentId, toDelete.size(), toAdd.size());
        } catch (IllegalArgumentException e) {
            log.warn("参数验证失败，Agent ID: {}", agentId, e);
            throw e;
        } catch (Exception e) {
            log.error("为Agent {} 设置工具关联关系时发生异常", agentId, e);
            throw new RuntimeException("设置工具关联关系失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 添加单个工具关联
     * 
     * @param agentId Agent ID
     * @param toolId 工具ID
     */
    @Transactional(rollbackFor = Exception.class)
    public void addAgentTool(String agentId, String toolId) {
        log.info("为Agent {} 添加工具关联，Tool ID: {}", agentId, toolId);
        
        try {
            // 参数验证
            if (agentId == null || agentId.trim().isEmpty()) {
                throw new IllegalArgumentException("Agent ID不能为空");
            }
            if (toolId == null || toolId.trim().isEmpty()) {
                throw new IllegalArgumentException("Tool ID不能为空");
            }
            
            // 检查关联是否已存在
            Long count = agentToolRelationRepository.selectCount(
                new LambdaQueryWrapper<AgentToolRelation>()
                    .eq(AgentToolRelation::getAgentId, agentId)
                    .eq(AgentToolRelation::getToolId, toolId)
            );
            
            if (count > 0) {
                log.warn("工具关联已存在，Agent ID: {}, Tool ID: {}", agentId, toolId);
                return;
            }
            
            // 创建新关联
            AgentToolRelation relation = new AgentToolRelation();
            relation.setAgentId(agentId);
            relation.setToolId(toolId);
            // id由BaseEntity的@TableId(type = IdType.ASSIGN_UUID)自动生成
            agentToolRelationRepository.insert(relation);
            
            log.info("工具关联添加成功，Agent ID: {}, Tool ID: {}", agentId, toolId);
        } catch (IllegalArgumentException e) {
            log.warn("参数验证失败，Agent ID: {}, Tool ID: {}", agentId, toolId, e);
            throw e;
        } catch (Exception e) {
            log.error("添加工具关联失败，Agent ID: {}, Tool ID: {}", agentId, toolId, e);
            throw new RuntimeException("添加工具关联失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 删除单个工具关联
     * 
     * @param agentId Agent ID
     * @param toolId 工具ID
     */
    @Transactional(rollbackFor = Exception.class)
    public void removeAgentTool(String agentId, String toolId) {
        log.info("为Agent {} 删除工具关联，Tool ID: {}", agentId, toolId);
        
        try {
            // 参数验证
            if (agentId == null || agentId.trim().isEmpty()) {
                throw new IllegalArgumentException("Agent ID不能为空");
            }
            if (toolId == null || toolId.trim().isEmpty()) {
                throw new IllegalArgumentException("Tool ID不能为空");
            }
            
            agentToolRelationRepository.delete(
                new LambdaQueryWrapper<AgentToolRelation>()
                    .eq(AgentToolRelation::getAgentId, agentId)
                    .eq(AgentToolRelation::getToolId, toolId)
            );
            
            log.info("工具关联删除成功，Agent ID: {}, Tool ID: {}", agentId, toolId);
        } catch (IllegalArgumentException e) {
            log.warn("参数验证失败，Agent ID: {}, Tool ID: {}", agentId, toolId, e);
            throw e;
        } catch (Exception e) {
            log.error("删除工具关联失败，Agent ID: {}, Tool ID: {}", agentId, toolId, e);
            throw new RuntimeException("删除工具关联失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 删除Agent的所有工具关联
     * 
     * @param agentId Agent ID
     */
    @Transactional(rollbackFor = Exception.class)
    public void removeAgentAllTools(String agentId) {
        log.info("为Agent {} 删除所有工具关联", agentId);
        
        try {
            if (agentId == null || agentId.trim().isEmpty()) {
                throw new IllegalArgumentException("Agent ID不能为空");
            }
            
            agentToolRelationRepository.delete(
                new LambdaQueryWrapper<AgentToolRelation>()
                    .eq(AgentToolRelation::getAgentId, agentId)
            );
            
            log.info("Agent {} 的所有工具关联已删除", agentId);
        } catch (IllegalArgumentException e) {
            log.warn("参数验证失败，Agent ID: {}", agentId, e);
            throw e;
        } catch (Exception e) {
            log.error("删除Agent所有工具关联失败，Agent ID: {}", agentId, e);
            throw new RuntimeException("删除工具关联失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 获取Agent关联的工具ID列表
     * 
     * @param agentId Agent ID
     * @return 工具ID列表
     */
    public List<String> getToolIdsByAgentId(String agentId) {
        try {
            if (agentId == null || agentId.trim().isEmpty()) {
                log.warn("Agent ID为空");
                return new ArrayList<>();
            }
            
            List<String> toolIds = agentToolRelationRepository.getToolIdsByAgentId(agentId);
            log.debug("获取Agent {} 的工具ID列表，数量: {}", agentId, toolIds != null ? toolIds.size() : 0);
            return toolIds != null ? toolIds : new ArrayList<>();
        } catch (Exception e) {
            log.error("获取Agent {} 的工具ID列表失败", agentId, e);
            return new ArrayList<>();
        }
    }
    
    /**
     * 获取工具关联的Agent ID列表
     * 
     * @param toolId 工具ID
     * @return Agent ID列表
     */
    public List<String> getAgentIdsByToolId(String toolId) {
        try {
            if (toolId == null || toolId.trim().isEmpty()) {
                log.warn("Tool ID为空");
                return new ArrayList<>();
            }
            
            List<String> agentIds = agentToolRelationRepository.getAgentIdsByToolId(toolId);
            log.debug("获取Tool {} 的Agent ID列表，数量: {}", toolId, agentIds != null ? agentIds.size() : 0);
            return agentIds != null ? agentIds : new ArrayList<>();
        } catch (Exception e) {
            log.error("获取Tool {} 的Agent ID列表失败", toolId, e);
            return new ArrayList<>();
        }
    }
    
    /**
     * 检查Agent是否关联了指定工具
     * 
     * @param agentId Agent ID
     * @param toolId 工具ID
     * @return 是否关联
     */
    public boolean isToolAssociated(String agentId, String toolId) {
        try {
            if (agentId == null || toolId == null) {
                return false;
            }
            
            Long count = agentToolRelationRepository.selectCount(
                new LambdaQueryWrapper<AgentToolRelation>()
                    .eq(AgentToolRelation::getAgentId, agentId)
                    .eq(AgentToolRelation::getToolId, toolId)
            );
            return count > 0;
        } catch (Exception e) {
            log.error("检查工具关联失败，Agent ID: {}, Tool ID: {}", agentId, toolId, e);
            return false;
        }
    }
}