package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.core.io.InputStreamResource;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URL;
import java.net.HttpURLConnection;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Enumeration;
import jakarta.servlet.http.HttpServletRequest;

import org.owasp.html.HtmlPolicyBuilder;
import org.owasp.html.PolicyFactory;

/**
 * 网页代理控制器
 * 用于解决iframe加载网页时的CORS问题，并处理X-Frame-Options等安全限制
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/proxy")
public class WebProxyController {
    
    // HTML 清理策略
    private static final PolicyFactory HTML_POLICY = new HtmlPolicyBuilder()
        .allowElements("a", "p", "div", "span", "img", "h1", "h2", "h3", "h4", "h5", "h6",
        "ul", "ol", "li", "strong", "em", "br", "hr", "table", "tr", "td", "th", "code", "pre")
        .allowAttributes("href", "title", "alt", "src").onElements("a", "img")
        .allowAttributes("class", "id", "style").onElements("div", "span", "p", "h1", "h2", "h3", "h4", "h5", "h6")
        .allowAttributes("colspan", "rowspan").onElements("td", "th")
        .toFactory();

    /**
     * 代理GET请求指定URL的内容
     * 
     * @param url 要代理的URL地址
     * @param token JWT token（可选）
     * @return 网页HTML内容
     */
    @GetMapping(produces = MediaType.TEXT_HTML_VALUE)
    public ResponseEntity<?> proxyUrl(@RequestParam String url, @RequestParam(required = false) String token) {
        try {
            log.debug("代理GET请求URL: {}, token: {}", url, token != null ? "provided" : "null");
            
            // 验证URL格式
            if (url == null || url.trim().isEmpty()) {
                log.warn("无效的URL参数: 为空");
                return createBadRequestResponse("URL is empty");
            }
            
            // 检查是否为file协议
            if (url.toLowerCase().startsWith("file://")) {
                // 处理本地文件
                return handleLocalFile(url);
            }
            
            // 处理相对路径URL
            if (!url.toLowerCase().startsWith("http://") && !url.toLowerCase().startsWith("https://")) {
                // 如果是相对路径，尝试补全为绝对路径
                if (url.startsWith("/")) {
                    // 假设是本地文件路径
                    url = "file://" + url;
                } else if (!url.toLowerCase().startsWith("file://")) {
                    log.warn("无效的URL scheme: {}", url);
                    return createBadRequestResponse("URL must start with http://, https:// or file://");
                }
            }
            
            // 对于大文件，使用流式传输
            if (isLargeFile(url)) {
                return streamLargeFile(url);
            }
            
            // 获取网页内容
            String content = fetchWebContent(url);
            log.debug("成功代理URL内容，长度: {} bytes", content.length());
            
            // 处理内容以提高兼容性
            String processedContent = processContentForEmbedding(content, url);
            
            // 添加调试信息
            log.debug("处理后内容长度: {} bytes", processedContent.length());
            
            return ResponseEntity.ok()
                .contentType(MediaType.TEXT_HTML)
                .body(processedContent);
            
        } catch (IOException e) {
            log.error("代理GET请求失败: {}", url, e);
            return createErrorResponse(e, url);
        } catch (Exception e) {
            log.error("代理GET请求异常: {}", url, e);
            return createErrorResponse(e, url);
        }
    }
    
    /**
     * 代理POST请求指定URL的内容
     * 
     * @param url 要代理的URL地址
     * @param token JWT token（可选）
     * @param body 请求体内容
     * @param request HTTP请求对象
     * @return 网页HTML内容
     */
    @PostMapping(produces = MediaType.TEXT_HTML_VALUE)
    public ResponseEntity<?> proxyPostUrl(@RequestParam String url, 
                                         @RequestParam(required = false) String token,
                                         @RequestBody(required = false) String body,
                                         HttpServletRequest request) {
        try {
            log.debug("代理POST请求URL: {}, token: {}", url, token != null ? "provided" : "null");
            
            // 验证URL格式
            if (url == null || url.trim().isEmpty()) {
                log.warn("无效的URL参数: 为空");
                return createBadRequestResponse("URL is empty");
            }
            
            // 获取网页内容
            String content = fetchWebContentWithMethod(url, "POST", body, request);
            log.debug("成功代理URL内容，长度: {} bytes", content.length());
            
            // 处理内容以提高兼容性
            String processedContent = processContentForEmbedding(content, url);
            
            return ResponseEntity.ok()
                .contentType(MediaType.TEXT_HTML)
                .body(processedContent);
                
        } catch (Exception e) {
            log.error("代理POST请求异常: {}", url, e);
            return createErrorResponse(e, url);
        }
    }
    
    /**
     * 处理本地文件请求
     * 
     * @param fileUrl 文件URL
     * @return 文件内容响应
     */
    private ResponseEntity<?> handleLocalFile(String fileUrl) {
        try {
            log.debug("处理本地文件请求: {}", fileUrl);
            
            // 将file:// URL转换为本地路径
            String filePath = fileUrl.substring("file://".length());
            // 处理Windows路径
            if (filePath.startsWith("/")) {
                // 去掉开头的斜杠（Windows file:// URL格式）
                filePath = filePath.substring(1);
            }
            
            // 规范化路径
            File file = new File(filePath);
            String canonicalPath = file.getCanonicalPath();
            
            // 安全检查：确保文件在允许的目录内
            if (!isFileAccessAllowed(canonicalPath)) {
                log.warn("拒绝访问本地文件，超出允许范围: {}", canonicalPath);
                return createForbiddenResponse("Access to this file is not allowed.");
            }
            
            // 检查文件是否存在
            if (!file.exists()) {
                log.warn("本地文件不存在: {}", canonicalPath);
                return createNotFoundResponse("The requested file was not found.");
            }
            
            // 检查是否为文件（而非目录）
            if (!file.isFile()) {
                log.warn("本地路径不是文件: {}", canonicalPath);
                return createBadRequestResponse("The requested path is not a file.");
            }
            
            // 读取文件内容
            String content = readFileContent(file);
            log.debug("成功读取本地文件内容，长度: {} bytes", content.length());
            
            // 处理内容以提高兼容性
            String processedContent = processContentForEmbedding(content, fileUrl);
            
            return ResponseEntity.ok()
                .contentType(MediaType.TEXT_HTML)
                .body(processedContent);
                
        } catch (Exception e) {
            log.error("处理本地文件异常: {}", fileUrl, e);
            return createErrorResponse(e, fileUrl);
        }
    }
    
    /**
     * 检查文件访问是否被允许
     * 
     * @param filePath 文件路径
     * @return 是否允许访问
     */
    private boolean isFileAccessAllowed(String filePath) {
        try {
            // 获取项目根目录
            Path projectRoot = Paths.get("").toAbsolutePath().normalize();
            Path requestedPath = Paths.get(filePath).normalize();
            
            // 检查文件是否在项目目录内
            return requestedPath.startsWith(projectRoot) && 
                   Files.exists(requestedPath) && 
                   !Files.isDirectory(requestedPath);
        } catch (Exception e) {
            log.error("检查文件访问权限时出错: ", e);
            return false;
        }
    }
    
    /**
     * 读取文件内容
     * 
     * @param file 文件对象
     * @return 文件内容
     * @throws IOException 读取异常
     */
    private String readFileContent(File file) throws IOException {
        try {
            return Files.readString(file.toPath(), StandardCharsets.UTF_8);
        } catch (IOException e) {
            log.error("读取文件失败: {}", file.getAbsolutePath(), e);
            throw e;
        }
    }
    
    /**
     * 获取网页内容
     * 默认启用自动重定向
     */
    private String fetchWebContent(String urlStr) throws IOException {
        return fetchWebContentWithAutoRedirects(urlStr);
    }
    
    /**
     * 获取网页内容，启用自动重定向
     */
    private String fetchWebContentWithAutoRedirects(String urlStr) throws IOException {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        
        try {
            URL url = URI.create(urlStr).toURL();
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            
            // 启用自动重定向（默认行为）
            connection.setInstanceFollowRedirects(true);
            
            // 设置请求头
            connection.setRequestProperty("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36");
            connection.setRequestProperty("Accept", "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8");
            connection.setRequestProperty("Accept-Language", "zh-CN,zh;q=0.9,en;q=0.8");
            connection.setRequestProperty("Accept-Encoding", "gzip, deflate, br");
            connection.setRequestProperty("Connection", "keep-alive");
            connection.setRequestProperty("Upgrade-Insecure-Requests", "1");
            connection.setConnectTimeout(15000); // 增加超时时间到15秒
            connection.setReadTimeout(15000);
            
            // 获取响应
            int responseCode = connection.getResponseCode();
            String xFrameOptions = connection.getHeaderField("X-Frame-Options");
            String csp = connection.getHeaderField("Content-Security-Policy");
            String contentType = connection.getHeaderField("Content-Type");
            
            log.debug("网页请求 - URL: {}, 响应码: {}, Content-Type: {}", urlStr, responseCode, contentType);
            
            if (xFrameOptions != null) {
                log.warn("检测到X-Frame-Options头部 - URL: {}, X-Frame-Options: {}", urlStr, xFrameOptions);
            }
            if (csp != null) {
                log.debug("检测到CSP头部 - URL: {}, CSP: {}", urlStr, csp);
            }
            
            // 检查Content-Type是否为HTML
            if (contentType != null && !contentType.toLowerCase().contains("text/html")) {
                log.warn("非HTML内容类型 - URL: {}, Content-Type: {}", urlStr, contentType);
                // 仍然继续处理，但记录警告
            }
            
            reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"));
            
            String line;
            while ((line = reader.readLine()) != null) {
                content.append(line).append("\n");
            }
            
        } catch (IOException e) {
            log.error("获取网页内容异常 - URL: {}, 错误堆栈: ", urlStr, e);
            throw e;
        } catch (Exception e) {
            log.error("获取网页内容未知异常 - URL: {}, 错误堆栈: ", urlStr, e);
            throw new IOException("获取网页内容时发生未知错误: " + e.getMessage(), e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    log.debug("关闭读取器失败: {}", e.getMessage());
                }
            }
        }
        
        return content.toString();
    }
    
    /**
     * 使用指定HTTP方法获取网页内容
     * 默认启用自动重定向
     */
    private String fetchWebContentWithMethod(String urlStr, String method, String body, HttpServletRequest request) throws IOException {
        return fetchWebContentWithMethodAndAutoRedirects(urlStr, method, body, request);
    }
    
    /**
     * 使用指定HTTP方法获取网页内容，启用自动重定向
     */
    private String fetchWebContentWithMethodAndAutoRedirects(String urlStr, String method, String body, HttpServletRequest request) throws IOException {
        URL url = URI.create(urlStr).toURL();        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        
        // 启用自动重定向
        connection.setInstanceFollowRedirects(true);
        
        // 设置请求方法
        connection.setRequestMethod(method);
        
        // 复制请求头
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = headerNames.nextElement();
            String headerValue = request.getHeader(headerName);
            // 过滤掉一些可能会干扰的头部
            if (!headerName.equalsIgnoreCase("host") && 
                !headerName.equalsIgnoreCase("content-length") &&
                !headerName.equalsIgnoreCase("connection")) {
                connection.setRequestProperty(headerName, headerValue);
            }
        }
        
        // 设置默认请求头
        connection.setRequestProperty("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36");
        connection.setConnectTimeout(15000);
        connection.setReadTimeout(15000);
        
        // 发送请求体（如果是POST）
        if ("POST".equals(method) && body != null) {
            connection.setDoOutput(true);
            try (OutputStream os = connection.getOutputStream()) {
                byte[] input = body.getBytes("utf-8");
                os.write(input, 0, input.length);
            }
        }
        
        // 读取响应
        StringBuilder content = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"))) {
            String line;
            while ((line = reader.readLine()) != null) {
                content.append(line).append("\n");
            }
        }
        
        return content.toString();
    }
    
    /**
     * 处理内容以提高在iframe中的兼容性
     * 移除或修改可能阻止嵌入的头部和脚本
     */
    private String processContentForEmbedding(String content, String originalUrl) {
        try {
            log.debug("开始处理内容以提高嵌入兼容性，原始长度: {}", content.length());
            
            // 使用OWASP HTML Sanitizer清理HTML内容
            String sanitized = HTML_POLICY.sanitize(content);
            
            // 移除X-Frame-Options meta标签
            String processed = sanitized.replaceAll("(?i)<meta[^>]*http-equiv\\s*=\\s*[\"']?X-Frame-Options[\"']?[^>]*/?>", "");
            
            // 移除Content-Security-Policy meta标签中可能阻止嵌入的指令
            processed = processed.replaceAll("(?i)<meta[^>]*http-equiv\\s*=\\s*[\"']?Content-Security-Policy[\"']?[^>]*content\\s*=\\s*[\"'][^\"']*frame-ancestors[^\"']*[\"'][^>]*/?>", "");
            
            // 移除可能阻止嵌入的JavaScript代码
            processed = processed.replaceAll("(?i)<script[^>]*>[^<]*top\\s*!=\\s*self[^<]*</script>", "");
            processed = processed.replaceAll("(?i)<script[^>]*>[^<]*window\\.top\\s*!=\\s*window[^<]*</script>", "");
            processed = processed.replaceAll("(?i)<script[^>]*>[^<]*document\\.location\\.ancestorOrigins[^<]*</script>", "");
            
            // 添加基础标签以确保相对链接正确
            if (!processed.contains("<base")) {
                String baseUrl = originalUrl.substring(0, originalUrl.lastIndexOf('/') + 1);
                processed = processed.replaceFirst("(?i)(<head[^>]*>)", "$1<base href=\"" + baseUrl + "\">");
            }
            
            // 添加样式以改善嵌入体验
            String styleAddition = "<style>" +
                "body { margin: 0; padding: 10px; font-family: Arial, sans-serif; }" +
                "img { max-width: 100%; height: auto; }" +
                "iframe { max-width: 100%; }" +
                "</style>";
                
            processed = processed.replaceFirst("(?i)(</head>)", styleAddition + "$1");
            
            log.debug("内容处理完成，处理后长度: {}", processed.length());
            return processed;
        } catch (Exception e) {
            log.error("处理内容以提高嵌入兼容性时出错: ", e);
            // 如果处理失败，返回原始内容
            return content;
        }
    }
    
    /**
     * 检查是否为大文件
     */
    private boolean isLargeFile(String url) {
        // 根据URL后缀判断是否为大文件
        return url.matches(".*\\.(pdf|zip|mp4|avi|mov|wmv|flv|mkv|webm|mp3|wav|flac|aac|ogg)$");
    }
    
    /**
     * 流式传输大文件
     * 默认启用自动重定向
     */
    private ResponseEntity<InputStreamResource> streamLargeFile(String urlStr) throws IOException {
        return streamLargeFileWithAutoRedirects(urlStr);
    }
    
    /**
     * 流式传输大文件，启用自动重定向
     */
    private ResponseEntity<InputStreamResource> streamLargeFileWithAutoRedirects(String urlStr) throws IOException {
        URL url = URI.create(urlStr).toURL();
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        
        // 启用自动重定向
        connection.setInstanceFollowRedirects(true);
        
        // 设置请求头
        connection.setRequestProperty("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36");
        connection.setConnectTimeout(15000);
        connection.setReadTimeout(15000);
        
        java.io.InputStream inputStream = connection.getInputStream();
        InputStreamResource resource = new InputStreamResource(inputStream);
        
        // 获取内容类型
        String contentType = connection.getContentType();
        if (contentType == null) {
            contentType = "application/octet-stream";
        }
        
        return ResponseEntity.ok()
                .contentType(MediaType.parseMediaType(contentType))
                .contentLength(connection.getContentLength())
                .header("Content-Disposition", "inline; filename=\"" + getFileNameFromUrl(urlStr) + "\"")
                .body(resource);
    }
    
    /**
     * 从URL中提取文件名
     */
    private String getFileNameFromUrl(String url) {
        try {
            String path = URI.create(url).toURL().getPath();
            int lastSlashIndex = path.lastIndexOf('/');
            if (lastSlashIndex >= 0 && lastSlashIndex < path.length() - 1) {
                return path.substring(lastSlashIndex + 1);
            }
        } catch (Exception e) {
            log.warn("从URL提取文件名失败: {}", url, e);
        }
        return "file";
    }
    
    /**
     * 创建错误响应
     */
    private ResponseEntity<String> createErrorResponse(Exception e, String url) {
        String errorMessage = e.getMessage();
        if (errorMessage == null || errorMessage.isEmpty()) {
            errorMessage = "Unknown error";
        }
        
        int statusCode = 500;
        String errorTitle = "Internal Server Error";
        
        if (e instanceof IOException) {
            statusCode = 502;
            errorTitle = "Network Error";
        }
        
        String htmlBody = String.format(
            "<html><body><h2>Error: %s</h2><p>An error occurred while fetching content from %s</p><p>%s</p><p>Type: %s</p></body></html>",
            errorTitle, url, errorMessage, e.getClass().getSimpleName()
        );
        
        return ResponseEntity.status(statusCode)
            .contentType(MediaType.TEXT_HTML)
            .body(htmlBody);
    }
    
    /**
     * 创建Bad Request响应
     */
    private ResponseEntity<String> createBadRequestResponse(String message) {
        String htmlBody = String.format(
            "<html><body><h2>Error: Bad Request</h2><p>%s</p></body></html>", 
            message
        );
        return ResponseEntity.badRequest()
            .contentType(MediaType.TEXT_HTML)
            .body(htmlBody);
    }
    
    /**
     * 创建Not Found响应
     */
    private ResponseEntity<String> createNotFoundResponse(String message) {
        String htmlBody = String.format(
            "<html><body><h2>Error: Not Found</h2><p>%s</p></body></html>", 
            message
        );
        return ResponseEntity.status(404)
            .contentType(MediaType.TEXT_HTML)
            .body(htmlBody);
    }
    
    /**
     * 创建Forbidden响应
     */
    private ResponseEntity<String> createForbiddenResponse(String message) {
        String htmlBody = String.format(
            "<html><body><h2>Error: Forbidden</h2><p>%s</p></body></html>", 
            message
        );
        return ResponseEntity.status(403)
            .contentType(MediaType.TEXT_HTML)
            .body(htmlBody);
    }
}