package pangea.hiagent.web.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.common.exception.BusinessException;
import pangea.hiagent.common.exception.ErrorCode;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.AgentDialogue;
import pangea.hiagent.model.LlmConfig;
import pangea.hiagent.model.Tool;
import pangea.hiagent.web.repository.AgentDialogueRepository;
import pangea.hiagent.web.repository.AgentRepository;
import pangea.hiagent.web.repository.LlmConfigRepository;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.llm.LlmModelFactory;

import java.util.ArrayList;
import java.util.List;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;

/**
 * Agent服务类
 * 负责Agent的管理和相关业务逻辑
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class AgentService {
    
    private final AgentRepository agentRepository;
    private final AgentDialogueRepository agentDialogueRepository;
    private final LlmConfigRepository llmConfigRepository;
    private final LlmModelFactory llmModelFactory;
    private final AgentToolRelationService agentToolRelationService;
    private final ToolService toolService;
    
    /**
     * 创建Agent
     */
    @Transactional
    @CacheEvict(value = {"agents", "agent"}, allEntries = true)
    public Agent createAgent(Agent agent) {
        log.info("创建Agent: {}", agent.getName());
        
        // 设置默认值
        if (agent.getTemperature() == null) {
            agent.setTemperature(0.7);
        }
        if (agent.getMaxTokens() == null) {
            agent.setMaxTokens(4096);
        }
        if (agent.getTopP() == null) {
            agent.setTopP(0.9);
        }
        if (agent.getTopK() == null) {
            agent.setTopK(50);
        }
        if (agent.getPresencePenalty() == null) {
            agent.setPresencePenalty(0.0);
        }
        if (agent.getFrequencyPenalty() == null) {
            agent.setFrequencyPenalty(0.0);
        }
        if (agent.getHistoryLength() == null) {
            agent.setHistoryLength(10);
        }
        if (agent.getStatus() == null || agent.getStatus().isEmpty()) {
            agent.setStatus("active");
        }
        
        agentRepository.insert(agent);
        return agent;
    }
    
    /**
     * 创建Agent并设置工具关联
     */
    @Transactional
    @CacheEvict(value = {"agents", "agent"}, allEntries = true)
    public Agent createAgentWithTools(Agent agent) {
        log.info("创建Agent并设置工具关联: {}", agent.getName());
        
        try {
            // 先创建Agent
            Agent createdAgent = createAgent(agent);
            
            // 如果Agent有工具列表，则设置工具关联
            if (agent.getTools() != null && !agent.getTools().isEmpty()) {
                // 提取工具ID列表
                List<String> toolIds = agent.getTools().stream()
                    .map(Tool::getId)
                    .filter(id -> id != null && !id.trim().isEmpty())
                    .distinct()
                    .toList();
                
                log.debug("Agent {} 设置工具数量: {}", createdAgent.getId(), toolIds.size());
                // 设置工具关联
                agentToolRelationService.setAgentTools(createdAgent.getId(), toolIds);
            }
            
            log.info("Agent {} 创建并关联工具成功", createdAgent.getId());
            return createdAgent;
        } catch (Exception e) {
            log.error("创建Agent并设置工具关联失败", e);
            throw new BusinessException(ErrorCode.SYSTEM_ERROR.getCode(), "创建Agent并设置工具关联失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新Agent
     */
    @Transactional
    @CacheEvict(value = {"agents", "agent"}, allEntries = true)
    public Agent updateAgent(Agent agent) {
        log.info("更新Agent: {}", agent.getId());
        
        try {
            // 验证必要字段
            if (agent.getId() == null || agent.getId().trim().isEmpty()) {
                throw new BusinessException(ErrorCode.PARAMETER_ERROR.getCode(), "Agent ID不能为空");
            }
            
            if (agent.getName() == null || agent.getName().trim().isEmpty()) {
                throw new BusinessException(ErrorCode.PARAMETER_ERROR.getCode(), "Agent名称不能为空");
            }
            
            if (agent.getDefaultModel() == null || agent.getDefaultModel().trim().isEmpty()) {
                throw new BusinessException(ErrorCode.PARAMETER_ERROR.getCode(), "默认模型不能为空");
            }
            
            if (agent.getStatus() == null || agent.getStatus().trim().isEmpty()) {
                throw new BusinessException(ErrorCode.PARAMETER_ERROR.getCode(), "状态不能为空");
            }
            
            // 保留原始所有者信息
            Agent existingAgent = agentRepository.selectById(agent.getId());
            if (existingAgent == null) {
                throw new BusinessException(ErrorCode.AGENT_NOT_FOUND.getCode(), "Agent不存在");
            }
            
            // 验证用户权限（确保用户是所有者）
            String currentUserId = UserUtils.getCurrentUserIdStatic();
            if (currentUserId == null) {
                log.warn("用户未认证，无法更新Agent: {}", agent.getId());
                throw new BusinessException(ErrorCode.UNAUTHORIZED.getCode(), "用户未认证");
            }
            
            if (!currentUserId.equals(existingAgent.getOwner())) {
                log.warn("权限不足，用户 {} 无法更新Agent: {}", currentUserId, agent.getId());
                throw new BusinessException(ErrorCode.PERMISSION_DENIED.getCode(), "权限不足，无法更新此Agent");
            }
            
            // 保留不可更改的字段
            agent.setOwner(existingAgent.getOwner());
            agent.setCreatedBy(existingAgent.getCreatedBy());
            agent.setCreatedAt(existingAgent.getCreatedAt());
            
            // 处理可能为null的数值字段，设置默认值
            if (agent.getTemperature() == null) {
                agent.setTemperature(existingAgent.getTemperature() != null ? existingAgent.getTemperature() : 0.7);
            }
            
            if (agent.getMaxTokens() == null) {
                agent.setMaxTokens(existingAgent.getMaxTokens() != null ? existingAgent.getMaxTokens() : 4096);
            }
            
            if (agent.getTopP() == null) {
                agent.setTopP(existingAgent.getTopP() != null ? existingAgent.getTopP() : 0.9);
            }
            
            if (agent.getTopK() == null) {
                agent.setTopK(existingAgent.getTopK() != null ? existingAgent.getTopK() : 50);
            }
            
            if (agent.getPresencePenalty() == null) {
                agent.setPresencePenalty(existingAgent.getPresencePenalty() != null ? existingAgent.getPresencePenalty() : 0.0);
            }
            
            if (agent.getFrequencyPenalty() == null) {
                agent.setFrequencyPenalty(existingAgent.getFrequencyPenalty() != null ? existingAgent.getFrequencyPenalty() : 0.0);
            }
            
            if (agent.getHistoryLength() == null) {
                agent.setHistoryLength(existingAgent.getHistoryLength() != null ? existingAgent.getHistoryLength() : 10);
            }
            
            // 处理Boolean字段，确保不会因为前端传递null而导致问题
            if (agent.getEnableRag() == null) {
                agent.setEnableRag(existingAgent.getEnableRag() != null ? existingAgent.getEnableRag() : false);
            }
            
            if (agent.getEnableReAct() == null) {
                agent.setEnableReAct(existingAgent.getEnableReAct() != null ? existingAgent.getEnableReAct() : false);
            }
            
            if (agent.getEnableStreaming() == null) {
                agent.setEnableStreaming(existingAgent.getEnableStreaming() != null ? existingAgent.getEnableStreaming() : true);
            }
            
            agentRepository.updateById(agent);
            log.info("成功更新Agent: {}", agent.getId());
            return agent;
        } catch (BusinessException e) {
            log.warn("更新Agent业务异常，Agent ID: {}, 错误代码: {}, 错误信息: {}", agent.getId(), e.getCode(), e.getMessage());
            throw e;
        } catch (Exception e) {
            log.error("更新Agent失败，Agent ID: {}", agent.getId(), e);
            throw new BusinessException(ErrorCode.SYSTEM_ERROR.getCode(), "更新Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新Agent并设置工具关联
     */
    @Transactional
    @CacheEvict(value = {"agents", "agent"}, allEntries = true)
    public Agent updateAgentWithTools(Agent agent) {
        log.info("更新Agent并设置工具关联: {}", agent.getId());
        
        try {
            // 先更新Agent
            Agent updatedAgent = updateAgent(agent);
            
            // 更新工具关联关系
            List<String> toolIds = new ArrayList<>();
            if (agent.getTools() != null && !agent.getTools().isEmpty()) {
                // 提取工具ID列表
                toolIds = agent.getTools().stream()
                    .map(Tool::getId)
                    .filter(id -> id != null && !id.trim().isEmpty())
                    .distinct()
                    .toList();
                log.debug("Agent {} 更新工具数量: {}", updatedAgent.getId(), toolIds.size());
            } else {
                log.debug("Agent {} 清除所有工具关联", updatedAgent.getId());
            }
            
            // 设置工具关联
            agentToolRelationService.setAgentTools(updatedAgent.getId(), toolIds);
            
            log.info("Agent {} 更新并关联工具成功", updatedAgent.getId());
            return updatedAgent;
        } catch (Exception e) {
            log.error("更新Agent并设置工具关联失败，Agent ID: {}", agent.getId(), e);
            throw new BusinessException(ErrorCode.SYSTEM_ERROR.getCode(), "更新Agent并设置工具关联失败: " + e.getMessage());
        }
    }
    
    /**
     * 删除Agent（包括相关的工具关联）
     */
    @Transactional
    @CacheEvict(value = {"agents", "agent"}, allEntries = true)
    public void deleteAgent(String id) {
        log.info("删除Agent: {}", id);
        
        try {
            // 首先删除Agent的所有工具关联
            agentToolRelationService.removeAgentAllTools(id);
            log.debug("已删除Agent {} 的所有工具关联", id);
            
            // 然后删除Agent
            agentRepository.deleteById(id);
            log.info("Agent {} 删除成功", id);
        } catch (Exception e) {
            log.error("删除Agent失败，Agent ID: {}", id, e);
            throw new BusinessException(ErrorCode.SYSTEM_ERROR.getCode(), "删除Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取Agent详情
     * 
     * @param id Agent ID
     * @return Agent对象，如果不存在则返回null
     */
    @Cacheable(value = "agent", key = "#id")
    public Agent getAgent(String id) {
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取Agent");
            return null;
        }
        return agentRepository.selectById(id);
    }
    
    /**
     * 获取带有关联工具信息的Agent详情
     * 
     * @param id Agent ID
     * @return Agent对象，如果不存在则返回null
     */
    public Agent getAgentWithTools(String id) {
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取Agent");
            return null;
        }
        
        Agent agent = agentRepository.selectById(id);
        if (agent != null) {
            // 获取关联的工具列表
            List<String> toolIds = agentToolRelationService.getToolIdsByAgentId(id);
            // 根据工具ID列表获取工具详情
            if (!toolIds.isEmpty()) {
                List<Tool> tools = toolService.listByIds(toolIds);
                agent.setTools(tools);
            } else {
                agent.setTools(new ArrayList<>());
            }
        }
        return agent;
    }
    
    /**
     * 根据ID获取Agent详情
     * 
     * @param id Agent ID
     * @return Agent对象，如果不存在则返回null
     */
    public Agent getAgentById(Long id) {
        if (id == null) {
            log.warn("尝试使用无效ID获取Agent");
            return null;
        }
        return agentRepository.selectById(id.toString());
    }    
    /**
     * 获取Agent列表
     * 
     * @return Agent列表
     */
    @Cacheable(value = "agents")
    public List<Agent> listAgents() {
        List<Agent> agents = agentRepository.selectList(null);
        log.info("获取到 {} 个Agent", agents != null ? agents.size() : 0);
        return agents != null ? agents : List.of();
    }
    
    /**
     * 分页获取Agent列表
     */
    public IPage<Agent> pageAgents(Long current, Long size, String name, String status) {
        Page<Agent> page = new Page<>(current, size);
        
        LambdaQueryWrapper<Agent> wrapper = new LambdaQueryWrapper<>();
        if (name != null) {
            wrapper.like(Agent::getName, name);
        }
        if (status != null) {
            wrapper.eq(Agent::getStatus, status);
        }
        
        // 使用优化的分页查询方法
        return agentRepository.selectPageWithOptimization(page, wrapper);
    }
    
    /**
     * 获取用户的Agent列表
     */
    @Cacheable(value = "agents", key = "#userId")
    public List<Agent> getUserAgents(String userId) {
        // 使用优化的查询方法
        return agentRepository.findActiveAgentsByOwnerWithExplicitColumns(userId);
    }
    
    /**
     * 保存对话记录
     */
    @Transactional
    public void saveDialogue(AgentDialogue dialogue) {
        // 确保ID被设置
        if (dialogue.getId() == null || dialogue.getId().isEmpty()) {
            dialogue.setId(java.util.UUID.randomUUID().toString());
        }
        
        // 确保创建时间和更新时间被设置
        if (dialogue.getCreatedAt() == null) {
            dialogue.setCreatedAt(java.time.LocalDateTime.now());
        }
        if (dialogue.getUpdatedAt() == null) {
            dialogue.setUpdatedAt(java.time.LocalDateTime.now());
        }
        
        agentDialogueRepository.insert(dialogue);
    }
    
    /**
     * 获取指定Agent和用户的对话历史记录
     * @param agentId Agent ID
     * @param userId 用户ID
     * @param limit 限制数量
     * @return 对话历史列表
     */
    public List<AgentDialogue> getDialogueHistory(String agentId, String userId, int limit) {
        LambdaQueryWrapper<AgentDialogue> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(AgentDialogue::getAgentId, agentId)
               .eq(AgentDialogue::getUserId, userId)
               .orderByDesc(AgentDialogue::getCreatedAt)
               .last("LIMIT " + limit);
        
        List<AgentDialogue> dialogues = agentDialogueRepository.selectList(wrapper);
        // 反转列表使其按时间正序排列
        java.util.Collections.reverse(dialogues);
        return dialogues;
    }
    
    /**
     * 根据Agent获取对应的LLM模型
     * 
     * @param agent Agent对象
     * @return 对应的ChatModel实例
     */
    public org.springframework.ai.chat.model.ChatModel getChatModelForAgent(Agent agent) {
        // 获取Agent配置的模型配置名称
        String modelConfigName = agent.getDefaultModel();
        log.info("获取Agent {} 的ChatModel，模型配置名称: {}", agent.getId(), modelConfigName);
        
        if (modelConfigName == null || modelConfigName.isEmpty()) {
            throw new IllegalArgumentException("Agent未配置默认模型");
        }
        
        // 根据模型配置名称查找对应的LLM配置
        LambdaQueryWrapper<LlmConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LlmConfig::getName, modelConfigName);
        wrapper.eq(LlmConfig::getEnabled, true);
        LlmConfig llmConfig = llmConfigRepository.selectOne(wrapper);
        
        if (llmConfig == null) {
            log.error("未找到启用的LLM配置: {}", modelConfigName);
            throw new IllegalArgumentException("未找到启用的LLM配置: " + modelConfigName);
        }
        
        log.info("找到LLM配置: {} ({})", llmConfig.getName(), llmConfig.getProvider());
        
        // 使用LlmModelFactory创建对应的ChatModel实例
        return llmModelFactory.createChatModel(llmConfig);
    }
}