package pangea.hiagent.tools;

import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import com.microsoft.playwright.options.WaitUntilState;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;

import java.io.File;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * 海信SSO认证工具类
 * 用于访问需要SSO认证的海信业务系统，自动完成登录并提取页面内容
 */
@Slf4j
@Component
public class HisenseSsoAuthTool {

    // SSO登录页面URL
    private static final String SSO_LOGIN_URL = "https://sso.hisense.com/login/";

    // 用户名输入框选择器
    private static final String USERNAME_INPUT_SELECTOR = "input[placeholder='账号名/海信邮箱/手机号']";
    
    // 密码输入框选择器
    private static final String PASSWORD_INPUT_SELECTOR = "input[placeholder='密码'][type='password']";
    
    // 登录按钮选择器
    private static final String LOGIN_BUTTON_SELECTOR = "#login-button";

    // Playwright实例
    private Playwright playwright;
    
    // 浏览器实例
    private Browser browser;
    
    // 共享的浏览器上下文，用于保持登录状态
    private BrowserContext sharedContext;
    
    // 上次登录时间
    private long lastLoginTime = 0;
    
    // 登录状态有效期（毫秒），设置为30分钟
    private static final long LOGIN_VALIDITY_PERIOD = 30 * 60 * 1000;

    // SSO用户名（从配置文件读取）
    @Value("${hisense.sso.username:}")
    private String ssoUsername;
    
    // SSO密码（从配置文件读取）
    @Value("${hisense.sso.password:}")
    private String ssoPassword;

    // 存储目录路径
    private static final String STORAGE_DIR = "storage";

    /**
     * 初始化Playwright和浏览器实例
     */
    @PostConstruct
    public void initialize() {
        try {
            log.info("正在初始化海信SSO认证工具的Playwright...");
            this.playwright = Playwright.create();
            // 使用chromium浏览器，无头模式（headless=true），适合服务器运行
            // 可根据需要修改为有头模式（headless=false）用于调试
            this.browser = playwright.chromium().launch(new BrowserType.LaunchOptions().setHeadless(true));
            // 初始化共享上下文
            this.sharedContext = browser.newContext();
            log.info("海信SSO认证工具的Playwright初始化成功");
        } catch (Exception e) {
            log.error("海信SSO认证工具的Playwright初始化失败: ", e);
        }
    }

    /**
     * 销毁Playwright资源
     */
    @PreDestroy
    public void destroy() {
        try {
            if (sharedContext != null) {
                sharedContext.close();
                log.info("海信SSO认证工具的共享浏览器上下文已关闭");
            }
            if (browser != null) {
                browser.close();
                log.info("海信SSO认证工具的浏览器实例已关闭");
            }
            if (playwright != null) {
                playwright.close();
                log.info("海信SSO认证工具的Playwright实例已关闭");
            }
        } catch (Exception e) {
            log.error("海信SSO认证工具的Playwright资源释放失败: ", e);
        }
    }

    /**
     * 工具方法：获取海信业务系统的网页内容（自动处理SSO认证）
     * 
     * @param businessSystemUrl 海信业务系统页面URL
     * @return 页面内容（HTML文本）
     */
    @Tool(description = "获取海信业务系统的网页内容（自动处理SSO认证）")
    public String getHisenseBusinessSystemContent(String businessSystemUrl) {
        log.info("开始获取海信业务系统内容，URL: {}", businessSystemUrl);
        
        // 校验SSO凭证是否配置
        if (ssoUsername == null || ssoUsername.isEmpty() || ssoPassword == null || ssoPassword.isEmpty()) {
            String errorMsg = "SSO用户名或密码未配置，海信SSO工具不可用";
            log.warn(errorMsg);
            return errorMsg;
        }
        
        long startTime = System.currentTimeMillis();
        
        // 参数校验
        if (businessSystemUrl == null || businessSystemUrl.isEmpty()) {
            String errorMsg = "业务系统URL不能为空";
            log.error(errorMsg);
            return errorMsg;
        }

        Page page = null;
        
        try {
            // 检查是否已有有效的登录会话
            boolean sessionValid = isSessionLoggedIn() && validateSession(businessSystemUrl);
            
            if (sessionValid) {
                log.info("检测到有效会话，直接使用共享上下文");
                page = sharedContext.newPage();
            } else {
                log.info("未检测到有效会话，使用共享上下文并重新登录");
                page = sharedContext.newPage();
                
                // 访问业务系统页面
                log.info("正在访问业务系统页面: {}", businessSystemUrl);
                page.navigate(businessSystemUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                
                // 检查是否重定向到了SSO登录页面
                String currentUrl = page.url();
                log.info("当前页面URL: {}", currentUrl);
                
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    log.info("检测到SSO登录页面，开始自动登录...");
                    // 执行SSO登录
                    performLoginAndUpdateStatus(page);
                    
                    // 等待登录完成并重定向回业务系统
                    page.waitForURL(businessSystemUrl, new Page.WaitForURLOptions().setTimeout(10000));
                    log.info("登录成功，已重定向回业务系统页面");
                } else {
                    // 即使没有跳转到登录页面，也更新登录时间
                    lastLoginTime = System.currentTimeMillis();
                    log.info("直接访问业务系统页面成功，无需SSO登录，更新会话时间");
                }
            }
            
            // 如果页面尚未导航到业务系统URL，则导航到该URL
            if (!page.url().equals(businessSystemUrl) && !page.url().startsWith(businessSystemUrl)) {
                log.info("正在访问业务系统页面: {}", businessSystemUrl);
                page.navigate(businessSystemUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            }
            
            // 提取页面内容
            String content = page.locator("body").innerText();
            long endTime = System.currentTimeMillis();
            log.info("成功获取业务系统页面内容，耗时: {} ms", endTime - startTime);
            
            return content;
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "获取海信业务系统内容失败: " + e.getMessage();
            log.error("获取海信业务系统内容失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        } finally {
            // 释放页面资源
            if (page != null) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    /**
     * 工具方法：处理海信请假审批
     * 
     * @param approvalUrl 请假审批页面URL
     * @param approvalOpinion 审批意见
     * @return 处理结果
     */
    @Tool(description = "处理海信请假审批、自驾车审批、调休审批")
    public String processHisenseLeaveApproval(String approvalUrl, String approvalOpinion) {
        log.info("开始处理海信请假审批，URL: {}", approvalUrl);
        
        // 校验SSO凭证是否配置
        if (ssoUsername == null || ssoUsername.isEmpty() || ssoPassword == null || ssoPassword.isEmpty()) {
            String errorMsg = "SSO用户名或密码未配置，海信SSO工具不可用";
            log.warn(errorMsg);
            return errorMsg;
        }
        
        long startTime = System.currentTimeMillis();
        
        // 参数校验
        if (approvalUrl == null || approvalUrl.isEmpty()) {
            String errorMsg = "审批URL不能为空";
            log.error(errorMsg);
            return errorMsg;
        }
        
        if (approvalOpinion == null || approvalOpinion.isEmpty()) {
            String errorMsg = "审批意见不能为空";
            log.error(errorMsg);
            return errorMsg;
        }
        
        Page page = null;
        
        try {
            // 检查是否已有有效的登录会话
            boolean sessionValid = isSessionLoggedIn() && validateSession(approvalUrl);
            
            if (sessionValid) {
                log.info("检测到有效会话，直接使用共享上下文");
                page = sharedContext.newPage();
            } else {
                log.info("未检测到有效会话，使用共享上下文并重新登录");
                page = sharedContext.newPage();
                
                // 访问审批页面
                log.info("正在访问审批页面: {}", approvalUrl);
                page.navigate(approvalUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                
                // 检查是否重定向到了SSO登录页面
                String currentUrl = page.url();
                log.info("当前页面URL: {}", currentUrl);
                
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    log.info("检测到SSO登录页面，开始自动登录...");
                    // 执行SSO登录
                    performLoginAndUpdateStatus(page);
                    
                    // 等待登录完成并重定向回审批页面
                    page.waitForURL(approvalUrl, new Page.WaitForURLOptions().setTimeout(10000));
                    log.info("登录成功，已重定向回审批页面");
                } else {
                    // 即使没有跳转到登录页面，也更新登录时间
                    lastLoginTime = System.currentTimeMillis();
                    log.info("直接访问审批页面成功，无需SSO登录，更新会话时间");
                }
            }
            
            // 如果页面尚未导航到审批URL，则导航到该URL
            if (!page.url().equals(approvalUrl) && !page.url().startsWith(approvalUrl)) {
                log.info("正在访问审批页面: {}", approvalUrl);
                page.navigate(approvalUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            }
            
            // 执行审批操作
            performApprovalOperation(page, approvalOpinion);
            
            // 截图并保存
            takeScreenshotAndSave(page, "leave_approval_success");
            
            long endTime = System.currentTimeMillis();
            log.info("请假审批处理完成，耗时: {} ms", endTime - startTime);
            
            return "请假审批处理成功";
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "请假审批处理失败: " + e.getMessage();
            log.error("请假审批处理失败，耗时: {} ms", endTime - startTime, e);
            
            // 如果页面对象存在，截图保存错误页面
            if (page != null) {
                try {
                    takeScreenshotAndSave(page, "leave_approval_fail");
                } catch (Exception screenshotException) {
                    log.warn("截图保存失败: {}", screenshotException.getMessage());
                }
            }
            
            return errorMsg;
        }
        // 注意：这里不再释放页面资源，以保持会话状态供后续使用
        /*finally {
            // 释放页面资源
            if (page != null) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
        }*/
    }

    /**
     * 检查当前会话是否已登录
     * 
     * @return true表示已登录且会话有效，false表示未登录或会话已过期
     */
    private boolean isSessionLoggedIn() {
        // 检查是否存在共享上下文
        if (sharedContext == null) {
            return false;
        }
        
        // 检查登录是否过期
        long currentTime = System.currentTimeMillis();
        if (currentTime - lastLoginTime > LOGIN_VALIDITY_PERIOD) {
            log.debug("会话已过期，上次登录时间: {}，当前时间: {}", lastLoginTime, currentTime);
            return false;
        }
        
        return true;
    }
    
    /**
     * 验证当前会话是否仍然有效
     * 通过访问一个需要登录的页面来验证会话状态
     * 
     * @param testUrl 用于验证会话的测试页面URL
     * @return true表示会话有效，false表示会话无效
     */
    private boolean validateSession(String testUrl) {
        if (!isSessionLoggedIn()) {
            return false;
        }
        
        try {
            Page page = sharedContext.newPage();
            try {
                page.navigate(testUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                String currentUrl = page.url();
                
                // 如果重定向到了登录页面，说明会话已失效
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    log.debug("会话验证失败，已重定向到登录页面");
                    return false;
                }
                
                log.debug("会话验证成功，当前页面URL: {}", currentUrl);
                return true;
            } finally {
                page.close();
            }
        } catch (Exception e) {
            log.warn("会话验证过程中发生异常: {}", e.getMessage());
            return false;
        }
    }
    
    /**
     * 执行登录并更新登录状态
     * 
     * @param page 当前页面对象
     * @throws Exception 登录过程中的异常
     */
    private void performLoginAndUpdateStatus(Page page) throws Exception {
        log.info("开始执行SSO登录流程");
        
        try {
            // 填入用户名
            log.debug("正在定位用户名输入框: {}", USERNAME_INPUT_SELECTOR);
            Locator usernameInput = page.locator(USERNAME_INPUT_SELECTOR);
            if (usernameInput.count() == 0) {
                throw new RuntimeException("未找到用户名输入框");
            }
            usernameInput.fill(ssoUsername);
            log.debug("用户名输入完成");

            // 填入密码
            log.debug("正在定位密码输入框: {}", PASSWORD_INPUT_SELECTOR);
            Locator passwordInput = page.locator(PASSWORD_INPUT_SELECTOR);
            if (passwordInput.count() == 0) {
                throw new RuntimeException("未找到密码输入框");
            }
            passwordInput.fill(ssoPassword);
            log.debug("密码输入完成");

            // 点击登录按钮
            log.debug("正在定位登录按钮: {}", LOGIN_BUTTON_SELECTOR);
            Locator loginButton = page.locator(LOGIN_BUTTON_SELECTOR);
            if (loginButton.count() == 0) {
                throw new RuntimeException("未找到登录按钮");
            }
            loginButton.click();
            log.info("登录按钮点击完成，等待登录响应");
            
            // 等待页面开始跳转（表示登录请求已发送）
            page.waitForLoadState(LoadState.NETWORKIDLE);
            
            // 更新登录时间
            lastLoginTime = System.currentTimeMillis();
            log.info("SSO登录成功，登录时间已更新");
        } catch (Exception e) {
            log.error("SSO登录过程中发生异常", e);
            throw new RuntimeException("SSO登录失败: " + e.getMessage(), e);
        }
    }

    /**
     * 执行审批操作
     * 
     * @param page 当前页面对象
     * @param approvalOpinion 审批意见
     * @throws Exception 审批过程中的异常
     */
    private void performApprovalOperation(Page page, String approvalOpinion) throws Exception {
        log.info("开始执行审批操作");
        
        try {
            // 定位审批操作单选框
            String operationRadioSelector = "input[type='radio'][alerttext=''][key='operationType'][name='oprGroup'][value='handler_pass:通过']";
            log.debug("正在定位审批操作单选框: {}", operationRadioSelector);
            Locator operationRadio = page.locator(operationRadioSelector);
            if (operationRadio.count() == 0) {
                throw new RuntimeException("未找到审批操作单选框");
            }
            operationRadio.click();
            log.debug("审批操作单选框选择完成");

            // 定位审批意见输入框并填入内容
            String opinionTextareaSelector = "textarea[name='fdUsageContent'][class='process_review_content'][key='auditNode']";
            log.debug("正在定位审批意见输入框: {}", opinionTextareaSelector);
            Locator opinionTextarea = page.locator(opinionTextareaSelector);
            if (opinionTextarea.count() == 0) {
                throw new RuntimeException("未找到审批意见输入框");
            }
            opinionTextarea.fill(approvalOpinion);
            log.debug("审批意见输入完成");

            // 定位并点击提交按钮
            String submitButtonSelector = "input[id='process_review_button'][class='process_review_button'][type='button'][value='提交']";
            log.debug("正在定位提交按钮: {}", submitButtonSelector);
            Locator submitButton = page.locator(submitButtonSelector);
            if (submitButton.count() == 0) {
                throw new RuntimeException("未找到提交按钮");
            }
            submitButton.click();
            log.info("提交按钮点击完成");
            
            // 等待提交完成
            page.waitForLoadState(LoadState.NETWORKIDLE);
            log.info("审批操作执行完成");
        } catch (Exception e) {
            log.error("审批操作过程中发生异常", e);
            throw new RuntimeException("审批操作失败: " + e.getMessage(), e);
        }
    }

    /**
     * 截图并保存到存储目录
     * 
     * @param page 当前页面对象
     * @param fileName 文件名前缀
     */
    private void takeScreenshotAndSave(Page page, String fileName) {
        try {
            // 确保存储目录存在
            File storageDir = new File(STORAGE_DIR);
            if (!storageDir.exists()) {
                storageDir.mkdirs();
            }
            
            // 生成带时间戳的文件名
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
            String fullFileName = String.format("%s_%s.png", fileName, timestamp);
            String filePath = Paths.get(STORAGE_DIR, fullFileName).toString();
            
            // 截图并保存
            page.screenshot(new Page.ScreenshotOptions().setPath(Paths.get(filePath)));
            log.info("截图已保存至: {}", filePath);
        } catch (Exception e) {
            log.error("截图保存失败: {}", e.getMessage(), e);
        }
    }
}