package pangea.hiagent.react;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.*;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.context.annotation.Lazy;
import pangea.hiagent.service.AgentService;
import pangea.hiagent.service.InfoCollectorService;
import pangea.hiagent.tools.HisenseTripTool;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;
import pangea.hiagent.core.AgentChatService;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import java.util.List;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * 重构后的默认ReAct执行器实现 - 支持真正的流式输出和完整的ReAct流程可观测性
 * 
 * 核心设计：
 * 1. 使用真正的流式处理机制，逐步返回结果给客户端
 * 2. 提供完整的ReAct流程监控和事件通知
 * 3. 实现异步处理以提高性能
 * 4. 完善的异常处理和资源管理
 * 5. 支持对话历史记忆功能
 */
@Slf4j
@Service
public class DefaultReactExecutor implements ReactExecutor {
    
    private final List<ReactCallback> reactCallbacks = new ArrayList<>();
    private final AtomicInteger stepCounter = new AtomicInteger(0);
    
    @Autowired
    @Lazy
    private IWorkPanelDataCollector workPanelCollector;
    
    @Autowired
    private MemoryService memoryService;

    @Autowired
    private AgentService agentService;

    @Autowired
    private InfoCollectorService infoCollectorService;
    
    /**
     * 添加ReAct回调
     * @param callback ReAct回调
     */
    @Override
    public void addReactCallback(ReactCallback callback) {
        if (callback != null) {
            reactCallbacks.add(callback);
        }
    }
    
    /**
     * 执行ReAct流程（同步方式）
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表
     * @return 最终答案
     */
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools) {
        return executeWithAgent(chatClient, userInput, tools, null);
    }
    
    /**
     * 执行ReAct流程（同步方式）- 支持Agent配置
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表
     * @param agent Agent对象（可选）
     * @return 最终答案
     */
    public String executeWithAgent(ChatClient chatClient, String userInput, List<Object> tools, Agent agent) {
        log.info("开始执行ReAct流程，用户输入: {}", userInput);
        
        // 重置步骤计数器
        stepCounter.set(0);
        
        try {
            // 触发思考步骤
            triggerThinkStep("开始处理用户请求: " + userInput);
            
            // 构建系统提示词
            String systemPrompt = "You are a helpful AI assistant that can use tools to answer questions. " +
                "Use the available tools when needed to gather information. " +
                "Think step by step and show your reasoning process. " +
                "When using tools, clearly indicate your thoughts, actions, and observations.";
            
            // 构建Prompt，包含历史对话记录
            Prompt prompt = buildPromptWithHistory(systemPrompt, userInput, agent);
            
            // 使用call()获取完整的LLM响应
            // 这会阻塞直到完成整个ReAct循环（思考→行動→观察→...→最终答案）
            log.info("使用call()方法处理ReAct流程，确保完整的工具调用循环");
            
            ChatResponse response = chatClient.prompt(prompt)
                    .tools(tools.toArray())
                    .call()
                    .chatResponse();
            
            // 获取响应文本
            String responseText = response.getResult().getOutput().getText();
            
            // 触发观察步骤
            triggerObservationStep(responseText);
            
            // 返回最终结果
            log.info("最终答案: {}", responseText);
            
            // 触发最终答案步骤
            triggerFinalAnswerStep(responseText);
            
            return responseText;
        } catch (Exception e) {
            log.error("执行ReAct流程时发生错误", e);
            return "处理请求时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 构建包含历史对话记录的Prompt
     * @param systemPrompt 系统提示词
     * @param userInput 用户输入
     * @param agent Agent对象（可选）
     * @return 构建的Prompt
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent) {
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        
        // 添加系统消息
        messages.add(new SystemMessage(systemPrompt));
        
        // 如果提供了Agent，添加历史对话记录
        if (agent != null) {
            try {
                // 生成会话ID
                String sessionId = memoryService.generateSessionId(agent);
                
                // 获取历史记录长度配置，默认为10
                int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
                
                // 获取历史消息
                List<org.springframework.ai.chat.messages.Message> historyMessages = 
                    memoryService.getHistoryMessages(sessionId, historyLength);
                
                // 添加历史消息到Prompt
                messages.addAll(historyMessages);
                
                // 将当前用户消息添加到内存中，以便下次对话使用
                memoryService.addUserMessageToMemory(sessionId, userInput);
            } catch (Exception e) {
                log.warn("获取历史对话记录时发生错误: {}", e.getMessage());
            }
        }
        
        // 添加当前用户消息到Prompt
        messages.add(new UserMessage(userInput));
        
        return new Prompt(messages);
    }
    
    /**
     * 流式执行ReAct流程 - 使用真正的流式处理机制
     * 
     * 核心改进点：
     * 1. 使用stream()方法实现真正的流式处理
     * 2. 逐步返回结果给客户端，实现实时数据传输
     * 3. 完整的ReAct流程监控
     * 4. 增强的错误处理机制
     * 5. 异步处理提高性能
     * 6. 正确捕获和记录工具调用事件
     * 7. 支持对话历史记忆功能
     * 
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表
     * @param tokenConsumer token处理回调函数
     */
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer) {
        executeStreamWithAgent(chatClient, userInput, tools, tokenConsumer, null);
    }
    
    /**
     * 流式执行ReAct流程 - 使用真正的流式处理机制（支持Agent配置）
     * 
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表
     * @param tokenConsumer token处理回调函数
     * @param agent Agent对象（可选）
     */
    public void executeStreamWithAgent(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent) {
        log.info("使用stream()方法处理ReAct流程，支持真正的流式输出");
        
        // 重置步骤计数器
        stepCounter.set(0);
        
        // 使用StringBuilder累积完整响应
        StringBuilder fullResponse = new StringBuilder();
        
        try {
            // 触发思考步骤
            triggerThinkStep("开始处理用户请求: " + userInput);
            
            // 构建系统提示词
            String systemPrompt = "You are a helpful AI assistant that can use tools to answer questions. " +
                "Use the available tools when needed to gather information. " +
                "Think step by step and show your reasoning process. " +
                "When using tools, clearly indicate your thoughts, actions, and observations.";
            
            // 构建Prompt，包含历史对话记录
            Prompt prompt = buildPromptWithHistory(systemPrompt, userInput, agent);
            HisenseTripTool tripTool =  new HisenseTripTool(agentService,infoCollectorService);
            tripTool.initialize();

            // 订阅流式响应
            chatClient.prompt(prompt)
                    .tools(tripTool)
                    .stream()
                    .chatResponse()
                    .subscribe(
                        chatResponse -> {
                            try {
                                // 获取token
                                String token = chatResponse.getResult().getOutput().getText();
                                
                                // 验证token是否有效
                                if (isValidToken(token)) {
                                    // 累积完整响应
                                    fullResponse.append(token);
                                    
                                    // 分析token内容，识别工具调用和结果
                                    analyzeAndRecordToolEvents(token, fullResponse.toString());
                                    
                                    // 实时发送token给客户端
                                    if (tokenConsumer != null) {
                                        tokenConsumer.accept(token);
                                    }
                                    
                                    // 记录思考过程
                                    processTokenForSteps(token);
                                }
                            } catch (Exception e) {
                                log.error("处理token时发生错误", e);
                            }
                        },
                        throwable -> {
                            log.error("流式处理出错", throwable);
                            // 检查是否是401 Unauthorized错误
                            if (isUnauthorizedError(throwable)) {
                                log.error("LLM返回401未授权错误: {}", throwable.getMessage());
                                sendErrorToConsumer(tokenConsumer, " 请配置API密钥");
                            } else {
                                recordStreamError(throwable.getMessage());
                                sendErrorToConsumer(tokenConsumer, throwable.getMessage());
                            }
                        },
                        () -> {
                            log.info("流式处理完成");
                            // 触发最终答案步骤
                            triggerFinalAnswerStep(fullResponse.toString());
                            
                            // 将助理回复添加到ChatMemory
                            if (agent != null) {
                                try {
                                    String sessionId = memoryService.generateSessionId(agent);
                                    memoryService.addAssistantMessageToMemory(sessionId, fullResponse.toString());
                                } catch (Exception e) {
                                    log.warn("保存助理回复到内存时发生错误: {}", e.getMessage());
                                }
                            }
                            
                            // 发送完成事件，包含完整内容
                            sendCompletionEvent(tokenConsumer, fullResponse.toString());
                        }
                    );
            
        } catch (Exception e) {
            log.error("流式执行ReAct流程时发生错误", e);
            recordStreamError(e.getMessage());
            sendErrorToConsumer(tokenConsumer, e.getMessage());
        }
    }
    
    /**
     * 检查token是否有效
     * @param token token字符串
     * @return 是否有效
     */
    private boolean isValidToken(String token) {
        return token != null && !token.isEmpty();
    }
    
    /**
     * 处理token以识别不同的步骤类型
     * @param token token字符串
     */
    private void processTokenForSteps(String token) {
        if (token.contains("Thought:")) {
            triggerThinkStep(token);
        } else if (token.contains("Action:")) {
            // 提取工具名称和参数
            String toolName = extractToolName(token);
            Object toolArgs = extractToolArgs(token);
            triggerActionStep(toolName != null ? toolName : "unknown tool", token, toolArgs);
        } else if (token.contains("Observation:")) {
            triggerObservationStep(token);
        }
    }
    
    /**
     * 向消费者发送错误信息
     * @param tokenConsumer token消费者
     * @param errorMessage 错误信息
     */
    private void sendErrorToConsumer(Consumer<String> tokenConsumer, String errorMessage) {
        if (tokenConsumer != null) {
            tokenConsumer.accept("[ERROR] " + errorMessage);
        }
    }
    
    /**
     * 发送完成事件
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应
     */
    private void sendCompletionEvent(Consumer<String> tokenConsumer, String fullResponse) {
        if (tokenConsumer instanceof AgentChatService.TokenConsumerWithCompletion) {
            log.debug("调用onComplete，内容长度: {}", fullResponse.length());
            ((AgentChatService.TokenConsumerWithCompletion) tokenConsumer).onComplete(fullResponse);
        } else if (tokenConsumer != null) {
            log.warn("tokenConsumer不是TokenConsumerWithCompletion实例");
            tokenConsumer.accept("");
        }
    }
    
    /**
     * 分析token内容，识别工具调用和结果
     * 这个方法通过分析响应中的特殊标记来识别工具调用
     */
    private void analyzeAndRecordToolEvents(String token, String fullResponse) {
        if (!isValidToken(token) || workPanelCollector == null) {
            return;
        }
        
        try {
            // 检查工具调用的标记
            // 通常格式为: "Tool: [工具名称]" 或 "Calling [工具名称]" 或类似的模式
            if (isToolCall(token)) {
                String toolName = extractToolName(token);
                if (isValidToolName(toolName)) {
                    // 记录工具调用开始
                    workPanelCollector.recordToolCallStart(toolName, "execute", extractToolArgs(token));
                }
            }
            
            // 检查工具结果的标记
            else if (isToolResult(token)) {
                String toolName = extractToolName(fullResponse);
                String result = extractToolResult(token);
                if (isValidToolName(toolName)) {
                    // 记录工具调用完成
                    workPanelCollector.recordToolCallComplete(toolName, result, "success");
                }
            }
            
            // 检查错误标记
            else if (isError(token)) {
                String toolName = extractToolName(fullResponse);
                if (isValidToolName(toolName)) {
                    // 记录工具调用错误
                    workPanelCollector.recordToolCallError(toolName, token);
                }
            }
        } catch (Exception e) {
            log.debug("分析工具调用事件时发生错误: {}", e.getMessage());
        }
    }
    
    /**
     * 检查是否为工具调用
     * @param token token字符串
     * @return 是否为工具调用
     */
    private boolean isToolCall(String token) {
        return token.contains("Tool:") || token.contains("Calling") || 
               token.contains("tool:") || token.contains("calling");
    }
    
    /**
     * 检查是否为工具结果
     * @param token token字符串
     * @return 是否为工具结果
     */
    private boolean isToolResult(String token) {
        return token.contains("Result:") || token.contains("result:") || 
               token.contains("Output:") || token.contains("output:");
    }
    
    /**
     * 检查是否为错误信息
     * @param token token字符串
     * @return 是否为错误信息
     */
    private boolean isError(String token) {
        return token.contains("Error:") || token.contains("error:") || 
               token.contains("Exception:") || token.contains("exception:");
    }
    
    /**
     * 检查工具名称是否有效
     * @param toolName 工具名称
     * @return 是否有效
     */
    private boolean isValidToolName(String toolName) {
        return toolName != null && !toolName.isEmpty();
    }
    
    /**
     * 从token中提取工具名称
     */
    private String extractToolName(String text) {
        if (text == null) return null;
        
        // 尝试从常见的工具调用格式中提取工具名称
        String[] patterns = {
            "Tool: (\\w+)",
            "tool: (\\w+)",
            "Calling (\\w+)",
            "calling (\\w+)",
            "Use (\\w+)",
            "use (\\w+)",
            "Tool\\((\\w+)\\)",
            "tool\\((\\w+)\\)"
        };
        
        for (String pattern : patterns) {
            java.util.regex.Pattern p = java.util.regex.Pattern.compile(pattern);
            java.util.regex.Matcher m = p.matcher(text);
            if (m.find()) {
                return m.group(1);
            }
        }
        
        return null;
    }
    
    /**
     * 从token中提取工具参数
     */
    private Object extractToolArgs(String text) {
        if (text == null) return null;
        
        // 简单的参数提取逻辑
        // 可以根据具体的工具调用格式进行更复杂的解析
        java.util.Map<String, Object> args = new java.util.HashMap<>();
        
        // 尝试提取括号内的内容作为参数
        java.util.regex.Pattern pattern = java.util.regex.Pattern.compile("\\(([^)]*)\\)");
        java.util.regex.Matcher matcher = pattern.matcher(text);
        if (matcher.find()) {
            args.put("params", matcher.group(1));
            return args;
        }
        
        return args.isEmpty() ? null : args;
    }
    
    /**
     * 从token中提取工具结果
     */
    private String extractToolResult(String token) {
        if (token == null) return "";
        
        // 简单的结果提取逻辑
        // 提取冒号后的内容
        int colonIndex = token.lastIndexOf(':');
        if (colonIndex >= 0 && colonIndex < token.length() - 1) {
            return token.substring(colonIndex + 1).trim();
        }
        
        return token.trim();
    }
    
    /**
     * 记录流式处理错误到工作面板
     */
    private void recordStreamError(String errorMessage) {
        if (workPanelCollector != null) {
            try {
                workPanelCollector.recordLog("流式处理错误: " + errorMessage, "error");
            } catch (Exception e) {
                log.debug("记录流式处理错误到工作面板失败: {}", e.getMessage());
            }
        }
    }
    
    /**
     * 触发思考步骤
     * @param content 思考内容
     */
    private void triggerThinkStep(String content) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.THOUGHT, content);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发行动步骤
     * @param toolName 工具名称
     * @param toolAction 工具行动
     * @param toolArgs 工具参数
     */
    private void triggerActionStep(String toolName, String toolAction, Object toolArgs) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.ACTION, "执行工具: " + toolName);
        
        ReactStep.ToolCallAction toolActionObj = new ReactStep.ToolCallAction(toolName, toolArgs);
        reactStep.setAction(toolActionObj);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发观察步骤
     * @param observation 观察内容
     */
    private void triggerObservationStep(String observation) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.OBSERVATION, observation);
        
        ReactStep.ToolObservation toolObservation = new ReactStep.ToolObservation(observation);
        reactStep.setObservation(toolObservation);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发最终答案步骤
     * @param finalAnswer 最终答案
     */
    private void triggerFinalAnswerStep(String finalAnswer) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.FINAL_ANSWER, finalAnswer);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 通知所有回调
     * @param reactStep ReAct步骤
     */
    private void notifyCallbacks(ReactStep reactStep) {
        for (ReactCallback callback : reactCallbacks) {
            try {
                callback.onStep(reactStep);
            } catch (Exception e) {
                log.error("执行ReAct回调时发生错误", e);
            }
        }
    }
    
    /**
     * 判断异常是否为401未授权错误
     * @param e 异常对象
     * @return 是否为401错误
     */
    private boolean isUnauthorizedError(Throwable e) {
        if (e == null) {
            return false;
        }
        
        // 检查异常消息中是否包含401 Unauthorized
        String message = e.getMessage();
        if (message != null && (message.contains("401 Unauthorized") || message.contains("Unauthorized"))) {
            return true;
        }
        
        // 递归检查cause
        return isUnauthorizedError(e.getCause());
    }
}