package pangea.hiagent.exception;

import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import pangea.hiagent.dto.ApiResponse;

import jakarta.servlet.http.HttpServletRequest;
import java.util.stream.Collectors;

/**
 * 全局异常处理器
 * 统一处理系统中的各种异常
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {
    
    /**
     * 处理业务异常
     */
    @ExceptionHandler(BusinessException.class)
    public ResponseEntity<ApiResponse<Void>> handleBusinessException(BusinessException e, HttpServletRequest request) {
        log.warn("业务异常: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("BUSINESS_ERROR")
                .details(e.getMessage())
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(e.getCode(), e.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(response);
    }
    
    /**
     * 处理参数验证异常
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ResponseEntity<ApiResponse<Void>> handleMethodArgumentNotValidException(
            MethodArgumentNotValidException e, HttpServletRequest request) {
        log.warn("参数验证异常: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        String errorMessage = e.getBindingResult().getFieldErrors().stream()
                .map(FieldError::getDefaultMessage)
                .collect(Collectors.joining("; "));
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("VALIDATION_ERROR")
                .details(errorMessage)
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.PARAMETER_ERROR.getCode(), 
                ErrorCode.PARAMETER_ERROR.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(response);
    }
    
    /**
     * 处理绑定异常
     */
    @ExceptionHandler(BindException.class)
    public ResponseEntity<ApiResponse<Void>> handleBindException(BindException e, HttpServletRequest request) {
        log.warn("绑定异常: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        String errorMessage = e.getBindingResult().getFieldErrors().stream()
                .map(fieldError -> fieldError.getField() + ": " + fieldError.getDefaultMessage())
                .collect(Collectors.joining("; "));
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("BIND_ERROR")
                .details(errorMessage)
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.PARAMETER_ERROR.getCode(), 
                ErrorCode.PARAMETER_ERROR.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(response);
    }
    
    /**
     * 处理参数类型不匹配异常
     */
    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public ResponseEntity<ApiResponse<Void>> handleMethodArgumentTypeMismatchException(
            MethodArgumentTypeMismatchException e, HttpServletRequest request) {
        log.warn("参数类型不匹配异常: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("ARGUMENT_TYPE_MISMATCH")
                .details("参数 '" + e.getName() + "' 类型不匹配，期望类型: " + e.getRequiredType().getSimpleName())
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.PARAMETER_ERROR.getCode(), 
                ErrorCode.PARAMETER_ERROR.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(response);
    }
    
    /**
     * 处理HTTP消息不可读异常
     */
    @ExceptionHandler(HttpMessageNotReadableException.class)
    public ResponseEntity<ApiResponse<Void>> handleHttpMessageNotReadableException(
            HttpMessageNotReadableException e, HttpServletRequest request) {
        log.warn("HTTP消息不可读异常: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("MESSAGE_NOT_READABLE")
                .details("请求体格式不正确或缺失")
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.PARAMETER_ERROR.getCode(), 
                ErrorCode.PARAMETER_ERROR.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(response);
    }
    
    /**
     * 处理未授权异常
     */
    @ExceptionHandler(org.springframework.security.access.AccessDeniedException.class)
    public ResponseEntity<ApiResponse<Void>> handleAccessDeniedException(
            org.springframework.security.access.AccessDeniedException e, HttpServletRequest request) {
        log.warn("访问被拒绝: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("ACCESS_DENIED")
                .details("您没有权限执行此操作")
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.FORBIDDEN.getCode(), 
                ErrorCode.FORBIDDEN.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.FORBIDDEN).body(response);
    }
    
    /**
     * 处理认证异常
     */
    @ExceptionHandler(org.springframework.security.authentication.AuthenticationServiceException.class)
    public ResponseEntity<ApiResponse<Void>> handleAuthenticationServiceException(
            org.springframework.security.authentication.AuthenticationServiceException e, HttpServletRequest request) {
        log.warn("认证服务异常: {} - URL: {}", e.getMessage(), request.getRequestURL());
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("AUTHENTICATION_FAILED")
                .details("认证失败，请检查您的凭证")
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.UNAUTHORIZED.getCode(), 
                ErrorCode.UNAUTHORIZED.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.UNAUTHORIZED).body(response);
    }
    
    /**
     * 处理系统异常
     * 增强版本：更好地处理SSE流式响应中的异常
     */
    @ExceptionHandler(Exception.class)
    public ResponseEntity<ApiResponse<Void>> handleException(Exception e, HttpServletRequest request) {
        // 检查是否是SSE流式响应异常
        if (isStreamingResponseException(request, e)) {
            // SSE流式响应异常，不能返回JSON响应
            // 检查是否是预期的客户端断开连接（IOException）
            if (e instanceof java.io.IOException || 
                (e.getCause() instanceof java.io.IOException)) {
                // 客户端连接中断是正常情况，记录为DEBUG级别
                if (log.isDebugEnabled()) {
                    log.debug("SSE流式传输中客户端连接中断 - URL: {} - 异常类型: {}", 
                            request.getRequestURL(), 
                            e.getClass().getSimpleName());
                }
            } else if (e instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
                // 异步请求不可用 - 客户端已断开
                if (log.isDebugEnabled()) {
                    log.debug("SSE异步请求不可用，客户端已断开连接 - URL: {}", request.getRequestURL());
                }
            } else {
                // 非IOException的SSE异常才记录为ERROR
                log.error("SSE流式处理异常 - URL: {} - 异常类型: {} - 异常消息: {}", 
                        request.getRequestURL(), 
                        e.getClass().getSimpleName(), 
                        e.getMessage(), 
                        e);
            }
            
            // 不再抛出异常，直接返回空响应以避免二次异常处理
            // 响应已经进入SSE流模式，无法切换回JSON格式
            return ResponseEntity.ok().build();
        }
        
        log.error("系统异常 - URL: {} - 异常类型: {} - 异常消息: {}", 
                request.getRequestURL(), 
                e.getClass().getSimpleName(), 
                e.getMessage(), 
                e);
        
        ApiResponse.ErrorDetail errorDetail = ApiResponse.ErrorDetail.builder()
                .type("SYSTEM_ERROR")
                .details("系统内部错误，请联系管理员")
                .build();
        
        ApiResponse<Void> response = ApiResponse.error(ErrorCode.SYSTEM_ERROR.getCode(), 
                ErrorCode.SYSTEM_ERROR.getMessage(), errorDetail);
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(response);
    }
    
    /**
     * 判断是否为SSE流式响应异常
     * 增强版本：更全面的检测逻辑
     * @param request 请求对象
     * @param e 异常对象
     * @return 是否为SSE流式响应异常
     */
    private boolean isStreamingResponseException(HttpServletRequest request, Exception e) {
        // 检查Accept头
        String acceptHeader = request.getHeader("Accept");
        boolean isAcceptingStream = acceptHeader != null && acceptHeader.contains("text/event-stream");
        
        // 检查Content-Type（响应已设置为SSE格式）
        String contentType = request.getHeader("Content-Type");
        boolean isStreamContent = contentType != null && contentType.contains("text/event-stream");
        
        // 检查请求路径（通常SSE端点包含stream或chat关键字）
        String requestUri = request.getRequestURI();
        boolean isStreamPath = requestUri != null && (requestUri.contains("stream") || 
                                                      requestUri.contains("chat") && requestUri.contains("event"));
        
        // 检查异常链中是否包含SSE相关异常
        boolean hasSseException = checkForSseException(e);
        
        // 检查是否是SSE操作中的标准异常
        boolean isSseOperationException = e instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException ||
                                         e instanceof java.io.IOException && e.getMessage() != null && 
                                         (e.getMessage().contains("Socket") ||
                                          e.getMessage().contains("软件中止") ||
                                          e.getMessage().contains("ServletOutputStream") ||
                                          e.getMessage().contains("Pipe"));
        
        return isAcceptingStream || isStreamContent || isStreamPath || hasSseException || isSseOperationException;
    }
    
    /**
     * 检查异常链中是否包含SSE相关的异常
     * 增强版本：更全面的异常关键字识别
     * @param e 异常对象
     * @return 是否包含SSE异常
     */
    private boolean checkForSseException(Exception e) {
        // 检查异常类型
        if (e instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
            return true;
        }
        
        // 检查异常消息中的SSE相关关键字
        String message = e.getMessage();
        if (message != null) {
            if (message.contains("SseEmitter") || 
                message.contains("SSE") || 
                message.contains("event-stream") ||
                message.contains("ServletOutputStream") ||
                message.contains("Socket") ||
                message.contains("Pipe") ||
                message.contains("software") ||
                message.contains("软件中止") ||
                message.contains("断开") ||
                message.contains("AsyncRequestNotUsable")) {
                return true;
            }
        }
        
        // 递归检查cause（最多检查5层深度以避免无限递归）
        if (e.getCause() != null && e.getCause() instanceof Exception) {
            Exception cause = (Exception) e.getCause();
            if (cause instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
                return true;
            }
            String causeMsg = cause.getMessage();
            if (causeMsg != null && (causeMsg.contains("Socket") || causeMsg.contains("Pipe"))) {
                return true;
            }
            return checkForSseException(cause);
        }
        
        return false;
    }
}