package pangea.hiagent.aspect;

import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

import java.util.HashMap;
import java.util.Map;

/**
 * 工具执行日志记录切面类
 * 自动记录带有@Tool注解的方法执行信息，包括方案名称、输入参数、输出结果、运行时长等
 */
@Slf4j
@Aspect
@Component
public class ToolExecutionLoggerAspect {

    @Autowired
    private IWorkPanelDataCollector workPanelDataCollector;

    /**
     * 环绕通知，拦截所有带有@Tool注解的方法
     * @param joinPoint 连接点
     * @return 方法执行结果
     * @throws Throwable 异常
     */
    @Around("@annotation(tool)")
    public Object logToolExecution(ProceedingJoinPoint joinPoint, Tool tool) throws Throwable {
        // 获取方法签名
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        String methodName = signature.getName();
        String className = signature.getDeclaringType().getSimpleName();
        String fullMethodName = className + "." + methodName;
        
        // 使用完整的工具名称（类名）以确保开始和完成事件能够正确匹配
        String toolName = className;
        
        // 获取工具描述
        String toolDescription = tool.description();
        
        // 获取方法参数
        String[] paramNames = signature.getParameterNames();
        Object[] args = joinPoint.getArgs();
        
        // 构建输入参数映射
        Map<String, Object> inputParams = new HashMap<>();
        if (paramNames != null && args != null) {
            for (int i = 0; i < paramNames.length; i++) {
                if (i < args.length) {
                    inputParams.put(paramNames[i], args[i]);
                }
            }
        }
        
        // 记录开始时间
        long startTime = System.currentTimeMillis();
        
        // 记录工具调用开始
        if (workPanelDataCollector != null) {
            try {
                workPanelDataCollector.recordToolCallStart(toolName, methodName, inputParams);
            } catch (Exception e) {
                log.warn("记录工具调用开始时发生错误: {}", e.getMessage());
            }
        }
        
        log.debug("开始执行工具方法: {}，描述: {}，参数: {}", fullMethodName, toolDescription, inputParams);
        
        try {
            // 执行原方法
            Object result = joinPoint.proceed();
            
            // 记录结束时间
            long endTime = System.currentTimeMillis();
            long executionTime = endTime - startTime;
            
            // 记录工具调用完成
            if (workPanelDataCollector != null) {
                try {
                    workPanelDataCollector.recordToolCallComplete(toolName, result, "success", executionTime);
                } catch (Exception e) {
                    log.warn("记录工具调用完成时发生错误: {}", e.getMessage());
                }
            }
            
            log.debug("工具方法执行成功: {}，描述: {}，耗时: {}ms，结果类型: {}", fullMethodName, toolDescription, executionTime, 
                     result != null ? result.getClass().getSimpleName() : "null");
            
            return result;
        } catch (Exception e) {
            // 记录结束时间
            long endTime = System.currentTimeMillis();
            long executionTime = endTime - startTime;
            
            // 记录工具调用错误
            if (workPanelDataCollector != null) {
                try {
                    workPanelDataCollector.recordToolCallComplete(toolName, e.getMessage(), "error", executionTime);
                } catch (Exception ex) {
                    log.warn("记录工具调用错误时发生错误: {}", ex.getMessage());
                }
            }
            
            log.error("工具方法执行失败: {}，描述: {}，耗时: {}ms，错误类型: {}，错误信息: {}", 
                     fullMethodName, toolDescription, executionTime, e.getClass().getSimpleName(), e.getMessage(), e);
            
            throw e;
        }
    }
}