package pangea.hiagent.core;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.messages.AssistantMessage;
import org.springframework.ai.chat.messages.UserMessage;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.chat.model.StreamingChatModel;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.ai.chat.memory.ChatMemory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.dto.AgentRequest;
import pangea.hiagent.dto.WorkPanelEvent;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.AgentDialogue;
import pangea.hiagent.service.AgentService;
import pangea.hiagent.core.ReActService;
import pangea.hiagent.workpanel.SseEventManager;
import pangea.hiagent.memory.SmartHistorySummarizer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;

/**
 * Agent 对话服务
 * 负责处理与 Agent 的对话逻辑
 */
@Slf4j
@Service
public class AgentChatService {

    @Autowired
    private AgentService agentService;

    @Autowired
    private ReActService reActService;

    @Autowired
    private ChatMemory chatMemory;
    
    @Autowired
    private SmartHistorySummarizer smartHistorySummarizer;

    private static final String DEFAULT_SYSTEM_PROMPT = "你是一个智能助手";

    /**
     * 处理 ReAct Agent 请求
     */
    public String handleReActAgentRequest(Agent agent, AgentRequest request, String userId) {
        log.info("使用ReAct Agent处理请求");
        // 使用ReAct Agent处理请求，传递userId以支持记忆功能
        String responseContent = reActService.processRequest(agent, request.getUserMessage(), userId);
        
        // 保存对话记录并返回结果
        return responseContent;
    }

    /**
     * 处理普通 Agent 请求
     */
    public String handleNormalAgentRequest(Agent agent, AgentRequest request, String userId) {
        log.info("使用普通Agent处理请求");
        
        // 为每个用户-Agent组合创建唯一的会话ID
        String sessionId = userId + "_" + agent.getId();
        
        // 使用ChatMemory管理对话历史
        // 添加用户消息到ChatMemory
        UserMessage userMessage = new UserMessage(request.getUserMessage());
        chatMemory.add(sessionId, Collections.singletonList(userMessage));
        
        // 获取系统提示词
        String systemPrompt = agent.getPromptTemplate() != null ? agent.getPromptTemplate() : DEFAULT_SYSTEM_PROMPT;
        
        // 构建Prompt，使用Agent配置的历史记录长度
        int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
        Prompt prompt = buildPrompt(systemPrompt, historyLength, sessionId);
        
        log.info("构建提示词完成");
        
        // 根据Agent配置获取对应的ChatModel
        ChatModel chatModel = agentService.getChatModelForAgent(agent);
        
        log.info("获取ChatModel成功");
        
        // 使用对应模型进行调用
        log.info("开始调用模型");
        try {
            org.springframework.ai.chat.model.ChatResponse chatResponse = chatModel.call(prompt);
            log.info("模型调用完成");
            
            // 提取助理回复
            String responseContent = chatResponse.getResult().getOutput().getText();
            
            log.info("模型调用成功，响应内容长度: {}", responseContent.length());
            
            // 将助理回复添加到ChatMemory
            AssistantMessage assistantMessage = new AssistantMessage(responseContent);
            chatMemory.add(sessionId, Collections.singletonList(assistantMessage));
            
            return responseContent;
        } catch (Exception e) {
            // 检查是否是401 Unauthorized错误
            if (isUnauthorizedError(e)) {
                log.error("LLM返回401未授权错误: {}", e.getMessage());
                throw new RuntimeException(" 请配置API密钥");
            } else {
                log.error("模型调用失败", e);
                throw new RuntimeException("模型调用失败: " + e.getMessage());
            }
        }
    }

    /**
     * 处理 ReAct Agent 流式请求
     */
    public void processReActAgentStream(AgentRequest request, Agent agent, String userId, 
                                  Consumer<String> tokenConsumer) {
        log.info("使用ReAct Agent处理流式请求");
        // 使用ReAct Agent流式处理请求，传递userId以支持记忆功能
        reActService.processRequestStream(agent, request.getUserMessage(), tokenConsumer, userId);
    }

    /**
     * 处理普通 Agent 流式请求
     */
    public void processNormalAgentStream(AgentRequest request, Agent agent, String userId,
                                       Consumer<String> tokenConsumer) {
        try {
            log.info("使用普通Agent处理流式请求");
            
            // 为每个用户-Agent组合创建唯一的会话ID
            String sessionId = userId + "_" + agent.getId();
            
            // 准备对话上下文
            prepareChatContext(request, sessionId);
            
            // 获取系统提示词
            String systemPrompt = agent.getPromptTemplate() != null ? agent.getPromptTemplate() : DEFAULT_SYSTEM_PROMPT;
            
            // 构建Prompt，使用Agent配置的历史记录长度
            int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
            Prompt prompt = buildPrompt(systemPrompt, historyLength, sessionId);
            
            // 获取流式模型
            StreamingChatModel streamingChatModel = getStreamingChatModel(agent);
            if (streamingChatModel == null) {
                log.warn("当前模型不支持流式输出");
                return;
            }
            
            // 流式处理
            handleStreamingResponse(tokenConsumer, prompt, streamingChatModel, sessionId);
        } catch (Exception e) {
            log.error("普通Agent流式处理失败", e);
        }
    }

    /**
     * 准备对话上下文
     */
    private void prepareChatContext(AgentRequest request, String sessionId) {
        UserMessage userMessage = new UserMessage(request.getUserMessage());
        chatMemory.add(sessionId, Collections.singletonList(userMessage));
    }

    /**
     * 构建Prompt
     */
    private Prompt buildPrompt(String systemPrompt, int historyLength, String sessionId) {
        List<org.springframework.ai.chat.messages.Message> historyMessages = chatMemory.get(sessionId, historyLength);
        
        // 使用智能历史摘要器优化历史消息
        List<org.springframework.ai.chat.messages.Message> summarizedMessages = 
            smartHistorySummarizer.summarize(historyMessages, historyLength);
        
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        messages.add(new org.springframework.ai.chat.messages.SystemMessage(systemPrompt));
        messages.addAll(summarizedMessages);
        return new Prompt(messages);
    }

    /**
     * 获取流式模型
     */
    private StreamingChatModel getStreamingChatModel(Agent agent) {
        try {
            ChatModel chatModel = agentService.getChatModelForAgent(agent);
            if (!(chatModel instanceof StreamingChatModel)) {
                log.warn("模型不支持流式输出: {}", chatModel.getClass().getName());
                return null;
            }
            return (StreamingChatModel) chatModel;
        } catch (Exception e) {
            log.error("获取流式模型失败", e);
            return null;
        }
    }

    /**
     * 处理流式响应
     */
    private void handleStreamingResponse(Consumer<String> tokenConsumer, Prompt prompt, 
                                       StreamingChatModel streamingChatModel, String sessionId) {
        StringBuilder fullText = new StringBuilder();
        
        streamingChatModel.stream(prompt)
            .subscribe(chatResponse -> {
                try {
                    String token = chatResponse.getResult().getOutput().getText();
                    if (token != null && !token.isEmpty()) {
                        fullText.append(token);
                        tokenConsumer.accept(token);
                    }
                } catch (Exception e) {
                    log.error("处理token时发生错误", e);
                }
            }, throwable -> {
                log.error("流式调用出错", throwable);
                // 检查是否是401 Unauthorized错误
                if (isUnauthorizedError(throwable)) {
                    log.error("LLM返回401未授权错误: {}", throwable.getMessage());
                    if (tokenConsumer != null) {
                        tokenConsumer.accept(" 请配置API密钥");
                    }
                }
            }, () -> {
                try {
                    // 添加助理回复到ChatMemory
                    AssistantMessage assistantMessage = new AssistantMessage(fullText.toString());
                    chatMemory.add(sessionId, Collections.singletonList(assistantMessage));
                    
                    // 发送完成事件，包含完整内容
                    if (tokenConsumer instanceof TokenConsumerWithCompletion) {
                        ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullText.toString());
                    }
                    
                    log.info("流式处理完成，总字符数: {}", fullText.length());
                } catch (Exception e) {
                    log.error("保存对话记录失败", e);
                }
            });
    }
    
    /**
     * 判断异常是否为401未授权错误
     * @param e 异常对象
     * @return 是否为401错误
     */
    private boolean isUnauthorizedError(Throwable e) {
        if (e == null) {
            return false;
        }
        
        // 检查异常消息中是否包含401 Unauthorized
        String message = e.getMessage();
        if (message != null && (message.contains("401 Unauthorized") || message.contains("Unauthorized"))) {
            return true;
        }
        
        // 递归检查cause
        return isUnauthorizedError(e.getCause());
    }

    /**
     * 创建token事件数据
     */
    public java.util.Map<String, Object> createTokenEventData(String token, String fullText) {
        java.util.Map<String, Object> data = new java.util.HashMap<>();
        data.put("token", token);
        data.put("fullText", fullText);
        data.put("isDone", false);
        return data;
    }
    
    /**
     * 创建优化的token事件数据（用于提高传输效率、支持打字机效果）
     * 
     * @param token 当前接收到的token
     * @return 优化后的数据
     */
    public java.util.Map<String, Object> createOptimizedTokenEventData(String token) {
        java.util.Map<String, Object> data = new java.util.HashMap<>();
        
        // 验证token是否有效
        if (token == null || token.isEmpty()) {
            return data;
        }
        
        // 添加token内容
        data.put("token", token);
        
        // 为前端打字机效果添加元数据
        data.put("tokenLength", token.length());
        
        // 只在DEBUG级别记录日志，避免生产环境性能下降
        if (log.isDebugEnabled()) {
            if (token.length() > 50) {
                log.debug("token事件已发送: length={}", token.length());
            } else {
                log.debug("token事件已发送: content={}", token);
            }
        }
        
        // 不再每次都发送完整的fullText，减少数据传输量
        // 只在完成事件时发送完整内容
        return data;
    }
    
    /**
     * 创建批量token事件数据
     * 优化版本：根据内容类型动态调整批量大小，提高传输效率
     * 
     * @param tokens 批量token字符串
     * @return 批量处理后的数据
     */
    public java.util.Map<String, Object> createBatchTokenEventData(String tokens) {
        java.util.Map<String, Object> data = new java.util.HashMap<>();
        
        // 验证tokens是否有效
        if (tokens == null || tokens.isEmpty()) {
            return data;
        }
        
        // 添加批量token内容
        data.put("token", tokens);
        
        // 为前端打字机效果添加元数据
        data.put("tokenLength", tokens.length());
        data.put("isBatch", true); // 标记这是批量数据
        
        // 只在DEBUG级别记录日志
        if (log.isDebugEnabled()) {
            log.debug("批量token事件已发送: length={}", tokens.length());
        }
        
        return data;
    }

    /**
     * 保存对话记录
     */
    public void saveDialogue(Agent agent, AgentRequest request, String userId, String responseContent) {
        try {
            // 创建对话记录
            AgentDialogue dialogue = AgentDialogue.builder()
                .agentId(request.getAgentId())
                .userMessage(request.getUserMessage())
                .agentResponse(responseContent)
                .userId(userId)
                .build();
            
            // 保存对话记录
            agentService.saveDialogue(dialogue);
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            throw new RuntimeException("保存对话记录失败", e);
        }
    }
    
    /**
     * 保存对话记录并发送完成事件
     */
    public void saveDialogueAndSendCompleteEvent(Agent agent, AgentRequest request, String userId, 
            String responseContent, org.springframework.web.servlet.mvc.method.annotation.SseEmitter emitter, 
            java.util.concurrent.atomic.AtomicBoolean isCompleted, AgentService agentService, SseEventManager sseEventManager) throws java.io.IOException {
        try {
            // 添加调试日志
            if (responseContent == null || responseContent.isEmpty()) {
                log.warn("saveDialogueAndSendCompleteEvent接收到空的responseContent!");
            } else {
                log.info("saveDialogueAndSendCompleteEvent接收到的responseContent长度: {}, 内容预览: {}", 
                    responseContent.length() > 100 ? responseContent.substring(0, 100) + "..." : responseContent);
            }
            
            // 创建并保存对话记录
            AgentDialogue dialogue = AgentDialogue.builder()
                .agentId(request.getAgentId())
                .userMessage(request.getUserMessage())
                .agentResponse(responseContent)
                .userId(userId)
                .build();
            agentService.saveDialogue(dialogue);
            
            // 发送完成事件
            java.util.Map<String, Object> data = new java.util.HashMap<>();
            data.put("fullText", responseContent);
            data.put("dialogueId", dialogue.getId());
            data.put("isDone", true);
            
            // 发送完成事件到前端
            sseEventManager.sendEvent(emitter, "complete", data, isCompleted);
        } catch (Exception e) {
            log.error("保存对话记录或发送完成事件失败", e);
            // 检查是否是客户端断开连接导致的异常
            if (e instanceof java.io.IOException || (e.getCause() instanceof java.io.IOException)) {
                log.debug("客户端连接已断开，忽略异常");
                isCompleted.set(true);
            }
            // 检查是否是响应已提交导致的异常
            else if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                log.debug("响应已提交，忽略异常");
                isCompleted.set(true);
            }
            // 其他异常情况下尝试发送错误信息
            else {
                log.warn("发送错误信息失败", e);
            }
        }
    }
    
    /**
     * 处理 ReAct Agent 流式请求（带 SSE）
     */
    public void processReActAgentStreamWithSse(AgentRequest request, Agent agent, String userId,
            SseEmitter emitter, SseEventManager sseEventManager, ReActService reActService,
            SaveDialogueAndSendCompleteEventFunction saveDialogueAndSendCompleteEvent) {
        // 如果传入的reActService为null，则使用自动注入的实例
        if (reActService == null) {
            reActService = this.reActService;
        }
        
        // 用于累积token的缓冲区
        StringBuilder tokenBuffer = new StringBuilder();
        // 批量大小阈值 - 适合中文流式输出，提高传输效率
        int BATCH_SIZE = 50; // 增加批量大小到50个字符，减少SSE事件频率
        // 时间间隔阈值（毫秒）- 保证响应及时性
        long FLUSH_INTERVAL = 200; // 200ms刷新间隔，平衡效率与响应速度
        // 上次刷新时间
        java.util.concurrent.atomic.AtomicLong lastFlushTime = new java.util.concurrent.atomic.AtomicLong(System.currentTimeMillis());
        // 是否已完成标记
        java.util.concurrent.atomic.AtomicBoolean isCompleted = new java.util.concurrent.atomic.AtomicBoolean(false);
        // 完整内容 - 使用对象包装以支持线程安全的最终赋值
        final java.util.concurrent.atomic.AtomicReference<String> fullTextRef = new java.util.concurrent.atomic.AtomicReference<>("");
        
        try {
            log.info("使用ReAct Agent处理流式请求，批量大小: {}, 刷新间隔: {}ms", BATCH_SIZE, FLUSH_INTERVAL);
            
            // 启动心跳保活机制
            sseEventManager.startHeartbeat(emitter, new java.util.concurrent.atomic.AtomicBoolean(false));
            
            // 订阅工作面板事件，并通过SSE推送到前端
            reActService.setWorkPanelEventSubscriber(event -> {
                try {
                    if (isCompleted.get()) {
                        if (log.isTraceEnabled()) {
                            log.trace("连接已完成，跳过推送工作面板事件");
                        }
                        return;
                    }
                    // 将工作面板事件转换为SSE事件推送
                    sseEventManager.sendWorkPanelEvent(emitter, (WorkPanelEvent) event, isCompleted);
                } catch (Exception e) {
                    if (log.isDebugEnabled()) {
                        log.debug("推送工作面板事件失败: {}", e.getMessage());
                    }
                }
            });
            
            // 使用ReAct Agent流式处理请求
            processReActAgentStream(request, agent, userId, new TokenConsumerWithCompletion() {
                @Override
                public void accept(String token) {
                    try {
                        if (isCompleted.get()) {
                            if (log.isTraceEnabled()) {
                                log.trace("连接已完成，跳过处理token");
                            }
                            return;
                        }
                        
                        // 验证token是否有效
                        if (token == null || token.isEmpty()) {
                            log.debug("ReAct Agent接收到空的token，跳过处理");
                            return;
                        }
                        
                        // 将token添加到完整内容中（原子引用安全更新）
                        String currentFullText = fullTextRef.get();
                        fullTextRef.set(currentFullText + token);
                        
                        // 将token添加到缓冲区
                        tokenBuffer.append(token);
                        
                        // 检查是否需要刷新缓冲区
                        // 优化逻辑：根据内容类型和长度动态调整刷新策略
                        long currentTime = System.currentTimeMillis();
                        boolean shouldFlush = false;
                        
                        // 如果缓冲区达到批量大小阈值，立即刷新
                        if (tokenBuffer.length() >= BATCH_SIZE) {
                            shouldFlush = true;
                        }
                        // 如果距离上次刷新时间超过间隔阈值，也需要刷新
                        else if ((currentTime - lastFlushTime.get()) >= FLUSH_INTERVAL) {
                            shouldFlush = true;
                        }
                        // 对于较短的内容，即使未达到阈值也适时刷新以提高响应速度
                        else if (tokenBuffer.length() > 0 && fullTextRef.get().length() < 100 && (currentTime - lastFlushTime.get()) >= FLUSH_INTERVAL / 2) {
                            shouldFlush = true;
                        }
                        
                        if (shouldFlush && tokenBuffer.length() > 0) {
                            // 发送批量token事件
                            String batchTokens = tokenBuffer.toString();
                            java.util.Map<String, Object> data = createBatchTokenEventData(batchTokens);
                            try {
                                sseEventManager.sendEvent(emitter, "token", data, isCompleted);
                                // 清空缓冲区
                                tokenBuffer.setLength(0);
                                // 更新上次刷新时间
                                lastFlushTime.set(currentTime);
                                
                                // 添加性能监控日志
                                if (log.isDebugEnabled()) {
                                    log.debug("发送批量token事件，大小: {}字符, 总内容长度: {}字符", 
                                        batchTokens.length(), fullTextRef.get().length());
                                }
                            } catch (Exception e) {
                                if (isCompleted.get()) {
                                    if (log.isTraceEnabled()) {
                                        log.trace("ReAct Agent连接已完成，无法发送SSE token事件: {}", e.getMessage());
                                    }
                                } else {
                                    log.debug("ReAct Agent发送SSE token事件失败: {}", e.getMessage());
                                }
                            }
                        }
                    } catch (Exception e) {
                        log.error("ReAct Agent处理token时发生错误", e);
                        if (!isCompleted.getAndSet(true)) {
                            try {
                                sseEventManager.sendError(emitter, "处理响应时发生错误: " + e.getMessage());
                            } catch (Exception ignored) {
                                log.debug("无法发送错误信息");
                            }
                        }
                    }
                }
                
                @Override
                public void onComplete(String fullContent) {
                    // ReAct处理完成时的回调 - 防止onComplete重复执行
                    if (!isCompleted.getAndSet(true)) {
                        log.info("ReAct Agent处理完成，总字符数: {}", fullContent != null ? fullContent.length() : 0);
                        
                        // 刷新剩余的缓冲区内容
                        if (tokenBuffer.length() > 0) {
                            try {
                                String remainingTokens = tokenBuffer.toString();
                                java.util.Map<String, Object> data = createBatchTokenEventData(remainingTokens);
                                sseEventManager.sendEvent(emitter, "token", data, isCompleted);
                                tokenBuffer.setLength(0);
                                if (log.isDebugEnabled()) {
                                    log.debug("刷新剩余缓冲区内容，大小: {}字符", remainingTokens.length());
                                }
                            } catch (Exception e) {
                                if (log.isDebugEnabled()) {
                                    log.debug("ReAct Agent发送剩余SSE token事件失败: {}", e.getMessage());
                                }
                            }
                        }
                        
                        // 保存完整内容到原子引用
                        fullTextRef.set(fullContent != null ? fullContent : "");
                        
                        // 保存对话记录并发送完成事件到前端
                        try {
                            saveDialogueAndSendCompleteEvent.execute(agent, request, userId, fullTextRef.get(), emitter, isCompleted, sseEventManager);
                        } catch (Exception e) {
                            log.error("保存对话记录或发送完成事件失败: {}", e.getMessage());
                            try {
                                sseEventManager.sendError(emitter, "保存对话记录失败: " + e.getMessage());
                            } catch (Exception ignored) {
                                if (log.isDebugEnabled()) {
                                    log.debug("无法发送错误信息");
                                }
                            }
                        }
                    }
                }
            });
            
            // 不需要额外的等待和重复检查，onComplete回调会处理所有完成逻辑
            // 这里只需要等待足够的时间让异步的onComplete回调执行完成
            try {
                // 通过轮询检查是否已完成，最多等待5秒
                long maxWaitTime = 10*60*1000;
                long startTime = System.currentTimeMillis();
                while (!isCompleted.get() && (System.currentTimeMillis() - startTime) < maxWaitTime) {
                    Thread.sleep(100); // 每100ms检查一次
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            
            // 如果在超时后仍未完成，发送超时错误（仅作为最后手段）
            if (!isCompleted.get()) {
                log.warn("ReAct Agent流式处理超时，isCompleted未被设置为true");
                try {
                    sseEventManager.sendError(emitter, "处理超时：未能及时完成处理");
                } catch (Exception ignored) {
                    if (log.isDebugEnabled()) {
                        log.debug("无法发送超时错误信息");
                    }
                }
            }
            
            // 关闭连接
            sseEventManager.completeEmitter(emitter, isCompleted);
        } catch (Exception e) {
            log.error("ReAct Agent流式处理失败: {}", e.getMessage(), e);
            // 检查是否是客户端断开连接导致的异常
            if (e instanceof java.io.IOException || (e.getCause() instanceof java.io.IOException)) {
                if (log.isDebugEnabled()) {
                    log.debug("客户端连接已断开");
                }
            } else {
                // 尝试发送错误信息
                if (!isCompleted.getAndSet(true)) {
                    try {
                        sseEventManager.sendError(emitter, "处理请求时发生错误: " + e.getMessage());
                    } catch (Exception sendError) {
                        if (log.isDebugEnabled()) {
                            log.debug("无法发送错误信息", sendError);
                        }
                    }
                }
            }
            sseEventManager.completeEmitter(emitter, isCompleted);
        }
    }
    
    /**
     * 处理普通 Agent 流式请求（带 SSE）
     */
    public void processNormalAgentStreamWithSse(AgentRequest request, Agent agent, String userId,
            SseEmitter emitter, SseEventManager sseEventManager, 
            CreateTokenEventDataFunction createTokenEventData, 
            SaveDialogueAndSendCompleteEventFunction saveDialogueAndSendCompleteEvent) {
        // 是否已完成标记
        java.util.concurrent.atomic.AtomicBoolean isCompleted = new java.util.concurrent.atomic.AtomicBoolean(false);
        
        try {
            log.info("使用普通Agent处理流式请求");
            
            // 启动心跳保活机制
            sseEventManager.startHeartbeat(emitter, new java.util.concurrent.atomic.AtomicBoolean(false));
            
            // 用于累积token的缓冲区
            StringBuilder tokenBuffer = new StringBuilder();
            // 批量大小阈值 - 适合中文流式输出，提高传输效率
            int BATCH_SIZE = 50; // 增加批量大小到50个字符，减少SSE事件频率
            // 时间间隔阈值（毫秒）- 保证响应及时性
            long FLUSH_INTERVAL = 200; // 200ms刷新间隔，平衡效率与响应速度
            // 上次刷新时间
            java.util.concurrent.atomic.AtomicLong lastFlushTime = new java.util.concurrent.atomic.AtomicLong(System.currentTimeMillis());
            // 完整内容 - 使用对象包装以支持线程安全的最终赋值
            final java.util.concurrent.atomic.AtomicReference<String> fullTextRef = new java.util.concurrent.atomic.AtomicReference<>("");
            
            // 使用普通Agent流式处理请求
            processNormalAgentStream(request, agent, userId, new TokenConsumerWithCompletion() {
                @Override
                public void accept(String token) {
                    try {
                        if (isCompleted.get()) {
                            if (log.isTraceEnabled()) {
                                log.trace("连接已完成，跳过处理token");
                            }
                            return;
                        }
                        
                        // 验证token是否有效
                        if (token == null || token.isEmpty()) {
                            return;
                        }
                        
                        // 将token添加到完整内容中
                        String currentFullText = fullTextRef.get();
                        fullTextRef.set(currentFullText + token);
                        
                        // 将token添加到缓冲区
                        tokenBuffer.append(token);
                        
                        // 检查是否需要刷新缓冲区
                        long currentTime = System.currentTimeMillis();
                        boolean shouldFlush = false;
                        
                        // 如果缓冲区达到批量大小阈值，立即刷新
                        if (tokenBuffer.length() >= BATCH_SIZE) {
                            shouldFlush = true;
                        }
                        // 如果距离上次刷新时间超过间隔阈值，也需要刷新
                        else if ((currentTime - lastFlushTime.get()) >= FLUSH_INTERVAL) {
                            shouldFlush = true;
                        }
                        
                        if (shouldFlush && tokenBuffer.length() > 0) {
                            // 发送批量token事件
                            String batchTokens = tokenBuffer.toString();
                            java.util.Map<String, Object> data = createBatchTokenEventData(batchTokens);
                            try {
                                sseEventManager.sendEvent(emitter, "token", data, isCompleted);
                                // 清空缓冲区
                                tokenBuffer.setLength(0);
                                // 更新上次刷新时间
                                lastFlushTime.set(currentTime);
                                
                                if (log.isDebugEnabled()) {
                                    log.debug("发送批量token事件，大小: {}字符, 总内容长度: {}字符", 
                                        batchTokens.length(), fullTextRef.get().length());
                                }
                            } catch (Exception e) {
                                if (isCompleted.get()) {
                                    if (log.isTraceEnabled()) {
                                        log.trace("普通 Agent连接已完成，无法发送token事件: {}", e.getMessage());
                                    }
                                } else {
                                    if (log.isDebugEnabled()) {
                                        log.debug("普通 Agent发送token事件失败: {}", e.getMessage());
                                    }
                                }
                            }
                        }
                    } catch (Exception e) {
                        log.error("普通 Agent处理token时发生错误", e);
                        if (!isCompleted.getAndSet(true)) {
                            try {
                                sseEventManager.sendError(emitter, "处理响应时发生错误: " + e.getMessage());
                            } catch (Exception ignored) {
                                if (log.isDebugEnabled()) {
                                    log.debug("无法发送错误信息");
                                }
                            }
                        }
                    }
                }
                
                @Override
                public void onComplete(String fullContent) {
                    // 普通Agent处理完成时的回调
                    if (isCompleted.get()) {
                        return; // 已完成，防止重复处理
                    }
                    
                    log.info("普通Agent处理完成，总字符数: {}", fullContent != null ? fullContent.length() : 0);
                    
                    // 刷新剩余的缓冲区内容
                    if (tokenBuffer.length() > 0) {
                        try {
                            String remainingTokens = tokenBuffer.toString();
                            java.util.Map<String, Object> data = createBatchTokenEventData(remainingTokens);
                            sseEventManager.sendEvent(emitter, "token", data, isCompleted);
                            tokenBuffer.setLength(0);
                            if (log.isDebugEnabled()) {
                                log.debug("刷新剩余缓冲区内容，大小: {}字符", remainingTokens.length());
                            }
                        } catch (Exception e) {
                            if (log.isDebugEnabled()) {
                                log.debug("普通Agent发送剩余token事件失败: {}", e.getMessage());
                            }
                        }
                    }
                    
                    // 保存完整内容到原子引用
                    fullTextRef.set(fullContent != null ? fullContent : "");
                    
                    // 发送完成事件（只在这里发送一次）
                    try {
                        // 一定要向前端发送完整的fullText
                        saveDialogueAndSendCompleteEvent.execute(agent, request, userId, fullTextRef.get(), emitter, isCompleted, sseEventManager);
                    } catch (Exception e) {
                        log.error("保存对话记录或发送完成事件失败: {}", e.getMessage());
                        if (!isCompleted.getAndSet(true)) {
                            try {
                                sseEventManager.sendError(emitter, "保存对话记录失败: " + e.getMessage());
                            } catch (Exception ignored) {
                                if (log.isDebugEnabled()) {
                                    log.debug("无法发送错误信息");
                                }
                            }
                        }
                    }
                }
            });
        } catch (Exception e) {
            log.error("普通Agent流式处理失败", e);
            // 发送错误信息给客户端
            if (!isCompleted.getAndSet(true)) {
                try {
                    sseEventManager.sendError(emitter, "处理请求时发生错误: " + e.getMessage());
                } catch (Exception sendError) {
                    if (log.isDebugEnabled()) {
                        log.debug("发送错误信息失败", sendError);
                    }
                }
            }
        }
    }
    
    /**
     * 创建token事件数据函数式接口
     */
    @FunctionalInterface
    public interface CreateTokenEventDataFunction {
        java.util.Map<String, Object> execute(String token, String fullText);
    }
    
    /**
     * 保存对话记录并发送完成事件函数式接口
     */
    @FunctionalInterface
    public interface SaveDialogueAndSendCompleteEventFunction {
        void execute(Agent agent, AgentRequest request, String userId, 
                String responseContent, SseEmitter emitter, java.util.concurrent.atomic.AtomicBoolean isCompleted, SseEventManager sseEventManager) throws java.io.IOException;
    }
    
    /**
     * Token消费者接口，支持完成回调
     */
    public interface TokenConsumerWithCompletion extends java.util.function.Consumer<String> {
        /**
         * 当流式处理完成时调用
         * @param fullContent 完整的内容
         */
        default void onComplete(String fullContent) {
            // 默认实现为空
        }
    }
}