package pangea.hiagent.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Paths;

/**
 * 网页代理控制器
 * 用于解决iframe加载网页时的CORS问题，并处理X-Frame-Options等安全限制
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/proxy")
public class WebProxyController {

    /**
     * 代理请求指定URL的内容
     * 
     * @param url 要代理的URL地址
     * @param token JWT token（可选）
     * @return 网页HTML内容
     */
    @GetMapping(produces = MediaType.TEXT_HTML_VALUE)
    public ResponseEntity<String> proxyUrl(@RequestParam String url, @RequestParam(required = false) String token) {
        try {
            log.debug("代理请求URL: {}, token: {}", url, token != null ? "provided" : "null");
            
            // 验证URL格式
            if (url == null || url.trim().isEmpty()) {
                log.warn("无效的URL参数: 为空");
                return ResponseEntity.badRequest()
                    .contentType(MediaType.TEXT_HTML)
                    .body("<html><body><h2>Error: URL is empty</h2><p>Please provide a valid URL to proxy.</p></body></html>");
            }
            
            // 检查是否为file协议
            if (url.toLowerCase().startsWith("file://")) {
                // 处理本地文件
                return handleLocalFile(url);
            }
            
            // 处理相对路径URL
            if (!url.toLowerCase().startsWith("http://") && !url.toLowerCase().startsWith("https://")) {
                // 如果是相对路径，尝试补全为绝对路径
                if (url.startsWith("/")) {
                    // 假设是本地文件路径
                    url = "file://" + url;
                } else if (!url.toLowerCase().startsWith("file://")) {
                    log.warn("无效的URL scheme: {}", url);
                    return ResponseEntity.badRequest()
                        .contentType(MediaType.TEXT_HTML)
                        .body("<html><body><h2>Error: Invalid URL scheme</h2><p>URL must start with http://, https:// or file://</p></body></html>");
                }
            }
            
            // 获取网页内容
            String content = fetchWebContent(url);
            log.debug("成功代理URL内容，长度: {} bytes", content.length());
            
            // 处理内容以提高兼容性
            String processedContent = processContentForEmbedding(content, url);
            
            // 添加调试信息
            log.debug("处理后内容长度: {} bytes", processedContent.length());
            
            return ResponseEntity.ok()
                .contentType(MediaType.TEXT_HTML)
                .body(processedContent);
            
        } catch (IOException e) {
            log.error("代理请求失败: {}", url, e);
            // 提供更详细的错误信息
            String errorMessage = e.getMessage();
            if (errorMessage == null || errorMessage.isEmpty()) {
                errorMessage = "Unknown network error";
            }
            return ResponseEntity.status(502)
                .contentType(MediaType.TEXT_HTML)
                .body("<html><body><h2>Error: Network error</h2><p>Failed to fetch content from " + url + "</p><p>" + errorMessage + "</p><p>Type: " + e.getClass().getSimpleName() + "</p></body></html>");
        } catch (Exception e) {
            log.error("代理请求异常: {}", url, e);
            // 提供更详细的错误信息
            String errorMessage = e.getMessage();
            if (errorMessage == null || errorMessage.isEmpty()) {
                errorMessage = "Unknown error";
            }
            return ResponseEntity.status(500)
                .contentType(MediaType.TEXT_HTML)
                .body("<html><body><h2>Error: Internal server error</h2><p>An unexpected error occurred while fetching content from " + url + "</p><p>" + errorMessage + "</p><p>Type: " + e.getClass().getSimpleName() + "</p></body></html>");
        }
    }
    
    /**
     * 处理本地文件请求
     * 
     * @param fileUrl 文件URL
     * @return 文件内容响应
     */
    private ResponseEntity<String> handleLocalFile(String fileUrl) {
        try {
            log.debug("处理本地文件请求: {}", fileUrl);
            
            // 将file:// URL转换为本地路径
            String filePath = fileUrl.substring("file://".length());
            // 处理Windows路径
            if (filePath.startsWith("/")) {
                // 去掉开头的斜杠（Windows file:// URL格式）
                filePath = filePath.substring(1);
            }
            
            // 规范化路径
            File file = new File(filePath);
            String canonicalPath = file.getCanonicalPath();
            
            // 安全检查：确保文件在允许的目录内
            if (!isFileAccessAllowed(canonicalPath)) {
                log.warn("拒绝访问本地文件，超出允许范围: {}", canonicalPath);
                return ResponseEntity.status(403)
                    .contentType(MediaType.TEXT_HTML)
                    .body("<html><body><h2>Error: Access denied</h2><p>Access to this file is not allowed.</p></body></html>");
            }
            
            // 检查文件是否存在
            if (!file.exists()) {
                log.warn("本地文件不存在: {}", canonicalPath);
                return ResponseEntity.status(404)
                    .contentType(MediaType.TEXT_HTML)
                    .body("<html><body><h2>Error: File not found</h2><p>The requested file was not found.</p></body></html>");
            }
            
            // 检查是否为文件（而非目录）
            if (!file.isFile()) {
                log.warn("本地路径不是文件: {}", canonicalPath);
                return ResponseEntity.status(400)
                    .contentType(MediaType.TEXT_HTML)
                    .body("<html><body><h2>Error: Not a file</h2><p>The requested path is not a file.</p></body></html>");
            }
            
            // 读取文件内容
            String content = readFileContent(file);
            log.debug("成功读取本地文件内容，长度: {} bytes", content.length());
            
            // 处理内容以提高兼容性
            String processedContent = processContentForEmbedding(content, fileUrl);
            
            return ResponseEntity.ok()
                .contentType(MediaType.TEXT_HTML)
                .body(processedContent);
                
        } catch (Exception e) {
            log.error("处理本地文件异常: {}", fileUrl, e);
            return ResponseEntity.status(500)
                .contentType(MediaType.TEXT_HTML)
                .body("<html><body><h2>Error: Internal server error</h2><p>An unexpected error occurred while reading the local file.</p><p>" + e.getMessage() + "</p></body></html>");
        }
    }
    
    /**
     * 检查文件访问是否被允许
     * 
     * @param filePath 文件路径
     * @return 是否允许访问
     */
    private boolean isFileAccessAllowed(String filePath) {
        try {
            // 获取项目根目录
            String projectRoot = Paths.get("").toAbsolutePath().toString();
            String canonicalProjectRoot = new File(projectRoot).getCanonicalPath();
            
            // 检查文件是否在项目目录内
            return filePath.startsWith(canonicalProjectRoot);
        } catch (Exception e) {
            log.error("检查文件访问权限时出错: ", e);
            return false;
        }
    }
    
    /**
     * 读取文件内容
     * 
     * @param file 文件对象
     * @return 文件内容
     * @throws IOException 读取异常
     */
    private String readFileContent(File file) throws IOException {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        
        try {
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8"));
            
            String line;
            while ((line = reader.readLine()) != null) {
                content.append(line).append("\n");
            }
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    log.debug("关闭读取器失败: {}", e.getMessage());
                }
            }
        }
        
        return content.toString();
    }
    
    /**
     * 获取网页内容
     */
    private String fetchWebContent(String urlStr) throws IOException {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        
        try {
            URL url = new URL(urlStr);
            URLConnection connection = url.openConnection();
            
            // 设置请求头
            connection.setRequestProperty("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36");
            connection.setRequestProperty("Accept", "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8");
            connection.setRequestProperty("Accept-Language", "zh-CN,zh;q=0.9,en;q=0.8");
            connection.setRequestProperty("Accept-Encoding", "gzip, deflate, br");
            connection.setRequestProperty("Connection", "keep-alive");
            connection.setRequestProperty("Upgrade-Insecure-Requests", "1");
            connection.setConnectTimeout(15000); // 增加超时时间到15秒
            connection.setReadTimeout(15000);
            
            // 获取响应头信息用于诊断
            java.net.HttpURLConnection httpConn = (java.net.HttpURLConnection) connection;
            int responseCode = httpConn.getResponseCode();
            String xFrameOptions = httpConn.getHeaderField("X-Frame-Options");
            String csp = httpConn.getHeaderField("Content-Security-Policy");
            String contentType = httpConn.getHeaderField("Content-Type");
            
            log.debug("网页请求 - URL: {}, 响应码: {}, Content-Type: {}", urlStr, responseCode, contentType);
            if (xFrameOptions != null) {
                log.warn("检测到X-Frame-Options头部 - URL: {}, X-Frame-Options: {}", urlStr, xFrameOptions);
            }
            if (csp != null) {
                log.debug("检测到CSP头部 - URL: {}, CSP: {}", urlStr, csp);
            }
            
            // 检查Content-Type是否为HTML
            if (contentType != null && !contentType.toLowerCase().contains("text/html")) {
                log.warn("非HTML内容类型 - URL: {}, Content-Type: {}", urlStr, contentType);
                // 仍然继续处理，但记录警告
            }
            
            reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"));
            
            String line;
            while ((line = reader.readLine()) != null) {
                content.append(line).append("\n");
            }
            
        } catch (IOException e) {
            log.error("获取网页内容异常 - URL: {}, 错误堆栈: ", urlStr, e);
            throw e;
        } catch (Exception e) {
            log.error("获取网页内容未知异常 - URL: {}, 错误堆栈: ", urlStr, e);
            throw new IOException("获取网页内容时发生未知错误: " + e.getMessage(), e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    log.debug("关闭读取器失败: {}", e.getMessage());
                }
            }
        }
        
        return content.toString();
    }
    
    /**
     * 处理内容以提高在iframe中的兼容性
     * 移除或修改可能阻止嵌入的头部和脚本
     */
    private String processContentForEmbedding(String content, String originalUrl) {
        try {
            log.debug("开始处理内容以提高嵌入兼容性，原始长度: {}", content.length());
            
            // 移除X-Frame-Options meta标签
            String processed = content.replaceAll("(?i)<meta[^>]*http-equiv\\s*=\\s*[\"']?X-Frame-Options[\"']?[^>]*/?>", "");
            
            // 移除Content-Security-Policy meta标签中可能阻止嵌入的指令
            processed = processed.replaceAll("(?i)<meta[^>]*http-equiv\\s*=\\s*[\"']?Content-Security-Policy[\"']?[^>]*content\\s*=\\s*[\"'][^\"']*frame-ancestors[^\"']*[\"'][^>]*/?>", "");
            
            // 移除可能阻止嵌入的JavaScript代码
            processed = processed.replaceAll("(?i)<script[^>]*>[^<]*top\\s*!=\\s*self[^<]*</script>", "");
            processed = processed.replaceAll("(?i)<script[^>]*>[^<]*window\\.top\\s*!=\\s*window[^<]*</script>", "");
            processed = processed.replaceAll("(?i)<script[^>]*>[^<]*document\\.location\\.ancestorOrigins[^<]*</script>", "");
            
            // 添加基础标签以确保相对链接正确
            if (!processed.contains("<base")) {
                String baseUrl = originalUrl.substring(0, originalUrl.lastIndexOf('/') + 1);
                processed = processed.replaceFirst("(?i)(<head[^>]*>)", "$1<base href=\"" + baseUrl + "\">");
            }
            
            // 添加样式以改善嵌入体验
            String styleAddition = "<style>" +
                "body { margin: 0; padding: 10px; font-family: Arial, sans-serif; }" +
                "img { max-width: 100%; height: auto; }" +
                "iframe { max-width: 100%; }" +
                "</style>";
                
            processed = processed.replaceFirst("(?i)(</head>)", styleAddition + "$1");
            
            log.debug("内容处理完成，处理后长度: {}", processed.length());
            return processed;
        } catch (Exception e) {
            log.error("处理内容以提高嵌入兼容性时出错: ", e);
            // 如果处理失败，返回原始内容
            return content;
        }
    }
}