// 对象池服务
export class ObjectPoolService<T> {
  private pool: T[] = [];
  private factory: () => T;
  private resetter?: (obj: T) => void;
  private maxSize: number;

  constructor(factory: () => T, resetter?: (obj: T) => void, maxSize: number = 100) {
    this.factory = factory;
    this.resetter = resetter;
    this.maxSize = maxSize;
  }

  /**
   * 从对象池获取对象
   * @returns 对象实例
   */
  acquire(): T {
    if (this.pool.length > 0) {
      return this.pool.pop()!;
    }
    return this.factory();
  }

  /**
   * 将对象归还到对象池
   * @param obj 对象实例
   */
  release(obj: T): void {
    if (this.resetter) {
      this.resetter(obj);
    }
    
    if (this.pool.length < this.maxSize) {
      this.pool.push(obj);
    }
  }

  /**
   * 清空对象池
   */
  clear(): void {
    this.pool = [];
  }

  /**
   * 获取对象池当前大小
   * @returns 对象池大小
   */
  size(): number {
    return this.pool.length;
  }
}