-- HiAgent数据库表结构初始化脚本
-- 用户表
CREATE TABLE IF NOT EXISTS sys_user (
  id varchar(36) NOT NULL,
  username varchar(50) NOT NULL UNIQUE,
  password varchar(255) NOT NULL,
  email varchar(100),
  nickname varchar(100),
  status varchar(20) DEFAULT 'active',
  role varchar(50) DEFAULT 'user',
  avatar varchar(255),
  last_login_time bigint,
  api_key varchar(255),
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_username ON sys_user (username);

-- OAuth2 提供者配置表
CREATE TABLE IF NOT EXISTS oauth2_provider (
  id varchar(36) NOT NULL,
  provider_name varchar(50) NOT NULL UNIQUE,
  display_name varchar(100),
  description text,
  auth_type varchar(50) NOT NULL DEFAULT 'authorization_code',
  authorize_url varchar(255),
  token_url varchar(255),
  userinfo_url varchar(255),
  client_id varchar(255),
  client_secret varchar(255),
  redirect_uri varchar(255),
  scope varchar(255),
  enabled tinyint DEFAULT 1,
  config_json json,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id),
  UNIQUE (provider_name)
);
CREATE INDEX IF NOT EXISTS idx_oauth2_enabled ON oauth2_provider (enabled);
CREATE INDEX IF NOT EXISTS idx_oauth2_auth_type ON oauth2_provider (auth_type);

-- OAuth2 账户关联表
CREATE TABLE IF NOT EXISTS oauth2_account (
  id varchar(36) NOT NULL,
  user_id varchar(36) NOT NULL,
  provider_name varchar(50) NOT NULL,
  remote_user_id varchar(255) NOT NULL,
  remote_username varchar(100),
  remote_email varchar(100),
  access_token varchar(1000),
  refresh_token varchar(1000),
  token_expiry timestamp,
  scope varchar(255),
  profile_data json,
  linked_at timestamp DEFAULT CURRENT_TIMESTAMP,
  last_login_at timestamp,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  deleted int DEFAULT 0,
  PRIMARY KEY (id),
  UNIQUE (provider_name, remote_user_id),
  UNIQUE (user_id, provider_name),
  FOREIGN KEY (user_id) REFERENCES sys_user(id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS idx_oauth2_user_id ON oauth2_account (user_id);
CREATE INDEX IF NOT EXISTS idx_oauth2_provider_name ON oauth2_account (provider_name);
CREATE INDEX IF NOT EXISTS idx_oauth2_remote_user_id ON oauth2_account (remote_user_id);

-- 登录模式配置表
CREATE TABLE IF NOT EXISTS login_mode_config (
  id varchar(36) NOT NULL,
  mode_name varchar(50) NOT NULL UNIQUE,
  mode_type varchar(50) NOT NULL,
  display_name varchar(100),
  description text,
  enabled tinyint DEFAULT 1,
  config_json json,
  display_order int DEFAULT 0,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  PRIMARY KEY (id),
  UNIQUE (mode_name)
);
CREATE INDEX IF NOT EXISTS idx_login_mode_enabled ON login_mode_config (enabled);

-- LLM配置表
CREATE TABLE IF NOT EXISTS llm_config (
  id varchar(36) NOT NULL,
  name varchar(100) NOT NULL UNIQUE,
  description text,
  provider varchar(50),
  model_name varchar(100),
  api_key varchar(255),
  base_url varchar(255),
  temperature decimal(3,2) DEFAULT 0.7,
  max_tokens int DEFAULT 4096,
  top_p decimal(3,2) DEFAULT 0.9,
  enabled tinyint DEFAULT 1,
  owner varchar(36),
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_enabled ON llm_config (enabled);
CREATE INDEX IF NOT EXISTS idx_provider ON llm_config (provider);
CREATE INDEX IF NOT EXISTS idx_name ON llm_config (name);

-- Agent表
CREATE TABLE IF NOT EXISTS agent (
  id varchar(36) NOT NULL,
  name varchar(100) NOT NULL,
  description text,
  status varchar(20) DEFAULT 'active',
  default_model varchar(100),
  system_prompt text,
  prompt_template text,
  temperature decimal(3,2) DEFAULT 0.7,
  max_tokens int DEFAULT 4096,
  top_p decimal(3,2) DEFAULT 0.9,
  top_k int DEFAULT 50,
  presence_penalty decimal(3,2) DEFAULT 0,
  frequency_penalty decimal(3,2) DEFAULT 0,
  history_length int DEFAULT 10,
  tools json,
  rag_collection_id varchar(36),
  rag_top_k int DEFAULT 5,
  rag_score_threshold decimal(3,2) DEFAULT 0.8,
  rag_prompt_template text,
  enable_rag tinyint DEFAULT 0,
  enable_re_act tinyint DEFAULT 0,
  enable_streaming tinyint DEFAULT 1,
  owner varchar(36),
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id),
  FOREIGN KEY (default_model) REFERENCES llm_config(name)
);
CREATE INDEX IF NOT EXISTS idx_owner ON agent (owner);
CREATE INDEX IF NOT EXISTS idx_status ON agent (status);
CREATE INDEX IF NOT EXISTS idx_created_at ON agent (created_at);

-- 文档表
CREATE TABLE IF NOT EXISTS document (
  id varchar(36) NOT NULL,
  name varchar(255) NOT NULL,
  type varchar(20),
  size bigint,
  status varchar(20) DEFAULT 'uploading',
  chunks int DEFAULT 0,
  collection_id varchar(36),
  file_path varchar(255),
  author varchar(100),
  source varchar(100),
  tags json,
  metadata json,
  embedding_model varchar(100),
  error_message text,
  owner varchar(36),
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);

-- 文档片段表
CREATE TABLE IF NOT EXISTS document_chunk (
  id varchar(36) NOT NULL,
  document_id varchar(36) NOT NULL,
  content longtext,
  page_number int,
  score decimal(3,2),
  sequence int,
  vector_id bigint,
  metadata json,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_document_id ON document_chunk (document_id);
CREATE INDEX IF NOT EXISTS idx_vector_id ON document_chunk (vector_id);

-- Agent对话表
CREATE TABLE IF NOT EXISTS agent_dialogue (
  id varchar(36) NOT NULL,
  agent_id varchar(36) NOT NULL,
  context_id varchar(36),
  user_message longtext,
  agent_response longtext,
  prompt_tokens int,
  completion_tokens int,
  total_tokens int,
  processing_time bigint,
  finish_reason varchar(50),
  tool_calls json,
  user_id varchar(36),
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_agent_id ON agent_dialogue (agent_id);
CREATE INDEX IF NOT EXISTS idx_user_id ON agent_dialogue (user_id);
CREATE INDEX IF NOT EXISTS idx_context_id ON agent_dialogue (context_id);
CREATE INDEX IF NOT EXISTS idx_created_at ON agent_dialogue (created_at);

-- 系统日志表
CREATE TABLE IF NOT EXISTS sys_log (
  id varchar(36) NOT NULL,
  operation_type varchar(50),
  resource_type varchar(50),
  resource_id varchar(36),
  user_id varchar(36),
  description text,
  request_params json,
  response_result json,
  success tinyint DEFAULT 1,
  error_message text,
  ip_address varchar(50),
  execution_time bigint,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_user_id ON sys_log (user_id);
CREATE INDEX IF NOT EXISTS idx_created_at ON sys_log (created_at);
CREATE INDEX IF NOT EXISTS idx_operation_type ON sys_log (operation_type);
CREATE INDEX IF NOT EXISTS idx_resource_type ON sys_log (resource_type);
CREATE INDEX IF NOT EXISTS idx_success ON sys_log (success);

-- 工具表
CREATE TABLE IF NOT EXISTS tool (
  id varchar(36) NOT NULL,
  name varchar(100) NOT NULL,
  display_name varchar(100),
  description text,
  category varchar(50),
  status varchar(20) DEFAULT 'active',
  parameters json,
  return_type varchar(50),
  return_schema json,
  implementation text,
  timeout bigint,
  api_endpoint varchar(255),
  http_method varchar(20),
  headers json,
  auth_type varchar(50),
  auth_config json,
  owner varchar(36),
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);

-- 定时器配置表
CREATE TABLE IF NOT EXISTS hiagent_timer_config (
  id varchar(36) NOT NULL,
  name varchar(100) NOT NULL,
  description text,
  cron_expression varchar(50) NOT NULL,
  enabled int DEFAULT 0,
  agent_id varchar(36),
  agent_name varchar(100),
  prompt_template text,
  params_json json,
  last_execution_time timestamp,
  next_execution_time timestamp,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark varchar(255),
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_timer_enabled ON hiagent_timer_config (enabled);
CREATE INDEX IF NOT EXISTS idx_timer_created_by ON hiagent_timer_config (created_by);
CREATE INDEX IF NOT EXISTS idx_timer_agent_id ON hiagent_timer_config (agent_id);

-- 定时器执行历史表
CREATE TABLE IF NOT EXISTS hiagent_timer_execution_history (
  id bigint NOT NULL AUTO_INCREMENT,
  timer_id varchar(36) NOT NULL,
  timer_name varchar(100) NOT NULL,
  execution_time timestamp DEFAULT CURRENT_TIMESTAMP,
  success int DEFAULT 0,
  result text,
  error_message text,
  actual_prompt text,
  execution_duration bigint,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_timer_history_timer_id ON hiagent_timer_execution_history (timer_id);
CREATE INDEX IF NOT EXISTS idx_timer_history_execution_time ON hiagent_timer_execution_history (execution_time);
CREATE INDEX IF NOT EXISTS idx_timer_history_success ON hiagent_timer_execution_history (success);
CREATE INDEX IF NOT EXISTS idx_timer_history_created_at ON hiagent_timer_execution_history (created_at);

-- 提示词模板表
CREATE TABLE IF NOT EXISTS hiagent_prompt_template (
  id varchar(36) NOT NULL,
  name varchar(100) NOT NULL,
  description text,
  template_content text NOT NULL,
  param_schema json,
  template_type varchar(50) DEFAULT 'system',
  is_system int DEFAULT 0,
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id)
);
CREATE INDEX IF NOT EXISTS idx_prompt_template_type ON hiagent_prompt_template (template_type);
CREATE INDEX IF NOT EXISTS idx_prompt_template_is_system ON hiagent_prompt_template (is_system);
CREATE INDEX IF NOT EXISTS idx_prompt_template_created_by ON hiagent_prompt_template (created_by);

-- 工具配置表
CREATE TABLE IF NOT EXISTS tool_configs (
  id varchar(36) NOT NULL,
  tool_name varchar(100) NOT NULL,
  param_name varchar(100) NOT NULL,
  param_value text,
  description text,
  default_value text,
  type varchar(50) NOT NULL DEFAULT 'string',
  required tinyint DEFAULT 0,
  group_name varchar(100) DEFAULT 'default',
  created_at timestamp DEFAULT CURRENT_TIMESTAMP,
  updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by varchar(36),
  updated_by varchar(36),
  deleted int DEFAULT 0,
  remark text,
  PRIMARY KEY (id),
  UNIQUE (tool_name, param_name)
);
CREATE INDEX IF NOT EXISTS idx_tool_configs_tool_name ON tool_configs (tool_name);
CREATE INDEX IF NOT EXISTS idx_tool_configs_group_name ON tool_configs (group_name);
CREATE INDEX IF NOT EXISTS idx_tool_configs_type ON tool_configs (type);