package pangea.hiagent.tool.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

/**
 * OAuth2.0授权工具类
 * 支持标准OAuth2.0认证流程，包括密码凭证流、令牌刷新和受保护资源访问
 */
@Component
public class OAuth2AuthorizationTool {

    private static final Logger logger = LoggerFactory.getLogger(OAuth2AuthorizationTool.class);

    /**
     * 使用密码凭证流获取访问令牌
     *
     * @param tokenUrl      令牌端点URL
     * @param clientId      客户端ID
     * @param clientSecret  客户端密钥
     * @param username      用户名
     * @param password      密码
     * @param scope         请求的作用域
     * @return 认证结果或错误信息
     */
    public String authorizeWithPasswordCredentials(String tokenUrl, String clientId, String clientSecret, 
                                                  String username, String password, String scope) {
        try {
            logger.info("开始OAuth2.0密码凭证授权流程");
            logger.debug("令牌URL: {}, 客户端ID: {}, 用户名: {}, 作用域: {}", tokenUrl, clientId, username, scope);

            // 构造请求参数
            StringBuilder requestBody = new StringBuilder();
            requestBody.append("grant_type=password&");
            requestBody.append("username=").append(username).append("&");
            requestBody.append("password=").append(password).append("&");
            requestBody.append("scope=").append(scope);

            // 发送POST请求
            String response = sendPostRequest(tokenUrl, clientId, clientSecret, requestBody.toString());

            logger.info("OAuth2.0密码凭证授权完成");
            return response;
        } catch (Exception e) {
            logger.error("OAuth2.0密码凭证授权失败", e);
            return "错误: OAuth2.0密码凭证授权失败 - " + e.getMessage();
        }
    }

    /**
     * 刷新访问令牌
     *
     * @param tokenUrl      令牌端点URL
     * @param clientId      客户端ID
     * @param clientSecret  客户端密钥
     * @param refreshToken  刷新令牌
     * @return 刷新结果或错误信息
     */
    public String refreshToken(String tokenUrl, String clientId, String clientSecret, String refreshToken) {
        try {
            logger.info("开始刷新OAuth2.0访问令牌");
            logger.debug("令牌URL: {}, 客户端ID: {}, 刷新令牌: {}", tokenUrl, clientId, refreshToken);

            // 构造请求参数
            StringBuilder requestBody = new StringBuilder();
            requestBody.append("grant_type=refresh_token&");
            requestBody.append("refresh_token=").append(refreshToken);

            // 发送POST请求
            String response = sendPostRequest(tokenUrl, clientId, clientSecret, requestBody.toString());

            logger.info("OAuth2.0访问令牌刷新完成");
            return response;
        } catch (Exception e) {
            logger.error("刷新OAuth2.0访问令牌失败", e);
            return "错误: 刷新OAuth2.0访问令牌失败 - " + e.getMessage();
        }
    }

    /**
     * 访问受保护资源
     *
     * @param resourceUrl   受保护资源URL
     * @param tokenUrl      令牌端点URL（用于获取新令牌）
     * @param clientId      客户端ID
     * @return 资源内容或错误信息
     */
    public String accessProtectedResource(String resourceUrl, String tokenUrl, String clientId) {
        try {
            logger.info("开始访问受保护资源");
            logger.debug("资源URL: {}, 令牌URL: {}, 客户端ID: {}", resourceUrl, tokenUrl, clientId);

            // 注意：这是一个的实现，实际应用中需要管理令牌的存储和使用
            // 这里只是演示如何访问受保护资源
            return "错误: 未找到有效的访问令牌。请先通过authorizeWithPasswordCredentials方法获取令牌。";
        } catch (Exception e) {
            logger.error("访问受保护资源失败", e);
            return "错误: 访问受保护资源失败 - " + e.getMessage();
        }
    }

    /**
     * 发送POST请求到OAuth2.0令牌端点
     *
     * @param tokenUrl    令牌端点URL
     * @param clientId    客户端ID
     * @param clientSecret 客户端密钥
     * @param requestBody 请求体内容
     * @return 服务器响应
     * @throws Exception 网络或IO异常
     */
    private String sendPostRequest(String tokenUrl, String clientId, String clientSecret, String requestBody) throws Exception {
        URL url = new URL(tokenUrl);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();

        // 设置请求头
        String authString = clientId + ":" + clientSecret;
        String encodedAuth = Base64.getEncoder().encodeToString(authString.getBytes(StandardCharsets.UTF_8));
        
        connection.setRequestMethod("POST");
        connection.setRequestProperty("Authorization", "Basic " + encodedAuth);
        connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
        connection.setRequestProperty("Accept", "application/json");
        connection.setDoOutput(true);

        // 发送请求体
        try (OutputStream os = connection.getOutputStream()) {
            byte[] input = requestBody.getBytes(StandardCharsets.UTF_8);
            os.write(input, 0, input.length);
        }

        // 读取响应
        int responseCode = connection.getResponseCode();
        StringBuilder response = new StringBuilder();
        
        try (BufferedReader br = new BufferedReader(
                new InputStreamReader(responseCode >= 200 && responseCode < 300 ? 
                                     connection.getInputStream() : connection.getErrorStream(), 
                                     StandardCharsets.UTF_8))) {
            String responseLine;
            while ((responseLine = br.readLine()) != null) {
                response.append(responseLine.trim());
            }
        }

        logger.debug("HTTP响应码: {}, 响应内容: {}", responseCode, response.toString());
        
        if (responseCode >= 200 && responseCode < 300) {
            return response.toString();
        } else {
            return "错误: HTTP " + responseCode + " - " + response.toString();
        }
    }
}