package pangea.hiagent.core;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Agent工具管理服务类
 * 负责管理Agent可用的工具列表
 */
@Slf4j
@Service
public class AgentToolManager {
    
    @Autowired
    private pangea.hiagent.service.ToolService toolService;
    
    @Autowired
    private ApplicationContext applicationContext;
    
    /**
     * 获取Agent可用的工具列表
     * @param agent Agent对象
     * @return 工具列表
     */
    public List<Tool> getAvailableTools(Agent agent) {
        try {
            log.info("获取Agent可用工具列表，Agent ID: {}, 名称: {}", agent.getId(), agent.getName());
            
            // 获取Agent所有者的所有活跃工具
            List<Tool> allTools = toolService.getUserToolsByStatus(agent.getOwner(), "active");
            log.info("用户所有活跃工具数量: {}", allTools != null ? allTools.size() : 0);
            
            if (allTools == null || allTools.isEmpty()) {
                log.warn("Agent: {} 没有配置可用的工具", agent.getId());
                return List.of();
            }
            
            // 如果Agent配置了特定的工具列表，则只返回配置的工具
            List<String> toolNames = agent.getToolNames();
            log.info("Agent配置的工具名称列表: {}", toolNames);
            
            if (toolNames != null && !toolNames.isEmpty()) {
                // 根据工具名称筛选工具
                List<Tool> filteredTools = filterToolsByName(allTools, toolNames);
                log.info("筛选后的工具数量: {}", filteredTools.size());
                return filteredTools;
            }
            
            return allTools;
        } catch (Exception e) {
            log.error("获取Agent可用工具时发生错误", e);
            return List.of();
        }
    }
    
    /**
     * 根据工具名称筛选工具
     * @param allTools 所有工具
     * @param toolNames 工具名称列表
     * @return 筛选后的工具列表
     */
    public List<Tool> filterToolsByName(List<Tool> allTools, List<String> toolNames) {
        return allTools.stream()
            .filter(tool -> toolNames.contains(tool.getName()))
            .collect(Collectors.toList());
    }
    
    /**
     * 根据工具名称集合筛选工具实例（用于ReActService）
     * @param allTools 所有工具实例
     * @param toolNames 工具名称集合
     * @return 筛选后的工具实例列表
     */
    public List<Object> filterToolsByInstances(List<Object> allTools, Set<String> toolNames) {
        log.debug("开始筛选工具实例，工具名称集合: {}", toolNames);
        
        if (toolNames == null || toolNames.isEmpty()) {
            log.debug("工具名称集合为空，返回所有工具实例");
            return allTools;
        }
        
        List<Object> filteredTools = allTools.stream()
            .filter(tool -> {
                // 获取工具类名（不含包名）
                String className = tool.getClass().getSimpleName();
                log.debug("检查工具类: {}", className);
                
                // 检查类名是否匹配
                boolean isMatch = toolNames.contains(className) ||
                    toolNames.stream().anyMatch(name ->
                        className.toLowerCase().contains(name.toLowerCase()));
                
                if (isMatch) {
                    log.debug("工具 {} 匹配成功", className);
                }
                
                return isMatch;
            })
            .collect(Collectors.toList());
            
        log.debug("筛选完成，返回 {} 个工具实例", filteredTools.size());
        return filteredTools;
    }
    
    /**
     * 构建工具描述文本
     * @param tools 工具列表
     * @return 工具描述文本
     */
    public String buildToolsDescription(List<Tool> tools) {
        if (tools.isEmpty()) {
            return "（暂无可用工具）";
        }
        
        StringBuilder description = new StringBuilder();
        for (int i = 0; i < tools.size(); i++) {
            Tool tool = tools.get(i);
            description.append(i + 1).append(". ");
            description.append(tool.getName());
            if (hasValue(tool.getDisplayName())) {
                description.append(" - ").append(tool.getDisplayName());
            }
            if (hasValue(tool.getDescription())) {
                description.append(" - ").append(tool.getDescription());
            }
            description.append("\n");
        }
        
        return description.toString();
    }
    
    /**
     * 检查字符串是否有值
     * @param value 字符串值
     * @return 是否有值
     */
    private boolean hasValue(String value) {
        return value != null && !value.isEmpty();
    }
    
    /**
     * 根据Agent获取可用的工具实例
     * @param agent Agent对象
     * @return 工具实例列表
     */
    public List<Object> getAvailableToolInstances(Agent agent) {
        // 获取Agent可用的工具定义
        List<Tool> availableTools = getAvailableTools(agent);
        
        // 获取所有Spring管理的bean名称
        String[] beanNames = applicationContext.getBeanDefinitionNames();
        
        // 根据工具名称筛选对应的工具实例
        List<Object> toolInstances = new ArrayList<>();
        Set<String> availableToolNames = availableTools.stream()
            .map(Tool::getName)
            .collect(Collectors.toSet());
            
        for (String beanName : beanNames) {
            Object bean = applicationContext.getBean(beanName);
            String simpleClassName = bean.getClass().getSimpleName();
            
            // 检查bean的类名是否与可用工具名称匹配
            if (availableToolNames.contains(simpleClassName)) {
                toolInstances.add(bean);
            }
        }
        
        log.info("智能体{}获取到的工具实例数量: {}", agent.getName(), toolInstances.size());

        return toolInstances;
    }
}