package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * Hisense模型适配器
 * 实现与Hisense模型服务的对接，完全遵循OpenAI API规范
 */
@Slf4j
@Component
public class HisenseModelAdapter extends AbstractOpenAiCompatibleAdapter {

    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            throw new IllegalArgumentException("无效的Hisense配置");
        }

        // 获取配置信息，直接使用配置中的值
        String baseUrl = config.getBaseUrl();
        String modelName = config.getModelName();

        // 确保baseUrl格式正确，移除可能存在的/chat/completions路径
        if (baseUrl != null && baseUrl.endsWith("/chat/completions")) {
            baseUrl = baseUrl.substring(0, baseUrl.length() - "/chat/completions".length());
        }
        
        // 移除可能存在的/v1后缀，因为OpenAiApi会自动添加/v1/chat/completions
        if (baseUrl != null && baseUrl.endsWith("/v1")) {
            baseUrl = baseUrl.substring(0, baseUrl.length() - "/v1".length());
        }

        log.info("创建Hisense ChatModel, baseUrl: {}, modelName: {}", baseUrl, modelName);

        return createOpenAiChatModel(config, baseUrl);
    }

    @Override
    public String getProviderName() {
        return "hisense";
    }

    @Override
    public boolean validateConfig(LlmConfig config) {
        // Hisense适配器至少需要启用配置
        // API密钥和基础URL可以使用默认值
        return validateOpenAiConfig(config, false); // Hisense不需要强制API密钥
    }
}