package pangea.hiagent.common.utils;

import lombok.extern.slf4j.Slf4j;

/**
 * 参数验证工具类
 * 提供常用的参数验证方法，减少重复代码
 */
@Slf4j
public class ValidationUtils {
    
    /**
     * 检查对象是否为null
     * 
     * @param obj 待检查的对象
     * @param paramName 参数名称
     * @return 如果对象为null返回true，否则返回false
     */
    public static boolean isNull(Object obj, String paramName) {
        if (obj == null) {
            log.warn("{}参数不能为空", paramName);
            return true;
        }
        return false;
    }
    
    /**
     * 检查字符串是否为空
     * 
     * @param str 待检查的字符串
     * @param paramName 参数名称
     * @return 如果字符串为空返回true，否则返回false
     */
    public static boolean isEmpty(String str, String paramName) {
        if (str == null || str.isEmpty()) {
            log.warn("{}参数不能为空", paramName);
            return true;
        }
        return false;
    }
    
    /**
     * 检查字符串是否为空白
     * 
     * @param str 待检查的字符串
     * @param paramName 参数名称
     * @return 如果字符串为空白返回true，否则返回false
     */
    public static boolean isBlank(String str, String paramName) {
        if (str == null || str.trim().isEmpty()) {
            log.warn("{}参数不能为空", paramName);
            return true;
        }
        return false;
    }
    
    /**
     * 检查集合是否为空
     * 
     * @param collection 待检查的集合
     * @param paramName 参数名称
     * @return 如果集合为空返回true，否则返回false
     */
    public static boolean isEmpty(java.util.Collection<?> collection, String paramName) {
        if (collection == null || collection.isEmpty()) {
            log.warn("{}参数不能为空", paramName);
            return true;
        }
        return false;
    }
    
    /**
     * 检查数组是否为空
     * 
     * @param array 待检查的数组
     * @param paramName 参数名称
     * @return 如果数组为空返回true，否则返回false
     */
    public static boolean isEmpty(Object[] array, String paramName) {
        if (array == null || array.length == 0) {
            log.warn("{}参数不能为空", paramName);
            return true;
        }
        return false;
    }
    
    /**
     * 检查数字是否小于等于0
     * 
     * @param number 待检查的数字
     * @param paramName 参数名称
     * @return 如果数字小于等于0返回true，否则返回false
     */
    public static boolean isNotPositive(Number number, String paramName) {
        if (number == null || number.doubleValue() <= 0) {
            log.warn("{}参数必须大于0", paramName);
            return true;
        }
        return false;
    }
    
    /**
     * 检查字符串长度是否超过指定长度
     * 
     * @param str 待检查的字符串
     * @param maxLength 最大长度
     * @param paramName 参数名称
     * @return 如果字符串长度超过指定长度返回true，否则返回false
     */
    public static boolean isLengthExceeds(String str, int maxLength, String paramName) {
        if (str != null && str.length() > maxLength) {
            log.warn("{}参数长度不能超过{}个字符", paramName, maxLength);
            return true;
        }
        return false;
    }
    
    /**
     * 检查对象是否为null，如果为null则抛出IllegalArgumentException异常
     * 
     * @param obj 待检查的对象
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果对象为null
     */
    public static void checkNotNull(Object obj, String paramName) {
        if (isNull(obj, paramName)) {
            throw new IllegalArgumentException(paramName + "不能为null");
        }
    }
    
    /**
     * 检查字符串是否为空，如果为空则抛出IllegalArgumentException异常
     * 
     * @param str 待检查的字符串
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果字符串为空
     */
    public static void checkNotEmpty(String str, String paramName) {
        if (isEmpty(str, paramName)) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }
    
    /**
     * 检查字符串是否为空白，如果为空白则抛出IllegalArgumentException异常
     * 
     * @param str 待检查的字符串
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果字符串为空白
     */
    public static void checkNotBlank(String str, String paramName) {
        if (isBlank(str, paramName)) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }
    
    /**
     * 检查集合是否为空，如果为空则抛出IllegalArgumentException异常
     * 
     * @param collection 待检查的集合
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果集合为空
     */
    public static void checkNotEmpty(java.util.Collection<?> collection, String paramName) {
        if (isEmpty(collection, paramName)) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }
    
    /**
     * 检查数组是否为空，如果为空则抛出IllegalArgumentException异常
     * 
     * @param array 待检查的数组
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果数组为空
     */
    public static void checkNotEmpty(Object[] array, String paramName) {
        if (isEmpty(array, paramName)) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }
    
    /**
     * 检查数字是否小于等于0，如果是则抛出IllegalArgumentException异常
     * 
     * @param number 待检查的数字
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果数字小于等于0
     */
    public static void checkPositive(Number number, String paramName) {
        if (isNotPositive(number, paramName)) {
            throw new IllegalArgumentException(paramName + "必须大于0");
        }
    }
    
    /**
     * 检查字符串长度是否超过指定长度，如果是则抛出IllegalArgumentException异常
     * 
     * @param str 待检查的字符串
     * @param maxLength 最大长度
     * @param paramName 参数名称
     * @throws IllegalArgumentException 如果字符串长度超过指定长度
     */
    public static void checkLength(String str, int maxLength, String paramName) {
        if (isLengthExceeds(str, maxLength, paramName)) {
            throw new IllegalArgumentException(paramName + "长度不能超过" + maxLength + "个字符");
        }
    }
}