package pangea.hiagent.web.dto;

import java.util.Map;

/**
 * 时间轴事件工厂类
 * 专门负责根据事件类型创建相应的事件DTO对象，遵循工厂模式设计原则
 */
public class TimelineEventFactory {
    
    /**
     * 根据事件类型创建相应的事件DTO对象
     * 这是工厂类的唯一公共入口方法，确保所有事件对象创建都通过工厂完成
     * 
     * @param eventType 事件类型
     * @param eventData 事件数据
     * @return 相应的事件DTO对象
     */
    public static WorkPanelEvent createTimelineEvent(String eventType, Map<String, Object> eventData) {
        if (eventType == null || eventData == null) {
            return null;
        }
        
        switch (eventType) {
            case "thought":
                return createThoughtEvent(eventData);
            case "tool_call":
            case "tool_result":
            case "tool_error":
                return createToolEvent(eventType, eventData);
            case "embed":
                return createEmbedEvent(eventData);
            case "log":
                return createLogEvent(eventData);
            case "result":
                return createResultEvent(eventData);
            default:
                // 对于其他类型的事件，创建基础事件对象
                return createBaseEvent(eventType, eventData);
        }
    }
    
    /**
     * 创建思考事件
     * 专门处理思考类型事件的创建
     */
    private static ThoughtEvent createThoughtEvent(Map<String, Object> eventData) {
        return ThoughtEvent.builder()
                .type(getStringValue(eventData, "type"))
                .title(getStringValue(eventData, "title"))
                .timestamp(getLongValue(eventData, "timestamp"))
                .metadata(getMapValue(eventData, "metadata"))
                .content(getStringValue(eventData, "content"))
                .thinkingType(getStringValue(eventData, "thinkingType"))
                .build();
    }
    
    /**
     * 创建工具事件
     * 统一处理所有工具相关事件的创建（调用、结果、错误）
     */
    private static ToolEvent createToolEvent(String eventType, Map<String, Object> eventData) {
        return ToolEvent.builder()
                .type(eventType)
                .title(getStringValue(eventData, "title"))
                .timestamp(getLongValue(eventData, "timestamp"))
                .metadata(getMapValue(eventData, "metadata"))
                .toolName(getStringValue(eventData, "toolName"))
                .toolAction(getStringValue(eventData, "toolAction"))
                .toolInput(getMapValue(eventData, "toolInput"))
                .toolOutput(eventData.get("toolOutput"))
                .toolStatus(getStringValue(eventData, "toolStatus"))
                .executionTime(getLongValue(eventData, "executionTime"))
                .errorMessage(getStringValue(eventData, "errorMessage"))
                .errorCode(getStringValue(eventData, "errorCode"))
                .build();
    }
    
    /**
     * 创建嵌入事件
     * 专门处理嵌入类型事件的创建
     */
    private static EmbedEvent createEmbedEvent(Map<String, Object> eventData) {
        return EmbedEvent.builder()
                .type(getStringValue(eventData, "type"))
                .title(getStringValue(eventData, "title"))
                .timestamp(getLongValue(eventData, "timestamp"))
                .metadata(getMapValue(eventData, "metadata"))
                .embedUrl(getStringValue(eventData, "embedUrl"))
                .embedType(getStringValue(eventData, "embedType"))
                .embedTitle(getStringValue(eventData, "embedTitle"))
                .embedHtmlContent(getStringValue(eventData, "embedHtmlContent"))
                .build();
    }
    
    /**
     * 创建日志事件
     * 专门处理日志类型事件的创建
     */
    private static LogEvent createLogEvent(Map<String, Object> eventData) {
        return LogEvent.builder()
                .type(getStringValue(eventData, "type"))
                .title(getStringValue(eventData, "title"))
                .timestamp(getLongValue(eventData, "timestamp"))
                .metadata(getMapValue(eventData, "metadata"))
                .content(getStringValue(eventData, "content"))
                .logLevel(getStringValue(eventData, "logLevel"))
                .build();
    }
    
    /**
     * 创建结果事件
     * 专门处理结果类型事件的创建
     */
    private static ResultEvent createResultEvent(Map<String, Object> eventData) {
        return ResultEvent.builder()
                .type(getStringValue(eventData, "type"))
                .title(getStringValue(eventData, "title"))
                .timestamp(getLongValue(eventData, "timestamp"))
                .metadata(getMapValue(eventData, "metadata"))
                .content(getStringValue(eventData, "content"))
                .build();
    }
    
    /**
     * 创建基础事件
     * 处理所有其他类型事件的创建
     */
    private static WorkPanelEvent createBaseEvent(String eventType, Map<String, Object> eventData) {
        return WorkPanelEvent.builder()
                .type(eventType)
                .title(getStringValue(eventData, "title"))
                .timestamp(getLongValue(eventData, "timestamp"))
                .metadata(getMapValue(eventData, "metadata"))
                .build();
    }
    
    /**
     * 从Map中获取字符串值
     * 工具方法，用于安全地从Map中提取字符串值
     */
    private static String getStringValue(Map<String, Object> map, String key) {
        Object value = map.get(key);
        return value != null ? value.toString() : null;
    }
    
    /**
     * 从Map中获取长整型值
     * 工具方法，用于安全地从Map中提取长整型值
     */
    private static Long getLongValue(Map<String, Object> map, String key) {
        Object value = map.get(key);
        if (value instanceof Number) {
            return ((Number) value).longValue();
        } else if (value instanceof String) {
            try {
                return Long.parseLong((String) value);
            } catch (NumberFormatException e) {
                return null;
            }
        }
        return null;
    }
    
    /**
     * 从Map中获取Map值
     * 工具方法，用于安全地从Map中提取嵌套Map值
     */
    @SuppressWarnings("unchecked")
    private static Map<String, Object> getMapValue(Map<String, Object> map, String key) {
        Object value = map.get(key);
        if (value instanceof Map) {
            return (Map<String, Object>) value;
        }
        return null;
    }
}