/**
 * 全局错误处理器
 * @param error - 捕获到的错误对象
 * @returns 表示错误是否已被处理
 */
export function handleGlobalError(error: any): boolean {
  // 记录异常到监控服务
  exceptionMonitoring.recordException(
    error.name || error.constructor?.name || 'UnknownError',
    error.message || '未知错误',
    error.stack
  )
  
  // 检查是否为HTTP错误响应
  if (error.response) {
    const { status, data } = error.response
    
    // 处理401未授权错误
    if (status === 401) {
      ElMessage.error('登录已过期，请重新登录')
      
      // 清除认证信息
      const authStore = useAuthStore()
      authStore.logout()
      
      // 跳转到登录页
      if (typeof window !== 'undefined') {
        setTimeout(() => {
          window.location.href = '/login'
        }, 100)
      }
      return true // 表示错误已被处理
    }
    
    // 处理其他HTTP错误
    const message = data?.message || `请求失败 (${status})`
    ElMessage.error(message)
    return true
  }
  
  // 处理网络错误
  if (error.request) {
    ElMessage.error('网络连接失败，请检查网络设置')
    return true
  }
  
  // 处理其他错误
  ElMessage.error(error.message || '未知错误')
  return true
}

/**
 * 包装异步操作，自动处理错误
 * @param asyncFn - 异步函数
 * @param errorMessage - 自定义错误消息前缀
 * @returns 异步函数的结果
 */
export async function withErrorHandling<T>(asyncFn: () => Promise<T>, errorMessage: string = ''): Promise<T> {
  try {
    return await asyncFn()
  } catch (error: any) {
    if (errorMessage) {
      error.message = `${errorMessage}: ${error.message}`
    }
    
    // 使用全局错误处理器
    handleGlobalError(error)
    
    // 重新抛出错误，以便调用者可以根据需要进一步处理
    throw error
  }
}
