// 内容展开管理hook
import { ref, nextTick } from 'vue'
import type { Ref } from 'vue'
import type { TimelineEvent } from '../types/timeline'

export function useContentExpansion(props: {
  events: TimelineEvent[]
}) {
  // 内容展开状态管理
  const contentExpandedStates = ref<Record<number, boolean>>({})
  const contentLineCounts = ref<Record<number, number>>({})
  const contentElements = ref<Record<number, HTMLElement>>({})

  // 获取内容展开状态
  const getContentExpandedState = (timestamp: number): boolean => {
    return contentExpandedStates.value[timestamp] || false
  }

  // 注册内容元素引用
  const setContentRef = (el: HTMLElement | null, timestamp: number) => {
    if (el) {
      contentElements.value[timestamp] = el
      // 更新行数计算
      updateLineCountForElement(timestamp)
    }
  }

  // 为特定元素更新行数计算
  const updateLineCountForElement = (timestamp: number) => {
    const event = props.events.find(e => e.timestamp === timestamp)
    if (event && event.content && contentElements.value[timestamp]) {
      contentLineCounts.value[timestamp] = calculateLineCount(event.content, contentElements.value[timestamp])
      // 如果内容超过两行，初始化为折叠状态
      if (contentLineCounts.value[timestamp] > 2 && contentExpandedStates.value[timestamp] === undefined) {
        contentExpandedStates.value[timestamp] = false
      }
    }
  }

  // 切换内容展开状态
  const toggleContentExpand = (timestamp: number) => {
    contentExpandedStates.value[timestamp] = !getContentExpandedState(timestamp)
  }

  // 检查是否应该显示切换按钮
  // 计算内容实际显示行数（考虑自动换行）
  const calculateLineCount = (content: string, element?: HTMLElement): number => {
    if (!content) return 0
    
    // 如果有DOM元素，使用更精确的方法计算行数
    if (element) {
      const computedStyle = window.getComputedStyle(element)
      const lineHeight = parseFloat(computedStyle.lineHeight)
      const height = element.scrollHeight // 使用scrollHeight更准确
      
      if (lineHeight > 0) {
        const actualLines = Math.ceil(height / lineHeight)
        return actualLines
      }
    }
    
    // 回退到基于换行符的简单计算
    const lineBreaks = content.split('\n').length
    return lineBreaks
  }

  const shouldShowToggle = (timestamp: number): boolean => {
    return contentLineCounts.value[timestamp] > 2
  }

  // 更新内容行数计数
  const updateLineCounts = () => {
    nextTick(() => {
      props.events.forEach((event) => {
        if (event.content) {
          // 行数将在元素引用设置时计算
          // 这里只初始化展开状态
          if (contentExpandedStates.value[event.timestamp] === undefined) {
            contentExpandedStates.value[event.timestamp] = false
          }
        }
      })
    })
  }

  return {
    contentExpandedStates,
    contentLineCounts,
    contentElements,
    getContentExpandedState,
    setContentRef,
    toggleContentExpand,
    shouldShowToggle,
    updateLineCounts
  }
}