package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.HttpStatus;
import com.fasterxml.jackson.databind.ObjectMapper;
import pangea.hiagent.model.Tool;
import pangea.hiagent.repository.ToolRepository;

import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.net.URI;

/**
 * Tool服务类
 * 负责工具的注册、管理和调用
 */
@Slf4j
@Service
public class ToolService {
    
    private final ToolRepository toolRepository;
    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;
    private final TimeToolService timeToolService;
    
    public ToolService(ToolRepository toolRepository, TimeToolService timeToolService) {
        this.toolRepository = toolRepository;
        this.restTemplate = new RestTemplate();
        this.objectMapper = new ObjectMapper();
        this.timeToolService = timeToolService;
    }
    
    /**
     * 注册工具
     */
    @Transactional
    public Tool registerTool(Tool tool) {
        log.info("注册工具: {}", tool.getName());
        toolRepository.insert(tool);
        return tool;
    }
    
    /**
     * 更新工具
     */
    @Transactional
    public Tool updateTool(Tool tool) {
        log.info("更新工具: {}", tool.getId());
        toolRepository.updateById(tool);
        return tool;
    }
    
    /**
     * 删除工具
     */
    @Transactional
    public void deleteTool(String id) {
        log.info("删除工具: {}", id);
        toolRepository.deleteById(id);
    }
    
    /**
     * 获取工具详情
     */
    public Tool getTool(String id) {
        return toolRepository.selectById(id);
    }
    
    /**
     * 获取工具列表
     */
    public List<Tool> listTools() {
        log.debug("开始获取工具列表");
        try {
            LambdaQueryWrapper<Tool> wrapper = new LambdaQueryWrapper<>();
            wrapper.eq(Tool::getStatus, "active");
            List<Tool> tools = toolRepository.selectList(wrapper);
            log.debug("成功获取工具列表，共 {} 条记录", tools.size());
            return tools;
        } catch (Exception e) {
            log.error("获取工具列表时发生异常", e);
            throw e;
        }
    }
    
    /**
     * 分页获取工具列表
     */
    public IPage<Tool> pageTools(Long current, Long size, String name, String category) {
        Page<Tool> page = new Page<>(current, size);
        
        LambdaQueryWrapper<Tool> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(Tool::getStatus, "active");
        if (name != null) {
            wrapper.like(Tool::getName, name);
        }
        if (category != null) {
            wrapper.eq(Tool::getCategory, category);
        }
        
        return toolRepository.selectPage(page, wrapper);
    }
    
    /**
     * 按分类获取工具列表
     */
    public List<Tool> getToolsByCategory(String category) {
        LambdaQueryWrapper<Tool> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(Tool::getCategory, category);
        wrapper.eq(Tool::getStatus, "active");
        return toolRepository.selectList(wrapper);
    }
    
    /**
     * 执行工具调用
     * 
     * @param toolName 工具名称
     * @param parameters 工具参数
     * @return 工具执行结果
     */
    public String executeTool(String toolName, Map<String, Object> parameters) {
        log.info("执行工具调用: {}, 参数: {}", toolName, parameters);
        
        // 根据工具名称获取工具信息
        LambdaQueryWrapper<Tool> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(Tool::getName, toolName);
        wrapper.eq(Tool::getStatus, "active");
        Tool tool = toolRepository.selectOne(wrapper);
        
        if (tool == null) {
            log.warn("工具未找到或未激活: {}", toolName);
            return "工具未找到或未激活: " + toolName;
        }
        
        // 根据工具类型执行不同的调用逻辑
        // 这里根据工具的具体类型来执行不同的逻辑
        switch (tool.getCategory()) {
            case "API":
                return executeApiTool(tool, parameters);
            case "FUNCTION":
                // 检查是否为特殊的时间工具
                if ("get_current_time".equals(tool.getName())) {
                    return timeToolService.executeTimeTool(tool, parameters);
                }
                return executeFunctionTool(tool, parameters);
            default:
                return executeDefaultTool(tool, parameters);
        }
    }
    
    /**
     * 执行API工具调用
     */
    private String executeApiTool(Tool tool, Map<String, Object> parameters) {
        log.info("执行API工具调用: {}", tool.getName());
        
        try {
            // 构建请求URL
            String apiUrl = tool.getApiEndpoint();
            if (apiUrl == null || apiUrl.isEmpty()) {
                return "工具配置错误：API端点未设置";
            }
            
            // 解析HTTP方法
            HttpMethod httpMethod = HttpMethod.GET;
            if (tool.getHttpMethod() != null) {
                try {
                    httpMethod = HttpMethod.valueOf(tool.getHttpMethod().toUpperCase());
                } catch (IllegalArgumentException e) {
                    log.warn("无效的HTTP方法: {}, 使用GET作为默认方法", tool.getHttpMethod());
                }
            }
            
            // 构建请求头
            HttpHeaders headers = new HttpHeaders();
            headers.set("Content-Type", "application/json");
            headers.set("User-Agent", "HiAgent/1.0");
            
            // 构建请求体
            HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(parameters, headers);
            
            // 发送HTTP请求
            ResponseEntity<String> response = restTemplate.exchange(
                apiUrl, 
                httpMethod, 
                requestEntity, 
                String.class
            );
            
            // 检查响应状态
            if (response.getStatusCode() == HttpStatus.OK) {
                return response.getBody();
            } else {
                return "API调用失败，状态码: " + response.getStatusCode();
            }
        } catch (Exception e) {
            log.error("API工具调用失败: {}", tool.getName(), e);
            return "API工具调用失败: " + e.getMessage();
        }
    }
    
    /**
     * 执行函数工具调用
     */
    private String executeFunctionTool(Tool tool, Map<String, Object> parameters) {
        log.info("执行函数工具调用: {}", tool.getName());
        
        // 这里应该实现实际的函数调用逻辑
        // 例如通过反射调用指定的Java方法
        // 目前返回模拟结果
        
        Map<String, Object> result = new HashMap<>();
        result.put("tool", tool.getName());
        result.put("status", "success");
        result.put("message", "函数工具调用成功");
        result.put("parameters", parameters);
        result.put("result", "这是函数执行的结果");
        
        try {
            return objectMapper.writeValueAsString(result);
        } catch (Exception e) {
            log.error("序列化函数调用结果失败", e);
            return result.toString();
        }
    }
    
    /**
     * 执行默认工具调用
     */
    private String executeDefaultTool(Tool tool, Map<String, Object> parameters) {
        log.info("执行默认工具调用: {}", tool.getName());
        
        // 默认的工具执行逻辑
        Map<String, Object> result = new HashMap<>();
        result.put("tool", tool.getName());
        result.put("status", "success");
        result.put("message", "默认工具调用成功");
        result.put("parameters", parameters);
        result.put("result", "这是默认工具执行的结果");
        
        try {
            return objectMapper.writeValueAsString(result);
        } catch (Exception e) {
            log.error("序列化默认工具调用结果失败", e);
            return result.toString();
        }
    }
}