// 二进制消息协议处理器
import { addLog } from '@/utils/logUtils';

/**
 * WebSocket二进制消息协议处理类
 * 
 * 协议格式：
 * ┌────────┬─────────┬─────────┬──────────────┬──────────────┐
 * │ 头字节 │ 消息ID  │ 总分片数 │ 当前分片索引 │    数据      │
 * │(1B)   │ (4B)    │ (2B)    │ (2B)         │   (可变)     │
 * └────────┴─────────┴─────────┴──────────────┴──────────────┘
 * 
 * 头字节定义：
 * bit 7-5: 消息类型 (000=data, 001=ack, 010=error)
 * bit 4-2: 编码方式 (000=raw, 001=gzip, 010=brotli)
 * bit 1-0: 保留位
 */
export class BinaryMessageHandler {
  // ========== 消息类型常量 ==========
  static readonly TYPE_DATA = 0x00;    // 数据帧
  static readonly TYPE_ACK = 0x01;     // 确认帧
  static readonly TYPE_ERROR = 0x02;   // 错误帧

  // ========== 编码类型常量 ==========
  static readonly ENCODING_RAW = 0x00;      // 无编码
  static readonly ENCODING_GZIP = 0x01;     // GZIP压缩
  static readonly ENCODING_BROTLI = 0x02;   // Brotli压缩

  // ========== 协议字段大小 ==========
  static readonly HEADER_SIZE = 12;    // 协议头大小（字节）

  /**
   * 解析二进制消息头
   * 
   * @param buffer 包含协议头的ArrayBuffer（至少12字节）
   * @returns 解码后的消息头对象
   */
  static decodeHeader(buffer: ArrayBuffer): {
    messageType: number;
    encoding: number;
    messageId: number;
    totalFragments: number;
    currentFragment: number;
  } {
    if (buffer.byteLength < this.HEADER_SIZE) {
      throw new Error(
        `消息头长度不足，期望至少${this.HEADER_SIZE}字节，实际${buffer.byteLength}字节`
      );
    }

    const view = new DataView(buffer, 0, this.HEADER_SIZE);

    // 解析第1字节
    const headerByte = view.getUint8(0);
    const messageType = (headerByte >> 5) & 0x07;
    const encoding = (headerByte >> 2) & 0x07;

    // 解析消息ID (4字节，大端序)
    const messageId = view.getUint32(1);

    // 解析总分片数 (2字节，大端序)
    const totalFragments = view.getUint16(5);

    // 解析当前分片索引 (2字节，大端序)
    const currentFragment = view.getUint16(7);

    // 验证解析结果
    this.validateMessageType(messageType);
    this.validateEncoding(encoding);

    if (totalFragments === 0 || totalFragments > 65535) {
      throw new Error(`总分片数无效: ${totalFragments}`);
    }

    if (currentFragment >= totalFragments) {
      throw new Error(`分片索引越界: ${currentFragment} >= ${totalFragments}`);
    }

    return {
      messageType,
      encoding,
      messageId,
      totalFragments,
      currentFragment
    };
  }

  /**
   * 从完整消息中提取数据部分（跳过12字节的协议头）
   * 
   * @param buffer 完整的消息ArrayBuffer
   * @returns 数据部分的Uint8Array
   */
  static extractData(buffer: ArrayBuffer): Uint8Array {
    if (buffer.byteLength < this.HEADER_SIZE) {
      throw new Error(`消息长度不足，期望至少${this.HEADER_SIZE}字节`);
    }

    return new Uint8Array(buffer, this.HEADER_SIZE);
  }

  /**
   * 验证消息类型
   */
  private static validateMessageType(type: number) {
    if (type !== this.TYPE_DATA && type !== this.TYPE_ACK && type !== this.TYPE_ERROR) {
      throw new Error(`无效的消息类型: ${type}`);
    }
  }

  /**
   * 验证编码方式
   */
  private static validateEncoding(encoding: number) {
    if (encoding !== this.ENCODING_RAW && 
        encoding !== this.ENCODING_GZIP && 
        encoding !== this.ENCODING_BROTLI) {
      throw new Error(`无效的编码方式: ${encoding}`);
    }
  }

  /**
   * 将Uint8Array转换为字符串（用于HTML数据）
   */
  static uint8ArrayToString(data: Uint8Array): string {
    const decoder = new TextDecoder('utf-8');
    return decoder.decode(data);
  }

  /**
   * 获取编码方式的名称（用于日志记录）
   */
  static getEncodingName(encoding: number): string {
    switch (encoding) {
      case this.ENCODING_RAW:
        return 'RAW（无编码）';
      case this.ENCODING_GZIP:
        return 'GZIP（压缩）';
      case this.ENCODING_BROTLI:
        return 'BROTLI（压缩）';
      default:
        return `未知(${encoding})`;
    }
  }

  /**
   * 获取消息类型的名称（用于日志记录）
   */
  static getMessageTypeName(type: number): string {
    switch (type) {
      case this.TYPE_DATA:
        return 'DATA（数据帧）';
      case this.TYPE_ACK:
        return 'ACK（确认帧）';
      case this.TYPE_ERROR:
        return 'ERROR（错误帧）';
      default:
        return `未知(${type})`;
    }
  }
}

/**
 * 二进制消息分片缓存管理
 * 支持重试机制和失败恢复
 */
export class BinaryFragmentBuffer {
  private fragments: Map<number, Map<number, Uint8Array>> = new Map(); // messageId -> (fragmentIndex -> data)
  private messageInfo: Map<number, { totalFragments: number; encoding: number; receiveTime: number; retryCount: number }> = new Map();
  private readonly TIMEOUT = 60000; // 60秒超时
  private readonly MAX_RETRIES = 3; // 最大重试次数

  /**
   * 添加分片数据
   * 支持重复接收相同分片（自动去重）
   * 
   * @param messageId 消息ID
   * @param totalFragments 总分片数
   * @param currentFragment 当前分片索引
   * @param encoding 编码方式
   * @param data 分片数据
   * @returns 如果所有分片都已接收，返回完整数据；否则返回null
   */
  addFragment(
    messageId: number,
    totalFragments: number,
    currentFragment: number,
    encoding: number,
    data: Uint8Array
  ): { data: Uint8Array; encoding: number } | null {
    // 初始化消息缓存
    if (!this.fragments.has(messageId)) {
      this.fragments.set(messageId, new Map());
      this.messageInfo.set(messageId, {
        totalFragments,
        encoding,
        receiveTime: Date.now(),
        retryCount: 0
      });
      addLog(
        `🔄 开始接收二进制消息，ID=${messageId}, 总分片=${totalFragments}, 编码=${BinaryMessageHandler.getEncodingName(encoding)}`,
        'info'
      );
    }

    const fragmentMap = this.fragments.get(messageId)!;

    // 检查是否已经接收过该分片（支持幂等性，允许重复接收）
    if (fragmentMap.has(currentFragment)) {
      const existingData = fragmentMap.get(currentFragment);
      // 验证数据一致性
      if (existingData && existingData.byteLength === data.byteLength) {
        const isIdentical = this.isArrayEqual(existingData, data);
        if (isIdentical) {
          addLog(
            `⚠️  分片${currentFragment + 1}/${totalFragments}已接收（内容一致），跳过重复`,
            'warn'
          );
          return null;
        } else {
          addLog(
            `❌ 分片${currentFragment + 1}/${totalFragments}数据不一致，覆盖旧数据`,
            'error'
          );
          // 数据不一致，覆盖
          fragmentMap.set(currentFragment, data);
          return null;
        }
      }
      return null;
    }

    // 添加分片
    fragmentMap.set(currentFragment, data);
    addLog(
      `📥 接收二进制分片 ${currentFragment + 1}/${totalFragments} (大小: ${data.byteLength}字节)`,
      'info'
    );

    // 检查是否已接收所有分片
    if (fragmentMap.size !== totalFragments) {
      return null;
    }

    // 所有分片已接收，合并数据
    addLog(`✅ 所有分片接收完成，开始合并数据...`, 'info');

    // 按顺序合并分片
    const totalSize = Array.from(fragmentMap.values()).reduce((sum, chunk) => sum + chunk.byteLength, 0);
    const merged = new Uint8Array(totalSize);
    let offset = 0;

    for (let i = 0; i < totalFragments; i++) {
      const fragment = fragmentMap.get(i);
      if (!fragment) {
        addLog(`❌ 分片${i}丢失，无法完成合并`, 'error');
        this.cleanup(messageId);
        throw new Error(`分片${i}丢失`);
      }
      merged.set(fragment, offset);
      offset += fragment.byteLength;
    }

    addLog(`✓ 数据合并完成，总大小: ${merged.byteLength}字节`, 'info');

    // 清理缓存
    this.cleanup(messageId);

    return {
      data: merged,
      encoding: this.messageInfo.get(messageId)?.encoding || BinaryMessageHandler.ENCODING_RAW
    };
  }

  /**
   * 清理特定消息的缓存
   */
  private cleanup(messageId: number) {
    this.fragments.delete(messageId);
    this.messageInfo.delete(messageId);
  }

  /**
   * 清理过期的消息缓存（超过30秒未完成）
   */
  cleanupExpired() {
    const now = Date.now();
    const expiredMessages: number[] = [];

    for (const [messageId, info] of this.messageInfo.entries()) {
      if (now - info.receiveTime > this.TIMEOUT) {
        expiredMessages.push(messageId);
        const fragmentMap = this.fragments.get(messageId);
        if (fragmentMap) {
          const progress = fragmentMap.size;
          const total = info.totalFragments;
          const percentage = ((progress / total) * 100).toFixed(1);
          addLog(
            `⏱️  消息${messageId}接收超时，进度${progress}/${total}(${percentage}%)，已清空缓存`,
            'warn'
          );
        }
      }
    }

    expiredMessages.forEach(id => this.cleanup(id));
  }

  /**
   * 获取当前缓存状态（用于调试）
   */
  getStatus(): string {
    const messageCount = this.fragments.size;
    const totalFragments = Array.from(this.fragments.values()).reduce((sum, map) => sum + map.size, 0);
    return `缓存消息数: ${messageCount}, 缓存分片数: ${totalFragments}`;
  }

  /**
   * 高效比较两个Uint8Array数组是否相等
   * @param a 第一个数组
   * @param b 第二个数组
   * @returns 如果相等返回true，否则返回false
   */
  private isArrayEqual(a: Uint8Array, b: Uint8Array): boolean {
    if (a.length !== b.length) return false;
    for (let i = 0; i < a.length; i++) {
      if (a[i] !== b[i]) return false;
    }
    return true;
  }
}

/**
 * 二进制消息处理函数
 * 在WebSocket消息处理中调用
 * 
 * 使用示例：
 * ```typescript
 * const fragmentBuffer = new BinaryFragmentBuffer();
 * 
 * ws.onmessage = (event) => {
 *   if (event.data instanceof ArrayBuffer) {
 *     const result = BinaryMessageHandler.handleBinaryMessage(
 *       event.data,
 *       fragmentBuffer,
 *       (data, url) => renderDOM(data, url)  // 完成回调
 *     );
 *   }
 * };
 * ```
 */
export function handleBinaryMessage(
  buffer: ArrayBuffer,
  fragmentBuffer: BinaryFragmentBuffer,
  onComplete?: (data: any, encoding: number) => void
): boolean {
  // 记录接收到的消息基本信息
  addLog(`📥 开始处理二进制消息，大小: ${buffer.byteLength} 字节`, 'debug');
  try {
    addLog(`📥 接收到WebSocket消息，大小: ${buffer.byteLength}字节`, 'debug');
    
    // 验证缓冲区大小
    if (buffer.byteLength < BinaryMessageHandler.HEADER_SIZE) {
      addLog(
        `❌ 接收到的消息过短 (${buffer.byteLength}字节，需要至少${BinaryMessageHandler.HEADER_SIZE}字节)`,
        'error'
      );
      return false;
    }
    
    // 解析协议头
    const header = BinaryMessageHandler.decodeHeader(buffer);

    addLog(
      `📨 接收二进制消息: 类型=${BinaryMessageHandler.getMessageTypeName(header.messageType)}, ` +
      `ID=${header.messageId}, 分片=${header.currentFragment + 1}/${header.totalFragments}, ` +
      `总大小=${buffer.byteLength}字节`,
      'debug'
    );

    // 处理不同的消息类型
    switch (header.messageType) {
      case BinaryMessageHandler.TYPE_DATA:
        // 数据帧：提取数据并处理
        const data = BinaryMessageHandler.extractData(buffer);
        const result = fragmentBuffer.addFragment(
          header.messageId,
          header.totalFragments,
          header.currentFragment,
          header.encoding,
          data
        );

        if (result) {
          // 所有分片已接收，处理完整数据
          addLog(
            `✓ 消息${header.messageId}已完整接收，开始处理数据...`,
            'info'
          );

          // 处理编码方式
          let decodedData: any;
          try {
            if (header.encoding === BinaryMessageHandler.ENCODING_RAW) {
              // 原始数据，直接转换为字符串
              const htmlString = BinaryMessageHandler.uint8ArrayToString(result.data);
              decodedData = htmlString; // 直接传递HTML字符串
              addLog(`✓ 原始数据处理完成，数据大小: ${htmlString.length} 字符`, 'info');
            } else if (header.encoding === BinaryMessageHandler.ENCODING_GZIP) {
              addLog('⚠️  GZIP压缩数据，需要额外的解压缩库支持（暂未实现）', 'warn');
              return false;
            } else if (header.encoding === BinaryMessageHandler.ENCODING_BROTLI) {
              addLog('⚠️  Brotli压缩数据，需要额外的解压缩库支持（暂未实现）', 'warn');
              return false;
            }

            if (onComplete) {
              addLog(`📤 调用onComplete回调，数据类型: ${typeof decodedData}`, 'debug');
              onComplete(decodedData, header.encoding);
            }
            return true;
          } catch (error) {
            addLog(`❌ 处理数据失败: ${(error as Error).message}`, 'error');
            return false;
          }
        }
        return true;

      case BinaryMessageHandler.TYPE_ACK:
        // 确认帧：客户端无需处理（用于服务器接收确认）
        addLog(`ℹ️  接收到确认帧 (ACK)，消息ID=${header.messageId}`, 'debug');
        return true;

      case BinaryMessageHandler.TYPE_ERROR:
        // 错误帧：提取错误信息并记录详细日志
        try {
          const errorData = BinaryMessageHandler.extractData(buffer);
          const errorMessage = BinaryMessageHandler.uint8ArrayToString(errorData);
          addLog(`❌ 服务器错误 (消息ID=${header.messageId}): ${errorMessage}`, 'error');
          addLog(`📝 错误详情: 缓冲区大小=${buffer.byteLength}字节`, 'debug');
          
          // 如果是第三方页面错误，给出友好提示
          if (errorMessage.includes('sso.hisense.com') && errorMessage.includes('setRequestHeader')) {
            addLog('💡 提示：这是海信SSO系统页面的JavaScript错误，不影响系统核心功能', 'info');
          }
        } catch (parseError) {
          const parseError_ = parseError as Error;
          addLog(
            `❌ 服务器错误，且错误信息解析失败: ${parseError_.message}`,
            'error'
          );
          addLog(`🔍 解析失败堆栈: ${parseError_.stack || 'N/A'}`, 'debug');
        }
        return false;

      default:
        addLog(`⚠️  未知的消息类型: ${header.messageType}`, 'warn');
        return false;
    }
  } catch (error) {
    const error_ = error as Error;
    const errorMsg = error_.message;
    const errorStack = error_.stack || 'N/A';
    addLog(`❌ 处理二进制消息失败: ${errorMsg}`, 'error');
    addLog(`📋 错误详情: ${errorMsg}`, 'debug');
    addLog(`📍 错误堆栈: ${errorStack}`, 'debug');
    
    // 记录接收到的原始数据大小用于调试
    addLog(
      `📊 接收缓冲区大小: ${buffer.byteLength}字节 (最小需要: ${BinaryMessageHandler.HEADER_SIZE}字节)`,
      'debug'
    );
    
    // 如果是头部解析错误，可能是网络问题，记录更多调试信息
    if (buffer.byteLength < BinaryMessageHandler.HEADER_SIZE) {
      addLog(
        `🔧 缓冲区过小，这可能表示网络传输问题或消息被截断`,
        'warn'
      );
    }
    
    return false;
  }
}
