package pangea.hiagent.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.ToolConfig;
import pangea.hiagent.repository.ToolConfigRepository;
import pangea.hiagent.service.ToolConfigService;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 工具配置服务实现类
 * 用于处理工具参数配置的读取和保存
 */
@Slf4j
@Service
public class ToolConfigServiceImpl implements ToolConfigService {
    
    @Autowired
    private ToolConfigRepository toolConfigRepository;
    
    @Override
    public Map<String, String> getToolParams(String toolName) {
        log.debug("获取工具参数配置，工具名称：{}", toolName);
        Map<String, String> params = new HashMap<>();
        
        try {
            List<Map<String, Object>> paramValues = toolConfigRepository.findParamValuesByToolName(toolName);
            for (Map<String, Object> paramValue : paramValues) {
                String paramName = (String) paramValue.get("param_name");
                String value = (String) paramValue.get("param_value");
                params.put(paramName, value);
            }
        } catch (Exception e) {
            log.error("获取工具参数配置失败：{}", e.getMessage(), e);
        }
        
        return params;
    }
    
    @Override
    public String getParamValue(String toolName, String paramName) {
        log.debug("获取工具参数值，工具名称：{}，参数名称：{}", toolName, paramName);
        
        try {
            ToolConfig toolConfig = toolConfigRepository.findByToolNameAndParamName(toolName, paramName);
            if (toolConfig != null) {
                return toolConfig.getParamValue();
            }
        } catch (Exception e) {
            log.error("获取工具参数值失败：{}", e.getMessage(), e);
        }
        
        return null;
    }
    
    @Override
    public void saveParamValue(String toolName, String paramName, String paramValue) {
        log.debug("保存工具参数值，工具名称：{}，参数名称：{}，参数值：{}", toolName, paramName, paramValue);
        
        try {
            ToolConfig toolConfig = toolConfigRepository.findByToolNameAndParamName(toolName, paramName);
            if (toolConfig != null) {
                toolConfig.setParamValue(paramValue);
                toolConfigRepository.updateById(toolConfig);
            } else {
                // 如果配置不存在，创建新配置
                toolConfig = new ToolConfig();
                toolConfig.setToolName(toolName);
                toolConfig.setParamName(paramName);
                toolConfig.setParamValue(paramValue);
                toolConfigRepository.insert(toolConfig);
            }
        } catch (Exception e) {
            log.error("保存工具参数值失败：{}", e.getMessage(), e);
        }
    }
    
    @Override
    public List<ToolConfig> getAllToolConfigs() {
        log.debug("获取所有工具配置");
        
        try {
            return toolConfigRepository.findAllActive();
        } catch (Exception e) {
            log.error("获取所有工具配置失败：{}", e.getMessage(), e);
            return List.of();
        }
    }
    
    @Override
    public ToolConfig getToolConfig(String toolName, String paramName) {
        log.debug("获取工具配置，工具名称：{}，参数名称：{}", toolName, paramName);
        
        try {
            return toolConfigRepository.findByToolNameAndParamName(toolName, paramName);
        } catch (Exception e) {
            log.error("获取工具配置失败：{}", e.getMessage(), e);
            return null;
        }
    }
    
    @Override
    public ToolConfig saveToolConfig(ToolConfig toolConfig) {
        log.debug("保存工具配置：{}", toolConfig);
        
        try {
            if (toolConfig.getId() != null) {
                toolConfigRepository.updateById(toolConfig);
            } else {
                // 检查是否已存在相同的工具名称和参数名称的配置
                ToolConfig existingConfig = toolConfigRepository.findByToolNameAndParamName(
                        toolConfig.getToolName(), toolConfig.getParamName());
                if (existingConfig != null) {
                    toolConfig.setId(existingConfig.getId());
                    toolConfigRepository.updateById(toolConfig);
                } else {
                    toolConfigRepository.insert(toolConfig);
                }
            }
            return toolConfig;
        } catch (Exception e) {
            log.error("保存工具配置失败：{}", e.getMessage(), e);
            return null;
        }
    }
    
    @Override
    public void deleteToolConfig(String id) {
        log.debug("删除工具配置，ID：{}", id);
        
        try {
            toolConfigRepository.deleteById(id);
        } catch (Exception e) {
            log.error("删除工具配置失败：{}", e.getMessage(), e);
        }
    }
    
    @Override
    public List<ToolConfig> getToolConfigsByToolName(String toolName) {
        log.debug("根据工具名称获取工具配置列表，工具名称：{}", toolName);
        
        try {
            return toolConfigRepository.findByToolName(toolName);
        } catch (Exception e) {
            log.error("根据工具名称获取工具配置列表失败：{}", e.getMessage(), e);
            return List.of();
        }
    }
}