import type { TimelineEvent } from '../types/timeline';

/**
 * SSE服务类，用于处理与后端的Server-Sent Events连接
 */
export class SseService {
  private eventSource: EventSource | null = null;
  private eventListeners: Map<string, Array<(data: any) => void>> = new Map();
  private reconnectAttempts = 0;
  private maxReconnectAttempts = 5;
  private reconnectDelay = 1000;
  private url: string;
  private isConnecting = false;

  constructor(url: string = '/api/v1/events') {
    this.url = url;
  }

  /**
   * 连接到SSE服务器
   */
  connect(): void {
    if (this.eventSource || this.isConnecting) {
      return;
    }

    this.isConnecting = true;
    this.reconnectAttempts = 0;

    try {
      // 创建EventSource连接
      this.eventSource = new EventSource(this.url);

      // 监听open事件
      this.eventSource.onopen = () => {
        console.log('[SSE] 连接已建立');
        this.reconnectAttempts = 0;
        this.isConnecting = false;
        this.dispatchEvent('connect', {});
      };

      // 监听message事件（默认事件）
      this.eventSource.onmessage = (event) => {
        try {
          const data = JSON.parse(event.data);
          console.log('[SSE] 收到消息:', data);
          this.dispatchEvent('message', data);
          
          // 如果是时间轴事件，分发特定事件
          if (data.type) {
            this.dispatchEvent('timeline-event', data);
          }
        } catch (error) {
          console.error('[SSE] 解析消息失败:', error);
        }
      };

      // 监听error事件
      this.eventSource.onerror = (error) => {
        console.error('[SSE] 连接错误:', error);
        this.isConnecting = false;
        this.dispatchEvent('error', error);
        this.handleReconnect();
      };

      // 监听特定事件类型
      this.eventSource.addEventListener('error', (event) => {
        try {
          const data = JSON.parse((event as MessageEvent).data);
          console.error('[SSE] 服务器错误:', data);
          this.dispatchEvent('server-error', data);
        } catch (error) {
          console.error('[SSE] 解析错误消息失败:', error);
        }
      });

      this.eventSource.addEventListener('token', (event) => {
        try {
          const data = JSON.parse((event as MessageEvent).data);
          console.log('[SSE] 收到Token:', data);
          this.dispatchEvent('token', data);
        } catch (error) {
          console.error('[SSE] 解析Token消息失败:', error);
        }
      });
    } catch (error) {
      console.error('[SSE] 创建连接失败:', error);
      this.isConnecting = false;
      this.handleReconnect();
    }
  }

  /**
   * 断开SSE连接
   */
  disconnect(): void {
    if (this.eventSource) {
      this.eventSource.close();
      this.eventSource = null;
      console.log('[SSE] 连接已断开');
      this.dispatchEvent('disconnect', {});
    }
    this.isConnecting = false;
    this.reconnectAttempts = 0;
  }

  /**
   * 处理重连逻辑
   */
  private handleReconnect(): void {
    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      console.error('[SSE] 达到最大重连次数，停止重连');
      this.dispatchEvent('connection-failed', { attempts: this.reconnectAttempts });
      window.dispatchEvent(new CustomEvent('sse-connection-failed', { detail: { attempts: this.reconnectAttempts } }));
      return;
    }

    this.reconnectAttempts++;
    const delay = this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1);
    
    console.log(`[SSE] 尝试重连... (${this.reconnectAttempts}/${this.maxReconnectAttempts})，延迟 ${delay}ms`);
    
    setTimeout(() => {
      this.connect();
    }, delay);
  }

  /**
   * 添加事件监听器
   * @param eventType 事件类型
   * @param callback 回调函数
   */
  on(eventType: string, callback: (data: any) => void): void {
    if (!this.eventListeners.has(eventType)) {
      this.eventListeners.set(eventType, []);
    }
    this.eventListeners.get(eventType)?.push(callback);
  }

  /**
   * 移除事件监听器
   * @param eventType 事件类型
   * @param callback 回调函数（可选，如果不提供则移除所有该类型的监听器）
   */
  off(eventType: string, callback?: (data: any) => void): void {
    if (!this.eventListeners.has(eventType)) {
      return;
    }

    if (callback) {
      const callbacks = this.eventListeners.get(eventType)?.filter(cb => cb !== callback) || [];
      this.eventListeners.set(eventType, callbacks);
    } else {
      this.eventListeners.delete(eventType);
    }
  }

  /**
   * 分发事件
   * @param eventType 事件类型
   * @param data 事件数据
   */
  private dispatchEvent(eventType: string, data: any): void {
    const callbacks = this.eventListeners.get(eventType) || [];
    callbacks.forEach(callback => {
      try {
        callback(data);
      } catch (error) {
        console.error(`[SSE] 执行事件监听器失败 (${eventType}):`, error);
      }
    });
  }

  /**
   * 检查连接状态
   */
  isConnected(): boolean {
    return this.eventSource !== null && this.eventSource.readyState === EventSource.OPEN;
  }
}

// 创建单例实例
const sseService = new SseService();

export default sseService;