package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;

import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.model.Agent;
import pangea.hiagent.web.dto.AgentRequest;

import java.io.IOException;

/**
 * SSE Token发射器
 * 专注于将token转换为SSE事件并发送
 * 无状态设计，每次使用时创建新实例
 */
@Slf4j
public class SseTokenEmitter implements TokenConsumerWithCompletion {
    
    private final UserSseService userSseService;
    
    // 所有状态通过构造函数一次性传入
    private final SseEmitter emitter;
    private final Agent agent;
    private final AgentRequest request;
    private final String userId;
    private final CompletionCallback completionCallback;
    private String emitterId;
    
    /**
     * 构造函数
     * @param userSseService SSE服务
     * @param emitter SSE发射器
     * @param agent Agent对象
     * @param request 请求对象
     * @param userId 用户ID
     * @param completionCallback 完成回调
     */
    public SseTokenEmitter(UserSseService userSseService, SseEmitter emitter, Agent agent, 
                           AgentRequest request, String userId, CompletionCallback completionCallback) {
        this.userSseService = userSseService;
        this.emitter = emitter;
        this.agent = agent;
        this.request = request;
        this.userId = userId;
        this.completionCallback = completionCallback;
    }
    
    /**
     * 无参构造函数，用于Spring容器初始化
     */
    public SseTokenEmitter() {
        this(null, null, null, null, null, null);
    }

    /**
     * 构造函数，用于Spring容器初始化（带UserSseService参数）
     */
    public SseTokenEmitter(UserSseService userSseService) {
        this(userSseService, null, null, null, null, null);
    }
    
    /**
     * 创建新的SseTokenEmitter实例
     * @param emitter SSE发射器
     * @param agent Agent对象
     * @param request 请求对象
     * @param userId 用户ID
     * @param completionCallback 完成回调
     * @return 新的SseTokenEmitter实例
     */
    public SseTokenEmitter createNewInstance(SseEmitter emitter, Agent agent, AgentRequest request, 
                                            String userId, CompletionCallback completionCallback) {
        return new SseTokenEmitter(userSseService, emitter, agent, request, userId, completionCallback);
    }
    
    @Override
    public void accept(String token) {
        // 使用JSON格式发送token，确保转义序列被正确处理
        try {
            if (emitter != null && userSseService.isEmitterValidSafe(emitter)) {
                // 检查是否是错误消息（以[错误]或[ERROR]开头）
                if (token != null && (token.startsWith("[错误]") || token.startsWith("[ERROR]"))) {
                    // 发送标准错误事件而不是纯文本
                    userSseService.sendErrorEvent(emitter, token);
                } else {
                    // 使用SSE标准事件格式发送token，以JSON格式确保转义序列正确处理
                    userSseService.sendTokenEvent(emitter, token);
                }
            } else {
                log.debug("SSE emitter已无效，跳过发送token");
            }
        } catch (IllegalStateException e) {
            // 处理emitter已关闭的情况，这通常是由于客户端断开连接
            log.debug("无法发送token，SSE emitter已关闭: {}", e.getMessage());
            // 将emitter标记为已完成，避免后续再次尝试发送
            if (emitter != null) {
                userSseService.removeEmitter(emitter);
            }
        } catch (IOException e) {
            // 处理IO异常，这通常是由于客户端断开连接或网络问题
            log.debug("无法发送token，IO异常: {}", e.getMessage());
            // 将emitter标记为已完成，避免后续再次尝试发送
            if (emitter != null) {
                userSseService.removeEmitter(emitter);
            }
        } catch (Exception e) {
            log.error("发送token失败", e);
            // 对于其他异常，也将emitter标记为已完成，避免后续再次尝试发送
            if (emitter != null) {
                userSseService.removeEmitter(emitter);
            }
        }
    }
    
    @Override
    public void onComplete(String fullContent) {
        try {
            if (emitter != null && !userSseService.isEmitterCompleted(emitter)) {
                // 发送完成事件
                emitter.send(SseEmitter.event().name("done").data("[DONE]").build());
                log.debug("完成信号已发送");
            }
            
            // 调用完成回调
            if (completionCallback != null) {
                completionCallback.onComplete(emitter, agent, request, userId, fullContent);
            }
        } catch (IllegalStateException e) {
            // 处理emitter已关闭的情况，这通常是由于客户端断开连接
            log.debug("无法发送完成信号，SSE emitter已关闭: {}", e.getMessage());
        } catch (IOException e) {
            // 处理IO异常，这通常是由于客户端断开连接或网络问题
            log.debug("无法发送完成信号，IO异常: {}", e.getMessage());
        } catch (Exception e) {
            log.error("处理完成事件失败", e);
        } finally {
            // 关闭连接
            closeEmitter();
        }
    }
    
    /**
     * 安全关闭SSE连接
     */
    public void closeEmitter() {
        try {
            if (emitter != null && !userSseService.isEmitterCompleted(emitter)) {
                // emitter.complete();
                log.debug("SSE连接已关闭");
            }
        } catch (Exception ex) {
            log.error("完成emitter时发生错误", ex);
        }
    }
    
    /**
     * 完成回调接口
     */
    @FunctionalInterface
    public interface CompletionCallback {
        void onComplete(SseEmitter emitter, Agent agent, AgentRequest request, String userId, String fullContent);
    }
    public void setEmitterId(String emitterId) {
        this.emitterId  = emitterId;
    }
    public String getEmitterId() {
        return emitterId;
    }
    public String getUserId() {
        return userId;
    }
}