package pangea.hiagent.web.service;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.data.AgentTaskStatus;
import pangea.hiagent.agent.data.AgentTaskStatusRepository;


import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 多Agent协作服务
 * 提供协作任务管理相关的业务逻辑
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class CollaborationService {

    private final AgentTaskStatusRepository taskStatusRepository;

    /**
     * 获取协作任务统计信息
     */
    public Map<String, Object> getCollaborationStats() {
        try {
            // 获取所有任务状态
            List<AgentTaskStatus> allTasks = taskStatusRepository.selectList(null);
            
            // 统计不同状态的任务数量
            Map<String, Long> statusCounts = allTasks.stream()
                    .filter(task -> task.getStatus() != null) // 过滤空状态的任务
                    .collect(Collectors.groupingBy(
                            AgentTaskStatus::getStatus,
                            Collectors.counting()
                    ));
            
            return Map.of(
                    "totalTasks", allTasks.size(),
                    "runningTasks", statusCounts.getOrDefault("RUNNING", 0L),
                    "successTasks", statusCounts.getOrDefault("SUCCESS", 0L),
                    "failedTasks", statusCounts.getOrDefault("FAIL", 0L),
                    "pendingTasks", statusCounts.getOrDefault("READY", 0L)
            );
        } catch (Exception e) {
            log.error("获取协作任务统计信息失败", e);
            return Map.of(
                    "totalTasks", 0L,
                    "runningTasks", 0L,
                    "successTasks", 0L,
                    "failedTasks", 0L,
                    "pendingTasks", 0L
            );
        }
    }

    /**
     * 获取协作任务列表
     */
    public List<AgentTaskStatus> getCollaborationTasks(String status, String keyword) {
        try {
            List<AgentTaskStatus> allTasks = taskStatusRepository.selectList(null);
            
            return allTasks.stream()
                    .filter(task -> {
                        boolean statusMatch = status == null || status.isEmpty() || 
                                (task.getStatus() != null && task.getStatus().equalsIgnoreCase(status));
                        boolean keywordMatch = keyword == null || keyword.isEmpty() || 
                                (task.getTaskId() != null && task.getTaskId().toLowerCase().contains(keyword.toLowerCase()));
                        return statusMatch && keywordMatch;
                    })
                    .collect(Collectors.toList());
        } catch (Exception e) {
            log.error("获取协作任务列表失败: status={}, keyword={}", status, keyword, e);
            return List.of();
        }
    }

    /**
     * 获取协作任务列表（带分页）
     */
    public List<AgentTaskStatus> getCollaborationTasks(String status, String keyword, int page, int size) {
        try {
            if (page < 1 || size < 1) {
                log.warn("分页参数无效: page={}, size={}", page, size);
                return List.of();
            }
            
            List<AgentTaskStatus> filteredTasks = getCollaborationTasks(status, keyword);
            
            int start = (page - 1) * size;
            int end = Math.min(start + size, filteredTasks.size());
            
            if (start >= filteredTasks.size()) {
                start = Math.max(0, filteredTasks.size() - size);
                end = filteredTasks.size();
            }
            
            if (start < filteredTasks.size()) {
                return filteredTasks.subList(start, end);
            } else {
                return List.of();
            }
        } catch (Exception e) {
            log.error("获取协作任务列表（分页）失败: status={}, keyword={}, page={}, size={}", status, keyword, page, size, e);
            return List.of();
        }
    }

    /**
     * 获取任务总数（根据过滤条件）
     */
    public int getTotalTaskCount(String status, String keyword) {
        try {
            return getCollaborationTasks(status, keyword).size();
        } catch (Exception e) {
            log.error("获取任务总数失败: status={}, keyword={}", status, keyword, e);
            return 0;
        }
    }

    /**
     * 重试任务
     */
    public void retryTask(String taskId) {
        if (taskId == null || taskId.trim().isEmpty()) {
            log.warn("任务ID为空，无法重试");
            return;
        }
        
        try {
            // 这里需要调用TaskStatusService的重试逻辑
            // 由于TaskStatusService.retryTask是私有方法，我们需要通过其他方式实现
            AgentTaskStatus status = taskStatusRepository.selectById(taskId);
            if (status != null) {
                int newRetryCount = (status.getRetryCount() != null ? status.getRetryCount() : 0) + 1;
                status.setRetryCount(newRetryCount);
                
                // 如果重试次数超过阈值，标记为失败
                if (newRetryCount > 3) { // 最多重试3次
                    status.setStatus("FAIL");
                } else {
                    // 更新状态为就绪状态以重新执行
                    status.setStatus("READY");
                }
                
                taskStatusRepository.updateById(status);
                log.info("任务{}重试，当前重试次数: {}", taskId, newRetryCount);
            } else {
                log.warn("找不到任务ID为{}的任务，无法重试", taskId);
            }
        } catch (Exception e) {
            log.error("重试任务失败: taskId={}", taskId, e);
            throw e;
        }
    }
}