package pangea.hiagent.agent.service;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.data.*;
import pangea.hiagent.web.service.CollaborationService;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

/**
 * 多Agent协同任务执行服务
 * 负责动态生成和执行多Agent协同任务
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class MultiAgentTaskExecutionService {

    private final MasterAgent masterAgent;
    private final CollaborationService collaborationService;
    private final MultiAgentTaskGeneratorService taskGeneratorService;

    /**
     * 执行多Agent协同任务
     * 
     * @param mainTask 主任务对象
     * @return 任务执行结果
     */
    public AgentResult executeCollaborationTask(MainTask mainTask) {
        log.info("开始执行多Agent协同任务: {}", mainTask.getTaskId());
        
        try {
            // 通过MasterAgent执行主任务
            AgentResult result = masterAgent.executeMainTask(mainTask);
            
            log.info("多Agent协同任务执行完成: {}，结果: {}", mainTask.getTaskId(), result.isSuccess());
            return result;
        } catch (Exception e) {
            log.error("执行多Agent协同任务失败: {}", mainTask.getTaskId(), e);
            return new AgentResult(false, "执行多Agent协同任务失败: " + e.getMessage(), null);
        }
    }

    /**
     * 异步执行多Agent协同任务
     * 
     * @param mainTask 主任务对象
     * @return CompletableFuture包装的任务执行结果
     */
    public CompletableFuture<AgentResult> executeCollaborationTaskAsync(MainTask mainTask) {
        return CompletableFuture.supplyAsync(() -> executeCollaborationTask(mainTask));
    }

    /**
     * 根据用户输入分析并执行多Agent协同任务
     * 
     * @param userInput 用户输入
     * @param agent 当前Agent
     * @return 任务执行结果
     */
    public AgentResult processUserInputForCollaboration(String userInput, pangea.hiagent.model.Agent agent) {
        log.info("处理用户输入以执行多Agent协同任务: {}", userInput);
        
        // 检查是否需要启动多Agent协同
        if (!taskGeneratorService.shouldStartCollaboration(userInput)) {
            log.info("用户输入不需要多Agent协同，返回空结果");
            return null;
        }
        
        // 生成协同任务
        MainTask mainTask = taskGeneratorService.generateCollaborationTask(userInput, agent);
        
        // 执行协同任务
        return executeCollaborationTask(mainTask);
    }

    /**
     * 异步处理用户输入并执行多Agent协同任务
     * 
     * @param userInput 用户输入
     * @param agent 当前Agent
     * @return CompletableFuture包装的任务执行结果
     */
    public CompletableFuture<AgentResult> processUserInputForCollaborationAsync(String userInput, pangea.hiagent.model.Agent agent) {
        return CompletableFuture.supplyAsync(() -> processUserInputForCollaboration(userInput, agent));
    }

    /**
     * 获取可用的从Agent列表
     * 
     * @return 从Agent能力标签列表
     */
    public List<String> getAvailableSlaveAgents() {
        return taskGeneratorService.getAvailableSlaveAgents();
    }

    /**
     * 检查是否需要启动多Agent协同
     * 
     * @param userInput 用户输入
     * @return 是否需要启动多Agent协同
     */
    public boolean shouldStartCollaboration(String userInput) {
        return taskGeneratorService.shouldStartCollaboration(userInput);
    }
}