package pangea.hiagent.agent.service;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.model.Agent;
import pangea.hiagent.web.dto.AgentRequest;

import java.util.function.BiConsumer;

/**
 * SSE Token发射器
 * 用于向客户端发送流式响应
 */
@Slf4j
@RequiredArgsConstructor
public class SseTokenEmitter implements TokenConsumerWithCompletion {

    private final UserSseService userSseService;
    private final SseEmitter emitter;
    private final Agent agent;
    private final AgentRequest request;
    private final String userId;
    private final BiConsumer<SseEmitter, String> completionCallback;

    /**
     * 发送token
     */
    public void emitToken(String token) {
        if (userSseService.isEmitterCompleted(emitter)) {
            log.debug("SSE连接已关闭，跳过发送token");
            return;
        }

        try {
            emitter.send(SseEmitter.event()
                    .name("token")
                    .data("{\"token\":\"" + escapeJson(token) + "\"}"));
        } catch (Exception e) {
            log.error("发送token失败", e);
            userSseService.closeEmitter(emitter);
        }
    }

    @Override
    public void accept(String token) {
        emitToken(token);
    }

    @Override
    public void onComplete(String fullContent) {
        emitComplete(fullContent);
        complete();
    }

    /**
     * 发送完整内容
     */
    public void emitComplete(String content) {
        if (userSseService.isEmitterCompleted(emitter)) {
            log.debug("SSE连接已关闭，跳过发送完成事件");
            return;
        }

        try {
            emitter.send(SseEmitter.event()
                    .name("complete")
                    .data("{\"content\":\"" + escapeJson(content) + "\"}"));
        } catch (Exception e) {
            log.error("发送完成事件失败", e);
        }
    }

    /**
     * 发送错误
     */
    public void emitError(String error) {
        if (userSseService.isEmitterCompleted(emitter)) {
            log.debug("SSE连接已关闭，跳过发送错误事件");
            return;
        }

        try {
            emitter.send(SseEmitter.event()
                    .name("error")
                    .data("{\"error\":\"" + escapeJson(error) + "\"}"));
        } catch (Exception e) {
            log.error("发送错误事件失败", e);
        }
    }

    /**
     * 检查发射器是否已完成
     */
    public boolean isCompleted() {
        return userSseService.isEmitterCompleted(emitter);
    }

    /**
     * 完成发射
     */
    public void complete() {
        if (!userSseService.isEmitterCompleted(emitter)) {
            try {
                log.info("SseTokenEmitter.complete: 完成SSE发射器");
                emitter.complete();
            } catch (Exception e) {
                log.error("完成SSE发射器失败", e);
            }
        }

        if (completionCallback != null) {
            completionCallback.accept(emitter, request.getUserMessage());
        }
    }

    /**
     * 转义JSON字符串中的特殊字符
     */
    private String escapeJson(String input) {
        if (input == null) {
            return null;
        }
        return input.replace("\\", "\\\\")
                  .replace("\"", "\\\"")
                  .replace("\n", "\\n")
                  .replace("\r", "\\r")
                  .replace("\t", "\\t");
    }
}