package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * 模块化配置主类
 * 统一管理所有扩展模块的配置
 */
@Slf4j
@Configuration
public class ModuleConfiguration {

    /**
     * 模块管理服务，用于动态控制模块状态
     */
    @Bean
    @RefreshScope
    @ConditionalOnProperty(prefix = "agent.module", name = "enabled", havingValue = "true", matchIfMissing = true)
    public ModuleManagementService moduleManagementService() {
        log.info("初始化模块管理服务");
        return new ModuleManagementService();
    }
}

/**
 * 模块管理服务，提供模块动态控制能力
 */
@Slf4j
class ModuleManagementService {
    
    private final java.util.Map<String, Boolean> moduleStates = new java.util.HashMap<>();
    
    /**
     * 检查指定模块是否启用
     */
    public boolean isModuleEnabled(String moduleName) {
        // 从内存中获取模块状态，如果不存在则使用默认值true
        Boolean state = moduleStates.get(moduleName);
        if (state != null) {
            log.debug("从内存获取模块状态: {} = {}", moduleName, state);
            return state;
        }
        
        // 尝试从配置中心或运行时配置获取模块状态
        // 这里可以扩展为从配置中心（如Nacos、Apollo）或数据库获取配置
        String configValue = System.getProperty("agent.module." + moduleName + ".enabled");
        if (configValue != null) {
            boolean configState = Boolean.parseBoolean(configValue);
            moduleStates.put(moduleName, configState);
            log.debug("从系统属性获取模块状态: {} = {}", moduleName, configState);
            return configState;
        }
        
        // 默认启用模块
        log.debug("模块状态未配置，使用默认值: {} = true", moduleName);
        return true;
    }
    
    /**
     * 动态启用模块
     */
    public void enableModule(String moduleName) {
        log.info("动态启用模块: {}", moduleName);
        moduleStates.put(moduleName, true);
        
        // 可以在这里添加额外的启用逻辑，如重新初始化模块组件等
        log.debug("模块已启用并更新状态: {}", moduleName);
    }
    
    /**
     * 动态禁用模块
     */
    public void disableModule(String moduleName) {
        log.info("动态禁用模块: {}", moduleName);
        moduleStates.put(moduleName, false);
        
        // 可以在这里添加额外的禁用逻辑，如清理模块资源等
        log.debug("模块已禁用并更新状态: {}", moduleName);
    }
    
    /**
     * 获取模块状态信息
     */
    public ModuleStatus getModuleStatus(String moduleName) {
        // 返回模块的当前状态信息
        return new ModuleStatus(moduleName, isModuleEnabled(moduleName), System.currentTimeMillis());
    }
}

/**
 * 模块状态信息类
 */
class ModuleStatus {
    private final String moduleName;
    private final boolean enabled;
    private final long timestamp;
    
    public ModuleStatus(String moduleName, boolean enabled, long timestamp) {
        this.moduleName = moduleName;
        this.enabled = enabled;
        this.timestamp = timestamp;
    }
    
    // Getters
    public String getModuleName() { return moduleName; }
    public boolean isEnabled() { return enabled; }
    public long getTimestamp() { return timestamp; }
}