package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 任务调度模块
 * 实现智能任务调度策略，支持多种调度算法
 */
@Slf4j
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.scheduler", name = "enabled", havingValue = "true", matchIfMissing = false)
public class TaskSchedulerModule {

    @Autowired
    private ModuleProperties moduleProperties;
    
    @Autowired
    private MasterAgent masterAgent;
    
    @Autowired
    private AgentTaskStatusRepository taskStatusRepository;

    // 任务队列
    private final PriorityBlockingQueue<ScheduledTask> taskQueue = new PriorityBlockingQueue<>();
    private final AtomicInteger taskCounter = new AtomicInteger(0);

    /**
     * 监听任务创建事件，进行任务调度
     */
    @Async
    @EventListener
    public void handleTaskCreatedEvent(AgentTaskEvent event) {
        if ("TASK_CREATED".equals(event.getEventType())) {
            scheduleTask(event.getTaskId());
        }
    }

    /**
     * 调度任务
     */
    private void scheduleTask(String taskId) {
        String strategy = moduleProperties.getScheduler().getStrategy();
        log.debug("使用调度策略: {} 调度任务: {}", strategy, taskId);

        ScheduledTask task = new ScheduledTask(taskId, System.currentTimeMillis(), getNextPriority());

        switch (strategy) {
            case "priority":
                scheduleByPriority(task);
                break;
            case "round-robin":
                scheduleRoundRobin(task);
                break;
            case "weighted":
                scheduleWeighted(task);
                break;
            default:
                scheduleByPriority(task); // 默认策略
        }
    }

    /**
     * 优先级调度
     */
    private void scheduleByPriority(ScheduledTask task) {
        taskQueue.offer(task);
        log.debug("任务按优先级调度: {}", task.getTaskId());
    }

    /**
     * 轮询调度
     */
    private void scheduleRoundRobin(ScheduledTask task) {
        // TODO 简化实现：直接添加到队列
        taskQueue.offer(task);
        log.debug("任务按轮询调度: {}", task.getTaskId());
    }

    /**
     * 加权调度
     */
    private void scheduleWeighted(ScheduledTask task) {
        // 根据权重调整优先级
        int weight = calculateWeight(task.getTaskId());
        task.setPriority(task.getPriority() + weight);
        taskQueue.offer(task);
        log.debug("任务按权重调度: {}, 权重: {}", task.getTaskId(), weight);
    }

    /**
     * 计算任务权重
     */
    private int calculateWeight(String taskId) {
        // 根据任务特征计算权重（示例实现）
        return taskId.hashCode() % 10;
    }

    /**
     * 获取下一个优先级
     */
    private int getNextPriority() {
        return taskCounter.incrementAndGet();
    }

    /**
     * 执行下一个任务
     */
    public CompletableFuture<AgentResult> executeNextTask() {
        ScheduledTask task = taskQueue.poll();
        if (task != null) {
            log.debug("执行调度任务: {}", task.getTaskId());
            
            // 从数据库获取任务详情
            AgentTaskStatus taskStatus = taskStatusRepository.selectById(task.getTaskId());
            if (taskStatus == null) {
                log.warn("任务不存在: {}", task.getTaskId());
                return CompletableFuture.completedFuture(new AgentResult(false, "Task not found: " + task.getTaskId(), null));
            }
            
            // 构建主任务对象并执行
            MainTask mainTask = buildMainTaskFromStatus(taskStatus);
            
            // 通过MasterAgent异步执行主任务
            return CompletableFuture.supplyAsync(() -> {
                try {
                    AgentResult result = masterAgent.executeMainTask(mainTask);
                    
                    // 更新任务状态
                    taskStatus.setStatus(result.isSuccess() ? "SUCCESS" : "FAIL");
                    taskStatus.setResult(result.getData() != null ? result.getData().toString() : "");
                    taskStatus.setUpdateTime(java.time.LocalDateTime.now());
                    taskStatusRepository.updateById(taskStatus);
                    
                    log.info("任务执行完成: taskId={}, success={}", task.getTaskId(), result.isSuccess());
                    return result;
                } catch (Exception e) {
                    log.error("任务执行失败: taskId={}", task.getTaskId(), e);
                    
                    // 更新失败状态
                    taskStatus.setStatus("FAIL");
                    taskStatus.setResult("执行异常: " + e.getMessage());
                    taskStatus.setUpdateTime(java.time.LocalDateTime.now());
                    taskStatusRepository.updateById(taskStatus);
                    
                    return new AgentResult(false, "Task execution failed: " + e.getMessage(), null);
                }
            });
        }
        return CompletableFuture.completedFuture(new AgentResult(false, "No task to execute", null));
    }
    
    /**
     * 从任务状态构建主任务对象
     */
    private MainTask buildMainTaskFromStatus(AgentTaskStatus status) {
        MainTask mainTask = new MainTask();
        mainTask.setTaskId(status.getTaskId());
        
        // 从状态数据中重建任务信息
        // 这里需要根据实际存储的格式进行反序列化
        // 示例：如果result字段包含序列化的子任务信息，需要反序列化
        
        return mainTask;
    }

    /**
     * 获取当前队列大小
     */
    public int getQueueSize() {
        return taskQueue.size();
    }

    /**
     * 清空任务队列
     */
    public void clearQueue() {
        taskQueue.clear();
        log.debug("清空任务调度队列");
    }
}

/**
 * 调度任务类
 */
class ScheduledTask implements Comparable<ScheduledTask> {
    private final String taskId;
    private final long timestamp;
    private int priority;

    public ScheduledTask(String taskId, long timestamp, int priority) {
        this.taskId = taskId;
        this.timestamp = timestamp;
        this.priority = priority;
    }

    @Override
    public int compareTo(ScheduledTask other) {
        // 优先级高的排在前面（数值小的优先级高）
        return Integer.compare(this.priority, other.priority);
    }

    // Getters and setters
    public String getTaskId() { return taskId; }
    public long getTimestamp() { return timestamp; }
    public int getPriority() { return priority; }
    public void setPriority(int priority) { this.priority = priority; }
}