# 扩展模块架构设计文档

## 概述

本项目采用插拔式架构支持功能模块的插拔式管理与热更新，核心原则如下：

- 模块开关通过@ConditionalOnProperty注解控制
- 配置热更新由@RefreshScope支持，无需重启应用
- 所有模块遵循高内聚低耦合原则，不侵入核心逻辑
- 已实现日志审计、结果校验、任务可视化、降级兜底四大内置扩展模块，分别用于链路追溯、数据完整性校验、任务监控展示和故障降级处理

## 扩展模块列表

### 1. 日志审计模块 (LogModuleAutoConfig)
- **功能**：记录Agent全生命周期日志，支持按taskId追溯链路
- **配置项**：`agent.module.log.enabled=true`
- **实现方式**：基于Spring事件监听机制，监听AgentTaskEvent事件

### 2. 结果校验模块 (CheckModuleAutoConfig)
- **功能**：对比结果哈希值，校验数据传输和计算过程中的完整性
- **配置项**：`agent.module.check.enabled=true`
- **实现方式**：主Agent聚合结果后，对比实际哈希与预期哈希

### 3. 任务可视化模块 (MonitorModuleAutoConfig)
- **功能**：基于Spring Boot Admin生成任务依赖图，暴露任务执行进度监控端点
- **配置项**：`agent.module.monitor.enabled=true`
- **实现方式**：自定义@Endpoint端点，生成DOT格式依赖图

### 4. 降级兜底模块 (FallbackModuleAutoConfig)
- **功能**：从Agent执行失败时自动触发降级逻辑，支持自定义兜底策略
- **配置项**：`agent.module.fallback.enabled=true`
- **实现方式**：基于SlaveAgent接口的fallback默认方法

### 5. 缓存管理模块 (CacheManagementModule)
- **功能**：实现通用缓存机制集中管理与重构流程，支持多级缓存策略
- **配置项**：`agent.module.cache.enabled=true`
- **实现方式**：基于Caffeine实现多级缓存存储

### 6. 任务调度模块 (TaskSchedulerModule)
- **功能**：实现智能任务调度策略，支持多种调度算法（优先级、轮询、加权）
- **配置项**：`agent.module.scheduler.enabled=true`
- **实现方式**：基于优先级队列的任务调度器

### 7. 性能监控模块 (PerformanceMonitorModule)
- **功能**：收集和分析系统性能指标，提供执行时间、成功率等统计信息
- **配置项**：`agent.module.performance.enabled=true`
- **实现方式**：通过@Endpoint暴露性能指标端点

### 8. 安全审计模块 (SecurityAuditModule)
- **功能**：统一处理安全相关事件，记录任务访问和数据访问日志
- **配置项**：`agent.module.security.enabled=true`
- **实现方式**：基于事件监听的安全审计机制

## 核心架构组件

### 1. ModuleProperties
统一的模块配置属性类，支持所有扩展模块的配置管理：

```java
@Data
@Component
@RefreshScope
@ConfigurationProperties(prefix = "agent.module")
public class ModuleProperties {
    private boolean enabled = true;
    private LogModuleProperties log = new LogModuleProperties();
    private CheckModuleProperties check = new CheckModuleProperties();
    private MonitorModuleProperties monitor = new MonitorModuleProperties();
    private FallbackModuleProperties fallback = new FallbackModuleProperties();
    private CacheModuleProperties cache = new CacheModuleProperties();
    private TaskSchedulerModuleProperties scheduler = new TaskSchedulerModuleProperties();
    private PerformanceMonitorModuleProperties performance = new PerformanceMonitorModuleProperties();
    private SecurityAuditModuleProperties security = new SecurityAuditModuleProperties();
}
```

### 2. AgentTaskEvent
统一的事件机制，支持模块间松耦合通信：

```java
public class AgentTaskEvent extends ApplicationEvent {
    private final String taskId;
    private final String eventType; // TASK_CREATED, STARTED, COMPLETED, FAILED, SUBTASK_STARTED, SUBTASK_COMPLETED, SUBTASK_FAILED
    private final long timestamp;
}
```

### 3. 模块自动配置模式
所有模块遵循相同的自动配置模式：

```java
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.{moduleName}", name = "enabled", havingValue = "true", matchIfMissing = false)
public class {ModuleName}Module {
    // 模块实现逻辑
}
```

## 配置示例

```yaml
agent:
  module:
    enabled: true  # 模块总开关
    log:
      enabled: true
      retentionDays: 30
      storagePath: "./logs"
    check:
      enabled: true
      strictMode: false
      algorithm: "SHA-256"
    monitor:
      enabled: true
      refreshInterval: 5000
      enableGraph: true
    fallback:
      enabled: true
      maxRetries: 3
      retryDelay: 1000
      defaultStrategy: "cache"
    cache:
      enabled: true
      cacheType: caffeine
      maxSize: 1000
      taskCacheTtl: 300000
      resultCacheTtl: 600000
      configCacheTtl: 3600000
    scheduler:
      enabled: true
      strategy: priority
      maxConcurrentTasks: 10
      taskTimeout: 300000
    performance:
      enabled: true
      collectionInterval: 5000
      enableMetrics: true
      enableTracing: true
    security:
      enabled: true
      logEnabled: true
      logTaskAccess: true
      logDataAccess: true
```

## 热更新机制

通过Spring Cloud的@RefreshScope注解实现配置热更新：

1. 模块配置类标注@RefreshScope
2. 通过Spring Boot Actuator的/actuator/refresh端点触发配置刷新
3. 运行时动态调整模块开关和参数配置

## 架构优势

1. **插拔式设计**：模块可独立开启/关闭
2. **热更新能力**：配置修改无需重启应用
3. **事件驱动**：模块间松耦合通信
4. **统一配置管理**：集中管理所有模块配置
5. **高可扩展性**：易于添加新模块
6. **集中缓存管理**：通用缓存机制集中管理与重构流程
7. **智能调度**：支持多种任务调度策略
8. **全面监控**：性能和安全事件统一监控

## 扩展指南

### 添加新模块的步骤：

1. 创建模块配置属性类（继承ModuleProperties）
2. 创建模块实现类，使用@ConditionalOnProperty控制开关
3. 使用@RefreshScope支持热更新
4. 通过事件机制与其他模块通信
5. 在application.yml中添加配置项
6. 编写相应测试用例

该架构确保了系统的可扩展性、可维护性和灵活性，支持快速迭代和功能扩展。