package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 主Agent核心实现，自动扫描注册从Agent
 */
@Slf4j
@Component
public class MasterAgent implements ApplicationEventPublisherAware {
    /** 从Agent能力注册表，key=ToolTag值，value=对应的从Agent实例 */
    private final Map<String, SlaveAgent> agentMap = new HashMap<>();
    private final TaskDependencyResolver dependencyResolver;
    private AgentTaskStatusRepository taskStatusRepository;
    
    private ApplicationEventPublisher eventPublisher;

    /** 
     * 构造函数自动注入所有从Agent，并按ToolTag分组
     */
    @Autowired
    public MasterAgent(List<SlaveAgent> slaveAgents,
                       TaskDependencyResolver dependencyResolver,
                       @org.springframework.context.annotation.Lazy AgentTaskStatusRepository taskStatusRepository) {
        this.dependencyResolver = dependencyResolver;
        this.taskStatusRepository = taskStatusRepository;
        for (SlaveAgent agent : slaveAgents) {
            ToolTag tag = agent.getClass().getAnnotation(ToolTag.class);
            if (tag != null) {
                agentMap.put(tag.value(), agent);
                log.info("注册从Agent: {} -> {}", tag.value(), agent.getClass().getSimpleName());
            }
        }
    }
    
    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.eventPublisher = applicationEventPublisher;
    }

    public Map<String, SlaveAgent> getAgentMap() {
        return agentMap;
    }

    /**
     * 执行主任务
     */
    public AgentResult executeMainTask(MainTask mainTask) {
        try {
            // 发布任务创建事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, mainTask.getTaskId(), "TASK_CREATED"));
            }
            
            // 发布任务开始事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, mainTask.getTaskId(), "STARTED"));
            }
            
            // 1. 拆解主任务为子任务
            List<SubTask> subTasks = splitMainTask(mainTask);
            // 2. 解析子任务依赖，生成执行批次
            List<List<SubTask>> batches = dependencyResolver.resolve(subTasks);
            // 3. 按批次执行子任务：同批次并行，跨批次串行
            for (List<SubTask> batch : batches) {
                CompletableFuture[] futures = batch.stream()
                        .map(subTask -> CompletableFuture.runAsync(() -> executeSubTask(subTask)))
                        .toArray(CompletableFuture[]::new);
                CompletableFuture.allOf(futures).join();
            }
            // 4. 聚合子任务结果并校验完整性
            AgentResult finalResult = aggregateSubTaskResults(mainTask.getTaskId());
            validateResult(finalResult, mainTask.getExpectedHash());
            
            // 发布任务完成事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, mainTask.getTaskId(), "COMPLETED"));
            }
            
            return finalResult;
        } catch (Exception e) {
            log.error("执行主任务失败: {}", e.getMessage(), e);
            
            // 发布任务失败事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, mainTask.getTaskId(), "FAILED"));
            }
            
            return new AgentResult(false, "执行主任务失败: " + e.getMessage(), null);
        }
    }

    /**
     * 执行单个子任务
     */
    private void executeSubTask(SubTask subTask) {
        try {
            // 发布子任务开始事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, subTask.getSubTaskId(), "SUBTASK_STARTED"));
            }
            
            SlaveAgent agent = agentMap.get(subTask.getToolTag());
            if (agent == null) {
                throw new IllegalArgumentException("No agent found for tool tag: " + subTask.getToolTag());
            }
            AgentResult result = agent.execute(subTask);
            
            // 上报子任务结果
            reportSubTaskResult(subTask.getSubTaskId(), result);
            
            // 发布子任务完成事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, subTask.getSubTaskId(), "SUBTASK_COMPLETED"));
            }
        } catch (Exception e) {
            log.error("执行子任务失败: {}", e.getMessage(), e);
            
            // 发布子任务失败事件
            if (eventPublisher != null) {
                eventPublisher.publishEvent(new AgentTaskEvent(this, subTask.getSubTaskId(), "SUBTASK_FAILED"));
            }
        }
    }

    // 模拟拆分主任务为子任务
    private List<SubTask> splitMainTask(MainTask mainTask) {
        // 这里应该根据具体业务逻辑拆分主任务
        // 如果子任务列表为空，则根据任务内容动态生成
        if (mainTask.getSubTasks() == null || mainTask.getSubTasks().isEmpty()) {
            // 根据主任务内容动态创建子任务
            List<SubTask> generatedSubTasks = generateSubTasksFromMainTask(mainTask);
            mainTask.setSubTasks(generatedSubTasks);
        }
        
        return mainTask.getSubTasks();
    }
    
    /**
     * 根据主任务内容动态生成子任务
     */
    private List<SubTask> generateSubTasksFromMainTask(MainTask mainTask) {
        List<SubTask> subTasks = new java.util.ArrayList<>();
        
        // 示例：根据任务名称或内容创建子任务
        String taskName = mainTask.getTaskName();
        if (taskName != null) {
            // 这里可以根据任务名称或内容解析出具体的子任务
            // 示例：如果任务名称包含"query"，则创建数据库查询子任务
            if (taskName.toLowerCase().contains("query")) {
                SubTask queryTask = new SubTask();
                queryTask.setSubTaskId(java.util.UUID.randomUUID().toString());
                queryTask.setTaskId(mainTask.getTaskId());
                queryTask.setToolTag("DB_QUERY");
                queryTask.setParams(java.util.Map.of(
                    "queryType", "SELECT",
                    "tableName", "agent_task_status",
                    "conditions", java.util.Map.of("status", "READY")
                ));
                subTasks.add(queryTask);
            }
            
            // 示例：如果任务名称包含"analyze"，则创建分析子任务
            if (taskName.toLowerCase().contains("analyze")) {
                SubTask analyzeTask = new SubTask();
                analyzeTask.setSubTaskId(java.util.UUID.randomUUID().toString());
                analyzeTask.setTaskId(mainTask.getTaskId());
                analyzeTask.setToolTag("ANALYSIS"); // 假设有一个分析工具
                analyzeTask.setParams(java.util.Map.of(
                    "action", "analyze",
                    "target", taskName
                ));
                subTasks.add(analyzeTask);
            }
        }
        
        // 如果没有匹配到特定模式，创建一个默认子任务
        if (subTasks.isEmpty()) {
            SubTask defaultTask = new SubTask();
            defaultTask.setSubTaskId(java.util.UUID.randomUUID().toString());
            defaultTask.setTaskId(mainTask.getTaskId());
            defaultTask.setToolTag("DEFAULT");
            defaultTask.setParams(java.util.Map.of(
                "taskName", taskName,
                "taskId", mainTask.getTaskId()
            ));
            subTasks.add(defaultTask);
        }
        
        return subTasks;
    }

    // 模拟聚合子任务结果
    private AgentResult aggregateSubTaskResults(String taskId) {
        // 这里应该根据具体业务逻辑聚合结果
        return new AgentResult(true, "任务执行成功", "Aggregated result for task: " + taskId);
    }

    // 模拟结果校验
    private void validateResult(AgentResult result, String expectedHash) {
        // 根据具体业务逻辑校验结果
        if (expectedHash == null || expectedHash.isEmpty()) {
            log.info("无预期哈希值，跳过结果校验");
            return;
        }
        
        try {
            // 计算实际结果的哈希值
            String actualResultStr = result.getData() != null ? result.getData().toString() : "";
            String actualHash = HashCalculator.calculateHash(actualResultStr);
            
            // 比较哈希值
            if (actualHash.equals(expectedHash)) {
                log.info("任务结果校验通过: actualHash={}", actualHash);
            } else {
                log.warn("任务结果校验失败: expectedHash={}, actualHash={}", expectedHash, actualHash);
                
                // 根据模块配置决定是否严格模式（严格模式下校验失败抛出异常）
                // 这里可以获取模块配置，暂时记录警告
                result.setSuccess(false); // 标记结果为失败
                result.setMessage("结果校验失败: " + result.getMessage());
            }
        } catch (Exception e) {
            log.error("结果校验过程中发生异常: {}", e.getMessage(), e);
            result.setSuccess(false);
            result.setMessage("结果校验异常: " + e.getMessage());
        }
    }

    // 上报子任务结果
    private void reportSubTaskResult(String subTaskId, AgentResult result) {
        log.info("上报子任务结果: {} -> {}", subTaskId, result.isSuccess());
        
        try {
            // 创建或更新子任务状态记录
            AgentTaskStatus taskStatus = new AgentTaskStatus();
            taskStatus.setTaskId(subTaskId);
            taskStatus.setStatus(result.isSuccess() ? "SUCCESS" : "FAIL");
            taskStatus.setResult(result.getData() != null ? result.getData().toString() : "");
            taskStatus.setResultHash(result.getResultHash());
            taskStatus.setUpdateTime(java.time.LocalDateTime.now());
            
            // 检查是否存在，如果存在则更新，否则插入
            AgentTaskStatus existingStatus = taskStatusRepository.selectById(subTaskId);
            if (existingStatus != null) {
                taskStatusRepository.updateById(taskStatus);
            } else {
                taskStatusRepository.insert(taskStatus);
            }
            
            log.debug("子任务结果已上报: subTaskId={}", subTaskId);
            
        } catch (Exception e) {
            log.error("上报子任务结果失败: subTaskId={}", subTaskId, e);
        }
    }
}