package pangea.hiagent.agent.data;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Component;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 缓存管理模块
 * 实现通用缓存机制集中管理与重构流程
 * 支持多级缓存策略和动态配置
 */
@Slf4j
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.cache", name = "enabled", havingValue = "true", matchIfMissing = false)
public class CacheManagementModule {

    @Autowired
    private ModuleProperties moduleProperties;

    // 多级缓存存储
    private final ConcurrentHashMap<String, Cache<String, Object>> cacheStores = new ConcurrentHashMap<>();

    /**
     * 初始化缓存管理器
     */
    public void initializeCache() {
        log.info("初始化缓存管理模块");
        
        // 根据配置创建不同类型的缓存
        createCacheStore("taskCache", moduleProperties.getCache().getTaskCacheTtl());
        createCacheStore("resultCache", moduleProperties.getCache().getResultCacheTtl());
        createCacheStore("configCache", moduleProperties.getCache().getConfigCacheTtl());
        
        log.info("缓存管理模块初始化完成");
    }

    /**
     * 创建指定名称的缓存存储
     */
    private void createCacheStore(String cacheName, long ttl) {
        Cache<String, Object> cache = Caffeine.newBuilder()
                .maximumSize(moduleProperties.getCache().getMaxSize())
                .expireAfterWrite(ttl, TimeUnit.MILLISECONDS)
                .recordStats()
                .build();
        
        cacheStores.put(cacheName, cache);
        log.debug("创建缓存存储: {}, TTL: {}ms", cacheName, ttl);
    }

    /**
     * 获取缓存值
     */
    public Object get(String cacheName, String key) {
        Cache<String, Object> cache = cacheStores.get(cacheName);
        if (cache == null) {
            log.warn("缓存存储不存在: {}", cacheName);
            return null;
        }
        
        Object value = cache.getIfPresent(key);
        if (value != null) {
            log.debug("从缓存获取数据: {} -> {}", key, cacheName);
        }
        return value;
    }

    /**
     * 存储缓存值
     */
    public void put(String cacheName, String key, Object value) {
        Cache<String, Object> cache = cacheStores.get(cacheName);
        if (cache == null) {
            log.warn("缓存存储不存在: {}", cacheName);
            return;
        }
        
        cache.put(key, value);
        log.debug("存储数据到缓存: {} -> {}", key, cacheName);
    }

    /**
     * 删除缓存值
     */
    public void evict(String cacheName, String key) {
        Cache<String, Object> cache = cacheStores.get(cacheName);
        if (cache != null) {
            cache.invalidate(key);
            log.debug("删除缓存数据: {} -> {}", key, cacheName);
        }
    }

    /**
     * 清空指定缓存存储
     */
    public void clearCache(String cacheName) {
        Cache<String, Object> cache = cacheStores.get(cacheName);
        if (cache != null) {
            cache.invalidateAll();
            log.debug("清空缓存存储: {}", cacheName);
        }
    }

    /**
     * 获取缓存统计信息
     */
    public String getCacheStats(String cacheName) {
        Cache<String, Object> cache = cacheStores.get(cacheName);
        if (cache != null) {
            return cache.stats().toString();
        }
        return "Cache not found: " + cacheName;
    }

    /**
     * 获取所有缓存统计信息
     */
    public String getAllCacheStats() {
        StringBuilder stats = new StringBuilder();
        stats.append("=== 缓存统计信息 ===\n");
        
        for (String cacheName : cacheStores.keySet()) {
            stats.append(cacheName).append(": ").append(getCacheStats(cacheName)).append("\n");
        }
        
        return stats.toString();
    }
}

/**
 * 缓存管理模块自动配置
 */
@Slf4j
@Configuration
@EnableCaching
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.cache", name = "enabled", havingValue = "true", matchIfMissing = false)
class CacheManagementModuleConfig {

    @Bean
    public CacheManager cacheManager() {
        log.info("配置缓存管理器");
        return new org.springframework.cache.caffeine.CaffeineCacheManager();
    }
}