package pangea.hiagent.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import jakarta.servlet.http.HttpServletRequest;
import java.lang.InheritableThreadLocal;

/**
 * 用户相关工具类
 * 提供统一的用户信息获取方法，支持异步线程安全
 */
@Slf4j
@Component
public class UserUtils {

    private  volatile JwtUtil jwtUtil;

    // 使用InheritableThreadLocal存储用户ID，支持异步线程继承
    private  final InheritableThreadLocal<String> USER_ID_THREAD_LOCAL = new InheritableThreadLocal<>();

    // 静态Holder模式确保单例
    private static class Holder {
        private static UserUtils INSTANCE;
    }

    public UserUtils(JwtUtil jwtUtil) {
        this.jwtUtil = jwtUtil;
        Holder.INSTANCE = this;
    }

    /**
     * 设置当前线程的用户ID
     * @param userId 用户ID
     */
    public  void setCurrentUserId(String userId) {
        if (StringUtils.hasText(userId)) {
            USER_ID_THREAD_LOCAL.set(userId);
            log.debug("设置当前线程的用户ID: {}", userId);
        } else {
            USER_ID_THREAD_LOCAL.remove();
            log.debug("清除当前线程的用户ID");
        }
    }

    /**
     * 清除当前线程的用户ID
     */
    public  void clearCurrentUserId() {
        USER_ID_THREAD_LOCAL.remove();
        log.debug("清除当前线程的用户ID");
    }

    /**
     * 从ThreadLocal获取用户ID
     * @return 用户ID，如果不存在则返回null
     */
    public  String getCurrentUserIdFromThreadLocal() {
        String userId = USER_ID_THREAD_LOCAL.get();
        if (userId != null) {
            log.debug("从ThreadLocal获取到用户ID: {}", userId);
        }
        else{
            userId="user-001";
        }
        return userId;
    }

    /**
     * 获取当前认证用户ID
     * 优先从ThreadLocal获取，其次从SecurityContext获取，最后从请求中解析JWT
     * @return 用户ID，如果未认证则返回null
     */
    public  String getCurrentUserId() {
        // 优先从ThreadLocal获取（支持异步线程）
        String userId = getCurrentUserIdFromThreadLocal();
        if (userId != null) {
            return userId;
        }

        // 从SecurityContext获取
        userId = getCurrentUserIdFromSecurityContext();
        if (userId != null) {
            setCurrentUserId(userId);
            return userId;
        }

        // 从请求中解析JWT
        userId = getCurrentUserIdFromRequest();
        if (userId != null) {
            setCurrentUserId(userId);
        }

        return userId;
    }

    /**
     * 从SecurityContext获取当前认证用户ID
     * @return 用户ID，如果未认证则返回null
     */
    private  String getCurrentUserIdFromSecurityContext() {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null && authentication.isAuthenticated() && !"anonymousUser".equals(authentication.getPrincipal())) {
                Object principal = authentication.getPrincipal();
                if (principal instanceof String) {
                    String userId = (String) principal;
                    log.debug("从SecurityContext获取到用户ID: {}", userId);
                    return userId;
                } else {
                    // 尝试获取principal的字符串表示
                    log.debug("Authentication principal类型: {}", principal.getClass().getName());
                    String userId = principal.toString();
                    log.debug("将principal转换为字符串获取用户ID: {}", userId);
                    return userId;
                }
            }
        } catch (Exception e) {
            log.error("从SecurityContext获取用户ID时发生异常", e);
        }
        return null;
    }

    /**
     * 从当前请求中提取JWT令牌并解析用户ID
     * @return 用户ID，如果无法解析则返回null
     */
    private  String getCurrentUserIdFromRequest() {
        try {
            RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
            if (requestAttributes instanceof ServletRequestAttributes) {
                HttpServletRequest request = ((ServletRequestAttributes) requestAttributes).getRequest();
                String token = extractTokenFromRequest(request);

                if (StringUtils.hasText(token) && getJwtUtil() != null) {
                    boolean isValid = getJwtUtil().validateToken(token);
                    log.debug("JWT验证结果: {}", isValid);

                    if (isValid) {
                        String userId = getJwtUtil().getUserIdFromToken(token);
                        log.debug("从JWT令牌中提取用户ID: {}", userId);
                        return userId;
                    } else {
                        log.warn("JWT验证失败，token可能已过期或无效");
                    }
                } else {
                    if (getJwtUtil() == null) {
                        log.warn("jwtUtil未初始化");
                    } else {
                        log.debug("未找到有效的token");
                    }
                }
            } else {
                log.debug("无法获取请求上下文，可能在异步线程中调用");
            }
        } catch (Exception e) {
            log.error("从请求中解析用户ID时发生异常", e);
        }
        return null;
    }

    /**
     * 从请求头或参数中提取Token
     */
    private  String extractTokenFromRequest(HttpServletRequest request) {
        // 首先尝试从请求头中提取Token
        String authHeader = request.getHeader("Authorization");
        log.debug("从请求头中提取Authorization: {}", authHeader);
        if (StringUtils.hasText(authHeader) && authHeader.startsWith("Bearer ")) {
            String token = authHeader.substring(7);
            log.debug("从Authorization头中提取到token");
            return token;
        }

        // 如果请求头中没有Token，则尝试从URL参数中提取
        String tokenParam = request.getParameter("token");
        log.debug("从URL参数中提取token参数: {}", tokenParam);
        if (StringUtils.hasText(tokenParam)) {
            log.debug("从URL参数中提取到token");
            return tokenParam;
        }

        log.debug("未找到有效的token");
        return null;
    }

    /**
     * 获取JwtUtil实例，确保线程安全
     */
    private  JwtUtil getJwtUtil() {
        if (jwtUtil == null) {
            synchronized (UserUtils.class) {
                if (jwtUtil == null) {
                    log.error("jwtUtil尚未初始化，请确保UserUtils已被Spring容器正确管理");
                }
            }
        }
        return jwtUtil;
    }

    /**
     * 检查当前用户是否已认证
     * @return true表示已认证，false表示未认证
     */
    public  boolean isAuthenticated() {
        return getCurrentUserId() != null;
    }

    /**
     * 检查用户是否是管理员
     * @param userId 用户ID
     * @return true表示是管理员，false表示不是管理员
     */
    public  boolean isAdminUser(String userId) {
        // 根据实际需求实现管理员检查逻辑
        return "admin".equals(userId) || "user-001".equals(userId);
    }

    // 以下是静态方法，用于支持静态调用

    /**
     * 获取UserUtils单例实例
     */
    private static UserUtils getInstance() {
        UserUtils instance = Holder.INSTANCE;
        if (instance == null) {
            // 如果还没有初始化，返回默认实现
            log.warn("UserUtils实例尚未初始化，使用默认实现");
            instance = new UserUtils(null);
        }
        return instance;
    }

    /**
     * 静态方法：设置当前线程的用户ID
     */
    public static void setCurrentUserIdStatic(String userId) {
        getInstance().setCurrentUserId(userId);
    }

    /**
     * 静态方法：清除当前线程的用户ID
     */
    public static void clearCurrentUserIdStatic() {
        getInstance().clearCurrentUserId();
    }

    /**
     * 静态方法：从ThreadLocal获取用户ID
     */
    public static String getCurrentUserIdFromThreadLocalStatic() {
        return getInstance().getCurrentUserIdFromThreadLocal();
    }

    /**
     * 静态方法：获取当前认证用户ID
     */
    public static String getCurrentUserIdStatic() {
        return getInstance().getCurrentUserId();
    }

    /**
     * 静态方法：检查当前用户是否已认证
     */
    public static boolean isAuthenticatedStatic() {
        return getInstance().isAuthenticated();
    }

    /**
     * 静态方法：检查用户是否是管理员
     */
    public static boolean isAdminUserStatic(String userId) {
        return getInstance().isAdminUser(userId);
    }
}
