package pangea.hiagent.web.controller;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.model.OAuth2Provider;
import pangea.hiagent.model.User;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.dto.OAuth2ProviderRequest;
import pangea.hiagent.web.dto.PageData;
import pangea.hiagent.web.service.AuthService;
import pangea.hiagent.web.service.OAuth2ProviderService;
import pangea.hiagent.web.service.UnifiedAuthService;

import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.view.RedirectView;

    
import com.baomidou.mybatisplus.core.metadata.IPage;
import java.util.List;
import jakarta.validation.Valid;

/**
 * 认证API控制器
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/auth")
public class AuthController {
    
    private final AuthService authService;
    private final UnifiedAuthService unifiedAuthService;
    private final OAuth2ProviderService oAuth2ProviderService;
    
    public AuthController(AuthService authService, UnifiedAuthService unifiedAuthService, 
                         OAuth2ProviderService oAuth2ProviderService) {
        this.authService = authService;
        this.unifiedAuthService = unifiedAuthService;
        this.oAuth2ProviderService = oAuth2ProviderService;
    }
    
    /**
     * 用户注册
     */
    @PostMapping("/register")
    public ApiResponse<User> register(@RequestBody RegisterRequest request) {
        try {
            User user = authService.register(request.getUsername(), request.getPassword(), request.getEmail());
            return ApiResponse.success(user, "注册成功");
        } catch (Exception e) {
            log.error("用户注册失败", e);
            return ApiResponse.error(3001, "注册失败: " + e.getMessage());
        }
    }
    
    /**
     * 用户登录
     */
    @PostMapping("/login")
    public ApiResponse<LoginResponse> login(@RequestBody LoginRequest request) {
        try {
            String token = authService.login(request.getUsername(), request.getPassword());
            LoginResponse response = LoginResponse.builder()
                    .token(token)
                    .expiresIn(7200)
                    .tokenType("Bearer")
                    .build();
            return ApiResponse.success(response, "登录成功");
        } catch (Exception e) {
            log.error("用户登录失败: 错误堆栈: ", e);
            return ApiResponse.error(2001, "登录失败: " + e.getMessage());
        }
    }
    
    /**
     * OAuth2 授权端点
     * 重定向用户到第三方授权服务器
     */
    @GetMapping("/oauth2/authorize")
    public RedirectView oauth2Authorize(@RequestParam String providerName) {
        try {
            log.info("处理 OAuth2 授权请求: providerName={}", providerName);
            // 获取提供者配置并构造授权 URL
            // 这里需要获取已配置的 OAuth2 提供者信息
            String authorizationUrl = unifiedAuthService.buildOAuth2AuthorizationUrl(providerName);
            return new RedirectView(authorizationUrl);
        } catch (Exception e) {
            log.error("OAuth2 授权处理失败: providerName={}, 错误堆栈: ", providerName, e);
            return new RedirectView("/login?error=oauth2_auth_failed");
        }
    }
    
    /**
     * OAuth2 回调端点
     * 接收来自授权服务器的回调，获取授权码
     */
    @GetMapping("/oauth2/callback")
    public RedirectView oauth2Callback(@RequestParam String code, @RequestParam String state, @RequestParam String providerName) {
        try {
            log.info("处理 OAuth2 回调: providerName={}, code={}, state={}", providerName, 
                     code.substring(0, Math.min(20, code.length())) + "...", state);
            
            // 使用授权码进行认证
            String token = unifiedAuthService.loginWithOAuth2(code, providerName);
            
            // 重定向到前端并传递 token
            return new RedirectView("/login?token=" + token + "&method=oauth2");
            
        } catch (Exception e) {
            log.error("OAuth2 回调处理失败: providerName={}, code={}, 错误堆栈: ", providerName, code, e);
            return new RedirectView("/login?error=oauth2_callback_failed");
        }
    }
    
    /**
     * OAuth2 令牌交换端点
     * 后端调用这个端点交换授权码为访问令牌
     */
    @PostMapping("/oauth2/token")
    public ApiResponse<LoginResponse> oauth2Token(@RequestBody OAuth2TokenRequest request) {
        try {
            log.info("处理 OAuth2 令牌交换: providerName={}", request.getProviderName());
            
            String token = unifiedAuthService.loginWithOAuth2(request.getAuthorizationCode(), request.getProviderName());
            
            LoginResponse response = LoginResponse.builder()
                    .token(token)
                    .expiresIn(7200)
                    .tokenType("Bearer")
                    .build();
            
            return ApiResponse.success(response, "OAuth2 认证成功");
            
        } catch (Exception e) {
            log.error("OAuth2 令牌交换失败: providerName={}, 错误堆栈: ", request.getProviderName(), e);
            return ApiResponse.error(2002, "OAuth2 认证失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取OAuth2提供商配置列表（分页）
     */
    @GetMapping("/oauth2/providers")
    public ApiResponse<PageData<OAuth2Provider>> listOAuth2Providers(
            @RequestParam(defaultValue = "1") Long current,
            @RequestParam(defaultValue = "10") Long size,
            @RequestParam(required = false) String providerName,
            @RequestParam(required = false) String displayName,
            @RequestParam(required = false) Integer enabled) {
        try {
            log.info("获取OAuth2提供商配置列表: current={}, size={}, providerName={}, displayName={}, enabled={}", 
                    current, size, providerName, displayName, enabled);
            
            IPage<OAuth2Provider> page = oAuth2ProviderService.pageProviders(current, size, providerName, displayName, enabled);
            PageData<OAuth2Provider> pageData = PageData.from(page);
            
            return ApiResponse.success(pageData, "获取OAuth2提供商配置列表成功");
        } catch (Exception e) {
            log.error("获取OAuth2提供商配置列表失败: 错误堆栈: ", e);
            return ApiResponse.error(4001, "获取OAuth2提供商配置列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取所有OAuth2提供商配置（不分页）
     */
    @GetMapping("/oauth2/providers/all")
    public ApiResponse<List<OAuth2Provider>> getAllOAuth2Providers() {
        try {
            log.info("获取所有OAuth2提供商配置");
            
            List<OAuth2Provider> providers = oAuth2ProviderService.listProviders();
            
            return ApiResponse.success(providers, "获取所有OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("获取所有OAuth2提供商配置失败: 错误堆栈: ", e);
            return ApiResponse.error(4002, "获取所有OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 根据ID获取OAuth2提供商配置
     */
    @GetMapping("/oauth2/providers/{id}")
    public ApiResponse<OAuth2Provider> getOAuth2ProviderById(@PathVariable String id) {
        try {
            log.info("根据ID获取OAuth2提供商配置: id={}", id);
            
            OAuth2Provider provider = oAuth2ProviderService.getProviderById(id);
            if (provider == null) {
                log.warn("OAuth2提供商配置不存在: id={}", id);
                return ApiResponse.error(4003, "OAuth2提供商配置不存在");
            }
            
            return ApiResponse.success(provider, "获取OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("根据ID获取OAuth2提供商配置失败: id={}, 错误堆栈: ", id, e);
            return ApiResponse.error(4004, "获取OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 创建OAuth2提供商配置
     */
    @PostMapping("/oauth2/providers")
    public ApiResponse<OAuth2Provider> createOAuth2Provider(@Valid @RequestBody OAuth2ProviderRequest request) {
        try {
            log.info("创建OAuth2提供商配置: providerName={}", request.getProviderName());
            
            OAuth2Provider provider = new OAuth2Provider();
            provider.setProviderName(request.getProviderName());
            provider.setDisplayName(request.getDisplayName());
            provider.setDescription(request.getDescription());
            provider.setAuthType(request.getAuthType());
            provider.setAuthorizeUrl(request.getAuthorizeUrl());
            provider.setTokenUrl(request.getTokenUrl());
            provider.setUserinfoUrl(request.getUserinfoUrl());
            provider.setClientId(request.getClientId());
            provider.setClientSecret(request.getClientSecret());
            provider.setRedirectUri(request.getRedirectUri());
            provider.setScope(request.getScope());
            provider.setEnabled(request.getEnabled());
            provider.setConfigJson(request.getConfigJson());
            
            // 设置创建人和更新人信息
            String currentUserId = pangea.hiagent.common.utils.UserUtils.getCurrentUserIdStatic();
            if (currentUserId != null && !currentUserId.isEmpty()) {
                provider.setCreatedBy(currentUserId);
                provider.setUpdatedBy(currentUserId);
            } else {
                log.warn("无法获取当前用户ID，使用默认值");
                provider.setCreatedBy("system");
                provider.setUpdatedBy("system");
            }
            
            OAuth2Provider savedProvider = oAuth2ProviderService.createProvider(provider);
            
            return ApiResponse.success(savedProvider, "创建OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("创建OAuth2提供商配置失败: providerName={}, 错误堆栈: ", request.getProviderName(), e);
            return ApiResponse.error(4005, "创建OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新OAuth2提供商配置
     */
    @PutMapping("/oauth2/providers/{id}")
    public ApiResponse<OAuth2Provider> updateOAuth2Provider(@PathVariable String id, 
                                                           @Valid @RequestBody OAuth2ProviderRequest request) {
        try {
            log.info("更新OAuth2提供商配置: id={}", id);
            
            OAuth2Provider provider = new OAuth2Provider();
            provider.setProviderName(request.getProviderName());
            provider.setDisplayName(request.getDisplayName());
            provider.setDescription(request.getDescription());
            provider.setAuthType(request.getAuthType());
            provider.setAuthorizeUrl(request.getAuthorizeUrl());
            provider.setTokenUrl(request.getTokenUrl());
            provider.setUserinfoUrl(request.getUserinfoUrl());
            provider.setClientId(request.getClientId());
            provider.setClientSecret(request.getClientSecret());
            provider.setRedirectUri(request.getRedirectUri());
            provider.setScope(request.getScope());
            provider.setEnabled(request.getEnabled());
            provider.setConfigJson(request.getConfigJson());
            
            // 设置更新人信息
            String currentUserId = pangea.hiagent.common.utils.UserUtils.getCurrentUserIdStatic();
            if (currentUserId != null && !currentUserId.isEmpty()) {
                provider.setUpdatedBy(currentUserId);
            } else {
                log.warn("无法获取当前用户ID，使用默认值");
                provider.setUpdatedBy("system");
            }
            
            OAuth2Provider updatedProvider = oAuth2ProviderService.updateProvider(id, provider);
            
            return ApiResponse.success(updatedProvider, "更新OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("更新OAuth2提供商配置失败: id={}, 错误堆栈: ", id, e);
            return ApiResponse.error(4006, "更新OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 删除OAuth2提供商配置
     */
    @DeleteMapping("/oauth2/providers/{id}")
    public ApiResponse<Void> deleteOAuth2Provider(@PathVariable String id) {
        try {
            log.info("删除OAuth2提供商配置: id={}", id);
            
            oAuth2ProviderService.deleteProvider(id);
            
            return ApiResponse.success(null, "删除OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("删除OAuth2提供商配置失败: id={}, 错误堆栈: ", id, e);
            return ApiResponse.error(4007, "删除OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 启用OAuth2提供商配置
     */
    @PostMapping("/oauth2/providers/{id}/enable")
    public ApiResponse<Void> enableOAuth2Provider(@PathVariable String id) {
        try {
            log.info("启用OAuth2提供商配置: id={}", id);
            
            oAuth2ProviderService.enableProvider(id);
            
            return ApiResponse.success(null, "启用OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("启用OAuth2提供商配置失败: id={}, 错误堆栈: ", id, e);
            return ApiResponse.error(4008, "启用OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 禁用OAuth2提供商配置
     */
    @PostMapping("/oauth2/providers/{id}/disable")
    public ApiResponse<Void> disableOAuth2Provider(@PathVariable String id) {
        try {
            log.info("禁用OAuth2提供商配置: id={}", id);
            
            oAuth2ProviderService.disableProvider(id);
            
            return ApiResponse.success(null, "禁用OAuth2提供商配置成功");
        } catch (Exception e) {
            log.error("禁用OAuth2提供商配置失败: id={}, 错误堆栈: ", id, e);
            return ApiResponse.error(4009, "禁用OAuth2提供商配置失败: " + e.getMessage());
        }
    }
    
    /**
     * 注册请求DTO
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class RegisterRequest {
        private String username;
        private String password;
        private String email;
    }
    
    /**
     * 登录请求DTO
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class LoginRequest {
        private String username;
        private String password;
    }
    
    /**
     * 登录响应 DTO
     */
    @Data
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    public static class LoginResponse {
        private String token;
        private Integer expiresIn;
        private String tokenType;
    }
    
    /**
     * OAuth2 令牌交换请求 DTO
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class OAuth2TokenRequest {
        private String authorizationCode;
        private String providerName;
    }
}