package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

/**
 * 结果校验模块自动配置
 * 基于@ConditionalOnProperty注解实现模块开关控制
 * 结合@RefreshScope支持配置热更新
 */
@Slf4j
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.check", name = "enabled", havingValue = "true", matchIfMissing = false)
public class CheckModuleAutoConfig {
    
    @Autowired
    private ModuleProperties moduleProperties;
    
    @Autowired
    private AgentTaskStatusRepository taskStatusRepository;

    /**
     * 监听Agent任务完成事件，校验结果完整性
     */
    @Async
    @EventListener
    public void handleAgentTaskCompletedEvent(AgentTaskEvent event) {
        if ("COMPLETED".equals(event.getEventType())) {
            log.info("开始校验任务结果: taskId={}", event.getTaskId());
            
            // 检查模块是否启用严格模式
            if (moduleProperties.getCheck().isStrictMode()) {
                log.info("启用严格模式进行结果校验");
            }
            
            // TODO 这里可以实现结果校验逻辑
            // 比如获取任务的结果哈希并与预期哈希进行比较
            validateTaskResult(event.getTaskId());
        }
    }
    
    /**
     * 校验任务结果的完整性和正确性
     */
    private void validateTaskResult(String taskId) {
        try {
            // 获取任务的实际结果
            String actualResult = getActualResult(taskId);
            String expectedHash = getExpectedHash(taskId);
            
            // 检查预期哈希是否存在
            if (expectedHash == null || expectedHash.isEmpty()) {
                log.info("任务无预期哈希值，跳过校验: taskId={}", taskId);
                return;
            }
            
            // 计算实际结果的哈希值
            String actualHash = HashCalculator.calculateHash(actualResult);
            
            // 比较哈希值
            if (actualHash.equals(expectedHash)) {
                log.info("任务结果校验通过: taskId={}, hash={}", taskId, actualHash);
                
                // 更新任务状态为校验通过
                updateTaskValidationStatus(taskId, "PASSED");
            } else {
                log.warn("任务结果校验失败: taskId={}, actualHash={}, expectedHash={}", 
                         taskId, actualHash, expectedHash);
                
                // 更新任务状态为校验失败
                updateTaskValidationStatus(taskId, "FAILED");
                
                // 根据模块配置决定是否严格模式（严格模式下校验失败触发告警）
                if (moduleProperties.getCheck().isStrictMode()) {
                    log.error("严格模式下校验失败，触发告警: taskId={}", taskId);
                    handleValidationFailure(taskId, actualHash, expectedHash);
                } else {
                    log.warn("宽松模式下校验失败，仅记录日志: taskId={}", taskId);
                }
            }
        } catch (Exception e) {
            log.error("任务结果校验异常: taskId={}, error={}", taskId, e.getMessage(), e);
            
            // 更新任务状态为校验异常
            updateTaskValidationStatus(taskId, "ERROR");
        }
    }
    
    /**
     * 更新任务校验状态
     */
    private void updateTaskValidationStatus(String taskId, String validationStatus) {
        try {
            log.info("任务校验状态更新: taskId={}, validationStatus={}", taskId, validationStatus);
            
            // 更新数据库中的校验状态字段
            AgentTaskStatus status = taskStatusRepository.selectById(taskId);
            if (status != null) {
                status.setValidationStatus(validationStatus);
                taskStatusRepository.updateById(status);
                log.debug("任务校验状态已更新到数据库: taskId={}, validationStatus={}", taskId, validationStatus);
            } else {
                log.warn("无法找到任务进行校验状态更新: taskId={}", taskId);
            }
        } catch (Exception e) {
            log.error("更新任务校验状态失败: taskId={}, status={}", taskId, validationStatus, e);
        }
    }
    
    /**
     * 获取任务的实际结果（从数据库获取）
     */
    private String getActualResult(String taskId) {
        // 从数据库获取任务的实际结果
        AgentTaskStatus taskStatus = taskStatusRepository.selectById(taskId);
        if (taskStatus != null) {
            return taskStatus.getResult() != null ? taskStatus.getResult() : "";
        }
        
        // 如果数据库中没有找到，返回空字符串
        log.warn("无法获取任务的实际结果: taskId={}", taskId);
        return "";
    }
    
    /**
     * 获取任务的预期哈希值（从数据库获取）
     */
    private String getExpectedHash(String taskId) {
        // 从数据库获取任务的预期哈希值
        // 通常预期哈希值应该在任务创建时设置并存储在resultHash字段中
        AgentTaskStatus taskStatus = taskStatusRepository.selectById(taskId);
        if (taskStatus != null && taskStatus.getResultHash() != null) {
            return taskStatus.getResultHash();
        }
        
        // 如果数据库中没有找到预期哈希值，返回null
        log.warn("无法获取任务的预期哈希值: taskId={}", taskId);
        return null;
    }
    
    /**
     * 处理校验失败的情况
     */
    private void handleValidationFailure(String taskId, String actualHash, String expectedHash) {
        log.warn("任务结果校验失败处理: taskId={}, actualHash={}, expectedHash={}", taskId, actualHash, expectedHash);
        
        // 实现具体的失败处理逻辑
        // 1. 发送告警通知
        sendAlertNotification(taskId, actualHash, expectedHash);
        
        // 2. 根据配置决定是否进行重试
        if (moduleProperties.getCheck().isRetryOnFailure()) {
            log.info("配置为校验失败时重试，任务ID: {}", taskId);
            // 这里可以实现重试逻辑，例如重新调度任务
            // TODO: 可以根据需要实现具体的重试机制
        }
        
        // 3. 记录失败详情到日志或监控系统
        logValidationFailureDetails(taskId, actualHash, expectedHash);
    }
    
    /**
     * 发送告警通知
     */
    private void sendAlertNotification(String taskId, String actualHash, String expectedHash) {
        log.error("【严重告警】任务校验失败: taskId={}, actualHash={}, expectedHash={}", 
                 taskId, actualHash, expectedHash);
        
        // 在实际应用中，这里可以集成告警系统，如发送邮件、短信或调用告警API
        // 例如: alertService.sendAlert("任务校验失败", "任务ID: " + taskId + "校验失败...");
    }
    
    /**
     * 记录校验失败详情
     */
    private void logValidationFailureDetails(String taskId, String actualHash, String expectedHash) {
        // 记录详细的失败信息，用于后续分析
        log.info("校验失败详情 - 任务ID: {}, 实际哈希: {}, 预期哈希: {}, 差异: {}", 
                 taskId, actualHash, expectedHash, 
                 !actualHash.equals(expectedHash));
    }
}