package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 简单的命令路由实现
 */
@Slf4j
@Component
public class SimpleCommandRouter implements CommandRouter {
    
    @Autowired
    private MasterAgent masterAgent;
    
    @Autowired
    private AgentTaskStatusRepository taskStatusRepository;
    
    @Override
    public void route(CooperateCommand command) {
        log.info("路由协同命令: messageId={}, taskId={}, action={}, sender={}, receiver={}", 
                command.getMessageId(), command.getTaskId(), command.getAction(), 
                command.getSenderAgentId(), command.getReceiverAgentId());
        
        // 根据动作类型进行路由处理
        switch (command.getAction()) {
            case "TASK_ASSIGN":
                handleTaskAssign(command);
                break;
            case "RESULT_REPORT":
                handleResultReport(command);
                break;
            case "FAIL_RETRY":
                handleFailRetry(command);
                break;
            default:
                log.warn("未知的命令动作类型: {}", command.getAction());
                break;
        }
    }
    
    private void handleTaskAssign(CooperateCommand command) {
        log.info("处理任务分配命令: taskId={}, receiver={}", command.getTaskId(), command.getReceiverAgentId());
        
        try {
            // 从命令参数中获取主任务信息
            String taskId = command.getTaskId();
            
            // 从数据库获取任务详情
            AgentTaskStatus taskStatus = taskStatusRepository.selectById(taskId);
            if (taskStatus == null) {
                log.error("任务不存在: taskId={}", taskId);
                return;
            }
            
            // 更新任务状态为RUNNING
            taskStatus.setStatus("RUNNING");
            taskStatus.setUpdateTime(java.time.LocalDateTime.now());
            taskStatusRepository.updateById(taskStatus);
            
            // 构建主任务对象并执行
            MainTask mainTask = buildMainTaskFromStatus(taskStatus);
            
            // 通过MasterAgent执行主任务
            AgentResult result = masterAgent.executeMainTask(mainTask);
            
            log.info("任务分配执行完成: taskId={}, success={}", taskId, result.isSuccess());
            
            // 更新任务状态
            taskStatus.setStatus(result.isSuccess() ? "SUCCESS" : "FAIL");
            taskStatus.setResult(result.getData() != null ? result.getData().toString() : "");
            taskStatus.setUpdateTime(java.time.LocalDateTime.now());
            taskStatusRepository.updateById(taskStatus);
            
        } catch (Exception e) {
            log.error("处理任务分配命令失败: taskId={}", command.getTaskId(), e);
        }
    }
    
    /**
     * 从任务状态构建主任务对象
     */
    private MainTask buildMainTaskFromStatus(AgentTaskStatus status) {
        MainTask mainTask = new MainTask();
        mainTask.setTaskId(status.getTaskId());
        
        // 这里需要从状态的result字段反序列化出原始任务信息
        // 实际实现中可能需要更复杂的反序列化逻辑
        mainTask.setSubTasks(List.of()); // 实际实现需要从状态中重建子任务
        
        return mainTask;
    }
    
    private void handleResultReport(CooperateCommand command) {
        log.info("处理结果报告命令: taskId={}, receiver={}", command.getTaskId(), command.getReceiverAgentId());
        
        try {
            String taskId = command.getTaskId();
            String subTaskId = (String) command.getParams().get("subTaskId");
            Object resultData = command.getParams().get("result");
            String resultHash = command.getResultHash();
            
            // 更新子任务状态到数据库
            AgentTaskStatus subTaskStatus = new AgentTaskStatus();
            subTaskStatus.setTaskId(subTaskId);
            subTaskStatus.setStatus("SUCCESS");
            subTaskStatus.setResult(resultData != null ? resultData.toString() : "");
            subTaskStatus.setResultHash(resultHash);
            subTaskStatus.setUpdateTime(java.time.LocalDateTime.now());
            
            // 检查是否存在，如果存在则更新，否则插入
            AgentTaskStatus existingStatus = taskStatusRepository.selectById(subTaskId);
            if (existingStatus != null) {
                taskStatusRepository.updateById(subTaskStatus);
            } else {
                taskStatusRepository.insert(subTaskStatus);
            }
            
            log.info("子任务结果报告处理完成: subTaskId={}, resultHash={}", subTaskId, resultHash);
            
        } catch (Exception e) {
            log.error("处理结果报告命令失败: taskId={}", command.getTaskId(), e);
        }
    }
    
    private void handleFailRetry(CooperateCommand command) {
        log.info("处理失败重试命令: taskId={}, receiver={}", command.getTaskId(), command.getReceiverAgentId());
        
        try {
            String taskId = command.getTaskId();
            
            // 获取任务状态
            AgentTaskStatus taskStatus = taskStatusRepository.selectById(taskId);
            if (taskStatus == null) {
                log.error("重试任务不存在: taskId={}", taskId);
                return;
            }
            
            // 检查重试次数是否超过限制
            Integer retryCount = taskStatus.getRetryCount();
            if (retryCount == null) {
                retryCount = 0;
            }
            
            if (retryCount >= 3) { // 最多重试3次
                log.warn("任务重试次数已达到上限: taskId={}, retryCount={}", taskId, retryCount);
                taskStatus.setStatus("FAIL");
                taskStatus.setUpdateTime(java.time.LocalDateTime.now());
                taskStatusRepository.updateById(taskStatus);
                return;
            }
            
            // 更新重试次数
            retryCount++;
            taskStatus.setRetryCount(retryCount);
            taskStatus.setStatus("READY"); // 重置状态为准备执行
            taskStatus.setUpdateTime(java.time.LocalDateTime.now());
            
            // 保存更新后的状态
            taskStatusRepository.updateById(taskStatus);
            
            // 重新执行任务
            MainTask mainTask = buildMainTaskFromStatus(taskStatus);
            AgentResult result = masterAgent.executeMainTask(mainTask);
            
            log.info("任务重试执行完成: taskId={}, success={}, retryCount={}", 
                     taskId, result.isSuccess(), retryCount);
            
            // 更新最终状态
            taskStatus.setStatus(result.isSuccess() ? "SUCCESS" : "FAIL");
            taskStatus.setResult(result.getData() != null ? result.getData().toString() : "");
            taskStatus.setUpdateTime(java.time.LocalDateTime.now());
            taskStatusRepository.updateById(taskStatus);
            
        } catch (Exception e) {
            log.error("处理失败重试命令失败: taskId={}", command.getTaskId(), e);
        }
    }
}