package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 性能监控模块
 * 收集和分析系统性能指标
 */
@Slf4j
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.performance", name = "enabled", havingValue = "true", matchIfMissing = false)
@Endpoint(id = "agent-performance")
public class PerformanceMonitorModule {

    @Autowired
    private ModuleProperties moduleProperties;

    // 性能指标存储
    private final Map<String, PerformanceMetric> metrics = new ConcurrentHashMap<>();
    private final AtomicLong totalTasks = new AtomicLong(0);
    private final AtomicLong completedTasks = new AtomicLong(0);
    private final AtomicLong failedTasks = new AtomicLong(0);
    private final AtomicLong totalExecutionTime = new AtomicLong(0);

    /**
     * 监听任务开始事件
     */
    @Async
    @EventListener
    public void handleTaskStartedEvent(AgentTaskEvent event) {
        if ("STARTED".equals(event.getEventType()) || "SUBTASK_STARTED".equals(event.getEventType())) {
            PerformanceMetric metric = new PerformanceMetric(event.getTaskId(), System.currentTimeMillis());
            metrics.put(event.getTaskId(), metric);
            totalTasks.incrementAndGet();
            log.debug("开始监控任务性能: {}", event.getTaskId());
        }
    }

    /**
     * 监听任务完成事件
     */
    @Async
    @EventListener
    public void handleTaskCompletedEvent(AgentTaskEvent event) {
        if ("COMPLETED".equals(event.getEventType()) || "SUBTASK_COMPLETED".equals(event.getEventType())) {
            PerformanceMetric metric = metrics.get(event.getTaskId());
            if (metric != null) {
                long endTime = System.currentTimeMillis();
                long executionTime = endTime - metric.getStartTime();
                metric.setEndTime(endTime);
                metric.setExecutionTime(executionTime);
                completedTasks.incrementAndGet();
                totalExecutionTime.addAndGet(executionTime);
                
                log.debug("任务执行完成: {}, 执行时间: {}ms", event.getTaskId(), executionTime);
            }
        }
    }

    /**
     * 监听任务失败事件
     */
    @Async
    @EventListener
    public void handleTaskFailedEvent(AgentTaskEvent event) {
        if ("FAILED".equals(event.getEventType()) || "SUBTASK_FAILED".equals(event.getEventType())) {
            PerformanceMetric metric = metrics.get(event.getTaskId());
            if (metric != null) {
                long endTime = System.currentTimeMillis();
                long executionTime = endTime - metric.getStartTime();
                metric.setEndTime(endTime);
                metric.setExecutionTime(executionTime);
                metric.setSuccess(false);
                failedTasks.incrementAndGet();
                
                log.debug("任务执行失败: {}, 执行时间: {}ms", event.getTaskId(), executionTime);
            }
        }
    }

    /**
     * 获取性能监控指标
     */
    @ReadOperation
    public Map<String, Object> getPerformanceMetrics() {
        Map<String, Object> result = new java.util.HashMap<>();
        
        try {
            // 计算平均执行时间
            long totalCompleted = completedTasks.get();
            long avgExecutionTime = totalCompleted > 0 ? totalExecutionTime.get() / totalCompleted : 0;
            
            result.put("totalTasks", totalTasks.get());
            result.put("completedTasks", completedTasks.get());
            result.put("failedTasks", failedTasks.get());
            result.put("successRate", totalCompleted > 0 ? (double) completedTasks.get() / totalCompleted * 100 : 0);
            result.put("avgExecutionTime", avgExecutionTime);
            result.put("activeTasks", metrics.size());
            result.put("timestamp", System.currentTimeMillis());
            result.put("status", "success");
        } catch (Exception e) {
            log.error("获取性能指标失败", e);
            result.put("status", "error");
            result.put("error", e.getMessage());
        }
        
        return result;
    }

    /**
     * 清空性能指标
     */
    public void clearMetrics() {
        metrics.clear();
        totalTasks.set(0);
        completedTasks.set(0);
        failedTasks.set(0);
        totalExecutionTime.set(0);
        log.debug("清空性能监控指标");
    }

    /**
     * 获取任务执行时间统计
     */
    public long getAverageExecutionTime() {
        long totalCompleted = completedTasks.get();
        return totalCompleted > 0 ? totalExecutionTime.get() / totalCompleted : 0;
    }

    /**
     * 获取任务成功率
     */
    public double getSuccessRate() {
        long total = totalTasks.get();
        return total > 0 ? (double) completedTasks.get() / total * 100 : 0;
    }
}

/**
 * 性能指标类
 */
class PerformanceMetric {
    private final String taskId;
    private final long startTime;
    private long endTime;
    private long executionTime;
    private boolean success = true;

    public PerformanceMetric(String taskId, long startTime) {
        this.taskId = taskId;
        this.startTime = startTime;
    }

    // Getters and setters
    public String getTaskId() { return taskId; }
    public long getStartTime() { return startTime; }
    public long getEndTime() { return endTime; }
    public void setEndTime(long endTime) { this.endTime = endTime; }
    public long getExecutionTime() { return executionTime; }
    public void setExecutionTime(long executionTime) { this.executionTime = executionTime; }
    public boolean isSuccess() { return success; }
    public void setSuccess(boolean success) { this.success = success; }
}