# 轻量化高可靠多Agent协同方案（Spring AI + ReAct）
**文档用途**：供AI解析并生成生产级多Agent协同代码，兼顾轻量化部署、高可靠性、功能完整性三大核心目标
**适用场景**：单机/小规模跨实例多Agent协同，基于Spring AI ReAct框架构建智能体系统
**技术栈**：Spring AI ReAct、Spring Cloud Stream、H2嵌入式数据库、Resilience4j、Spring Boot Actuator

**实现状态**：✅ 已完成核心组件开发

## 一、 核心架构设计
1.  **架构模式**：单Spring Boot应用+模块化隔离，主从Agent职责固化+动态能力链编排，无中心节点依赖
2.  **核心原则**：本地状态持久化+可靠消息投递+重试幂等+故障自愈，避免复杂中间件依赖
3.  **部署形态**：单Jar包部署，支持通过配置切换主/从/混合模式，资源占用可控（默认JVM堆内存512M）

## 二、 通信层设计（双模可靠通信）
### 2.1 通信模式选型
| 模式       | 适用场景               | 实现方案                                                                 | 可靠性保障措施                             |
|------------|------------------------|--------------------------------------------------------------------------|--------------------------------------------|
| 内存队列   | 单机单应用部署         | 基于`ConcurrentLinkedQueue`实现进程内Agent通信                           | 消息ID全局唯一，接收方幂等去重             |
| 远程MQ     | 跨实例分布式部署       | Spring Cloud Stream + RabbitMQ，开启`publisher-confirm-type: correlated` | 消息发布确认+返回机制+死信队列兜底         |

### 2.2 标准化消息结构定义 ✅ 已实现
```java
import lombok.Data;
import java.util.Map;
import java.util.UUID;

@Data
public class CooperateCommand {
    /** 全局唯一消息ID，用于幂等校验 */
    private String messageId;
    /** 关联主任务ID */
    private String taskId;
    /** 发送方Agent ID */
    private String senderAgentId;
    /** 接收方Agent ID */
    private String receiverAgentId;
    /** 动作类型：TASK_ASSIGN/RESULT_REPORT/FAIL_RETRY */
    private String action;
    /** 业务参数体 */
    private Map<String, Object> params;
    /** 结果校验哈希值 */
    private String resultHash;

    public CooperateCommand() {
        this.messageId = UUID.randomUUID().toString();
    }
}
```

**实现位置**：`pangea.hiagent.agent.data.CooperateCommand`

### 2.3 双模消息总线核心实现 ✅ 已实现
```java
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * 双模消息总线核心实现
 */
@Component
public class DualModeMessageBus {
    private final ConcurrentLinkedQueue<CooperateCommand> localQueue = new ConcurrentLinkedQueue<>();
    private final IdempotentChecker idempotentChecker;
    private final CommandRouter commandRouter;

    /** 是否启用远程MQ通信，通过配置文件控制 */
    private final boolean useRemote;

    public DualModeMessageBus(IdempotentChecker idempotentChecker,
                              CommandRouter commandRouter,
                              @Value("${agent.comm.use-remote:false}") boolean useRemote) {
        this.idempotentChecker = idempotentChecker;
        this.commandRouter = commandRouter;
        this.useRemote = useRemote;
    }

    /** 发送协同指令 */
    public void send(CooperateCommand command) {
        // 如果启用远程通信，这里需要通过其他方式发送（如HTTP或WebSocket）
        // 为简化实现，当前只使用本地队列
        localQueue.offer(command);
    }

    /** 消费本地队列消息，定时轮询 */
    @Scheduled(fixedRate = 100)
    public void consumeLocalQueue() {
        while (!localQueue.isEmpty()) {
            CooperateCommand command = localQueue.poll();
            if (command != null) {
                handleCommand(command);
            }
        }
    }

    /** 消息处理核心逻辑，包含幂等校验 */
    private void handleCommand(CooperateCommand command) {
        if (!idempotentChecker.check(command.getMessageId())) {
            return;
        }
        commandRouter.route(command);
    }
}

// 幂等校验器接口
public interface IdempotentChecker {
    boolean check(String messageId);
}

// 指令路由接口
public interface CommandRouter {
    void route(CooperateCommand command);
}
```

**实现位置**：
- `pangea.hiagent.agent.data.DualModeMessageBus`
- `pangea.hiagent.agent.data.SimpleIdempotentChecker`
- `pangea.hiagent.agent.data.SimpleCommandRouter`

## 三、 协同核心设计（主从分工+动态能力链）
### 3.1 主从Agent角色与职责边界
| 角色       | 核心职责                                                                 | 严格禁止操作                     | 核心能力支撑                               |
|------------|--------------------------------------------------------------------------|----------------------------------|--------------------------------------------|
| 主Agent    | 1. 接收上层任务并拆解为子任务<br>2. 解析子任务依赖关系，生成执行批次<br>3. 并行/串行执行子任务<br>4. 聚合子任务结果并校验<br>5. 故障兜底（重试/降级/熔断） | 直接执行具体工具调用逻辑         | 依赖解析、灰度执行、结果聚合、故障监控     |
| 从Agent    | 1. 通过`@ToolTag`注册能力标识<br>2. 执行单一类型工具任务<br>3. 自动重试+结果哈希生成<br>4. 上报任务执行状态 | 任务拆解、依赖管理、结果聚合     | 工具调用、幂等执行、失败重试、降级兜底     |

### 3.2 能力标签注解与自动注册机制
```java
import org.springframework.stereotype.Component;
import java.lang.annotation.*;

/**
 * 从Agent能力标签注解，用于标识Agent的工具能力
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Documented
@Component
public @interface ToolTag {
    /** 能力标识，如DB_QUERY/DATA_CALC/FORMAT_EXPORT */
    String value();
}

/**
 * 从Agent通用接口
 */
public interface SlaveAgent {
    AgentResult execute(SubTask subTask);
}

/**
 * 从Agent实现示例 - 数据库查询Agent
 */
@ToolTag("DB_QUERY")
public class DbQuerySlaveAgent implements SlaveAgent {
    @Override
    public AgentResult execute(SubTask subTask) {
        Map<String, Object> result = DbUtils.query(subTask.getParams());
        AgentResult agentResult = new AgentResult();
        agentResult.setData(result);
        agentResult.setSuccess(true);
        agentResult.setResultHash(MD5Util.calculate(result));
        return agentResult;
    }
}

/**
 * 主Agent核心实现，自动扫描注册从Agent
 */
@Component
public class MasterAgent {
    /** 从Agent能力注册表，key=ToolTag值，value=对应的从Agent实例 */
    private final Map<String, SlaveAgent> agentMap = new HashMap<>();
    private final TaskDependencyResolver dependencyResolver;

    /**
     * 构造函数自动注入所有从Agent，并按ToolTag分组
     */
    public MasterAgent(List<SlaveAgent> slaveAgents,
                       TaskDependencyResolver dependencyResolver) {
        this.dependencyResolver = dependencyResolver;
        for (SlaveAgent agent : slaveAgents) {
            ToolTag tag = agent.getClass().getAnnotation(ToolTag.class);
            if (tag != null) {
                agentMap.put(tag.value(), agent);
            }
        }
    }
}
```

**实现位置**：
- `pangea.hiagent.agent.data.ToolTag`
- `pangea.hiagent.agent.data.SlaveAgent`
- `pangea.hiagent.agent.data.MasterAgent`
- `pangea.hiagent.agent.data.DbQuerySlaveAgent`

### 3.3 子任务依赖解析与批次执行
```java
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Component
public class TaskDependencyResolver {
    /**
     * 解析子任务依赖关系，生成执行批次
     * 同批次任务可并行执行，跨批次任务串行执行
     */
    public List<List<SubTask>> resolve(List<SubTask> subTasks) {
        List<List<SubTask>> batches = new ArrayList<>();
        Set<String> completedTaskIds = new HashSet<>();

        // 初始批次：无依赖的子任务
        List<SubTask> firstBatch = subTasks.stream()
                .filter(t -> t.getDependOn().isEmpty())
                .collect(Collectors.toList());
        batches.add(firstBatch);
        completedTaskIds.addAll(firstBatch.stream().map(SubTask::getSubTaskId).collect(Collectors.toList()));

        // 递归解析后续批次
        while (completedTaskIds.size() < subTasks.size()) {
            List<SubTask> nextBatch = subTasks.stream()
                    .filter(t -> !completedTaskIds.contains(t.getSubTaskId()))
                    .filter(t -> completedTaskIds.containsAll(t.getDependOn()))
                    .collect(Collectors.toList());
            if (nextBatch.isEmpty()) {
                throw new IllegalStateException("Circular dependency detected in subtasks");
            }
            batches.add(nextBatch);
            completedTaskIds.addAll(nextBatch.stream().map(SubTask::getSubTaskId).collect(Collectors.toList()));
        }
        return batches;
    }
}
```

**实现位置**：`pangea.hiagent.agent.data.TaskDependencyResolver`

/**
 * 主Agent任务执行核心逻辑
 */
public class MasterAgent {
    // ... 省略已有代码 ...

    /**
     * 执行主任务
     */
    public AgentResult executeMainTask(MainTask mainTask) {
        // 1. 拆解主任务为子任务
        List<SubTask> subTasks = splitMainTask(mainTask);
        // 2. 解析子任务依赖，生成执行批次
        List<List<SubTask>> batches = dependencyResolver.resolve(subTasks);
        // 3. 按批次执行子任务：同批次并行，跨批次串行
        for (List<SubTask> batch : batches) {
            CompletableFuture[] futures = batch.stream()
                    .map(subTask -> CompletableFuture.runAsync(() -> executeSubTask(subTask)))
                    .toArray(CompletableFuture[]::new);
            CompletableFuture.allOf(futures).join();
        }
        // 4. 聚合子任务结果并校验完整性
        AgentResult finalResult = aggregateSubTaskResults(mainTask.getTaskId());
        validateResult(finalResult, mainTask.getExpectedHash());
        return finalResult;
    }

    /**
     * 执行单个子任务
     */
    private void executeSubTask(SubTask subTask) {
        SlaveAgent agent = agentMap.get(subTask.getToolTag());
        if (agent == null) {
            throw new IllegalArgumentException("No agent found for tool tag: " + subTask.getToolTag());
        }
        AgentResult result = agent.execute(subTask);
        // 上报子任务结果
        reportSubTaskResult(subTask.getSubTaskId(), result);
    }

    // ... 省略结果聚合、校验、上报等辅助方法 ...
}
```

### 3.4 动态能力链编排与灰度执行
```java
/**
 * 动态能力链定义
 */
public class AgentChain {
    private final List<SlaveAgent> agentList = new ArrayList<>();

    public void addAgent(SlaveAgent agent) {
        agentList.add(agent);
    }

    /**
     * 链式执行：前一个Agent的输出作为下一个Agent的输入
     */
    public AgentResult execute(SubTask initialTask) {
        AgentResult currentResult = new AgentResult();
        currentResult.setData(initialTask.getParams());
        currentResult.setSuccess(true);

        for (SlaveAgent agent : agentList) {
            if (!currentResult.isSuccess()) {
                throw new AgentExecuteException("Chain execution stopped due to previous failure");
            }
            // 构造下一个Agent的输入任务
            SubTask nextTask = new SubTask();
            nextTask.setParams(currentResult.getData());
            nextTask.setToolTag(getToolTag(agent));
            // 执行当前Agent
            currentResult = agent.execute(nextTask);
        }
        return currentResult;
    }

    /**
     * 获取Agent的ToolTag值
     */
    private String getToolTag(SlaveAgent agent) {
        return agent.getClass().getAnnotation(ToolTag.class).value();
    }
}

/**
 * 能力链构建器
 */
@Component
public class AgentChainBuilder {
    private final Map<String, SlaveAgent> agentMap;

    public AgentChainBuilder(MasterAgent masterAgent) {
        this.agentMap = masterAgent.getAgentMap();
    }

    /**
     * 根据工具标签列表构建能力链
     */
    public AgentChain buildChain(List<String> toolTags) {
        AgentChain chain = new AgentChain();
        for (String tag : toolTags) {
            SlaveAgent agent = agentMap.get(tag);
            if (agent == null) {
                throw new IllegalArgumentException("Unknown tool tag: " + tag);
            }
            chain.addAgent(agent);
        }
        return chain;
    }
}

/**
 * 灰度执行器，支持按比例/白名单灰度发布新能力链
 */
@Component
public class GrayChainExecutor {
    private final AgentChainBuilder chainBuilder;
    private final int grayRatio;
    private final Set<String> whiteList;

    public GrayChainExecutor(AgentChainBuilder chainBuilder,
                             @Value("${agent.gray.ratio:10}") int grayRatio,
                             @Value("${agent.gray.white-list:test_task_001}") String whiteListStr) {
        this.chainBuilder = chainBuilder;
        this.grayRatio = grayRatio;
        this.whiteList = Set.of(whiteListStr.split(","));
    }

    /**
     * 执行灰度能力链
     */
    public AgentResult executeGray(List<String> grayTags, List<String> stableTags, SubTask task) {
        // 白名单任务优先使用灰度链
        if (whiteList.contains(task.getTaskId())) {
            return chainBuilder.buildChain(grayTags).execute(task);
        }
        // 按比例随机分配灰度流量
        int hash = Math.abs(task.getTaskId().hashCode() % 100);
        if (hash < grayRatio) {
            return chainBuilder.buildChain(grayTags).execute(task);
        }
        // 默认使用稳定链
        return chainBuilder.buildChain(stableTags).execute(task);
    }
}
```

**实现位置**：
- `pangea.hiagent.agent.data.AgentChain`
- `pangea.hiagent.agent.data.AgentChainBuilder`
- `pangea.hiagent.agent.data.GrayChainExecutor`

### 3.5 从Agent熔断与重试增强
```java
import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.circuitbreaker.CircuitBreakerRegistry;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryRegistry;

/**
 * 带熔断和重试能力的从Agent包装类
 */
public class CircuitBreakerSlaveAgent implements SlaveAgent {
    private final SlaveAgent delegate;
    private final CircuitBreaker circuitBreaker;
    private final Retry retry;

    public CircuitBreakerSlaveAgent(SlaveAgent delegate,
                                    CircuitBreakerRegistry circuitBreakerRegistry,
                                    RetryRegistry retryRegistry) {
        this.delegate = delegate;
        String agentName = delegate.getClass().getSimpleName();
        // 熔断配置：默认失败率50%触发熔断，熔断时间10秒
        this.circuitBreaker = circuitBreakerRegistry.circuitBreaker(agentName);
        // 重试配置：默认3次重试，指数退避策略
        this.retry = retryRegistry.retry(agentName);
    }

    @Override
    public AgentResult execute(SubTask subTask) {
        return Retry.decorateSupplier(retry,
                CircuitBreaker.decorateSupplier(circuitBreaker,
                        () -> delegate.execute(subTask))).get();
    }
}

/**
 * 从Agent自动配置类，为所有从Agent添加熔断重试能力
 */
@Configuration
public class SlaveAgentAutoConfig {
    @Bean
    public List<SlaveAgent> slaveAgents(List<SlaveAgent> rawAgents,
                                        CircuitBreakerRegistry circuitBreakerRegistry,
                                        RetryRegistry retryRegistry) {
        return rawAgents.stream()
                .map(agent -> new CircuitBreakerSlaveAgent(agent, circuitBreakerRegistry, retryRegistry))
                .collect(Collectors.toList());
    }
}
```

**实现位置**：
- `pangea.hiagent.agent.data.CircuitBreakerSlaveAgent`
- `pangea.hiagent.agent.data.SlaveAgentAutoConfig`

## 四、 状态管理设计（本地持久化+故障自愈）
### 4.1 任务状态存储方案
使用**H2嵌入式数据库**持久化任务状态，无需额外部署，表结构SQL如下：
```sql
CREATE TABLE IF NOT EXISTS agent_task_status (
    task_id VARCHAR(64) PRIMARY KEY COMMENT '任务唯一标识',
    status VARCHAR(32) NOT NULL COMMENT '任务状态：READY/RUNNING/SUCCESS/FAIL',
    dependencies VARCHAR(256) COMMENT '依赖子任务ID，逗号分隔',
    result TEXT COMMENT '任务结果JSON字符串',
    result_hash VARCHAR(64) COMMENT '结果MD5哈希值，用于校验完整性',
    retry_count INT DEFAULT 0 COMMENT '已重试次数',
    timeout BIGINT COMMENT '任务超时时间戳',
    update_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '状态更新时间'
);

-- 创建任务状态索引，提升查询效率
CREATE INDEX idx_task_status ON agent_task_status(status);
CREATE INDEX idx_task_update_time ON agent_task_status(update_time);
```

**实现位置**：
- SQL文件：`src/main/resources/agent-task-status-schema.sql`
- 实体类：`pangea.hiagent.agent.data.AgentTaskStatus`
- Repository：`pangea.hiagent.agent.data.AgentTaskStatusRepository`

### 4.2 故障自愈核心机制 ✅ 已实现
1.  **超时兜底**：主Agent通过定时任务扫描`RUNNING`状态且超过超时时间的任务，自动触发重试或标记为失败
    ```java
    @Scheduled(fixedRate = 5000)
    public void checkTimeoutTasks() {
        List<String> timeoutTaskIds = taskStatusRepository.findTimeoutTasks(System.currentTimeMillis());
        for (String taskId : timeoutTaskIds) {
            retryTask(taskId);
        }
    }
    ```
2.  **崩溃恢复**：应用重启后，主Agent自动扫描`RUNNING`状态的任务，基于数据库中存储的状态数据恢复执行
    ```java
    @PostConstruct
    public void recoverUnfinishedTasks() {
        List<AgentTaskStatus> runningTasks = taskStatusRepository.findByStatus("RUNNING");
        for (AgentTaskStatus status : runningTasks) {
            MainTask task = rebuildTaskFromStatus(status);
            CompletableFuture.runAsync(() -> executeMainTask(task));
        }
    }
    ```
3.  **降级兜底**：从Agent执行失败且重试次数耗尽后，触发预设的降级逻辑，返回默认值或缓存数据
    ```java
    public interface SlaveAgent {
        default AgentResult fallback(SubTask subTask, Exception e) {
            AgentResult result = new AgentResult();
            result.setSuccess(false);
            result.setMessage("Fallback triggered: " + e.getMessage());
            result.setData(subTask.getFallbackData());
            return result;
        }
    }
    ```

**实现位置**：`pangea.hiagent.agent.data.TaskStatusService`

## 五、 功能扩展模块（插拔式+热更新）
### 5.1 模块实现原则
- 基于`@ConditionalOnProperty`注解实现模块开关控制
- 结合`@RefreshScope`支持配置热更新，无需重启应用
- 所有模块遵循“高内聚低耦合”原则，不侵入核心逻辑

### 5.2 内置核心扩展模块
| 模块名称       | 开关配置项                  | 核心功能                                                                 | 实现方式                                                                 |
|----------------|-----------------------------|--------------------------------------------------------------------------|--------------------------------------------------------------------------|
| 日志审计模块   | agent.module.log.enabled=true | 记录Agent全生命周期日志，支持按taskId追溯链路，持久化到H2数据库          | 基于Spring事件监听机制，监听AgentTaskEvent事件                           |
| 结果校验模块   | agent.module.check.enabled=true | 对比结果哈希值，校验数据传输和计算过程中的完整性，防止数据篡改           | 主Agent聚合结果后，对比实际哈希与预期哈希                                 |
| 任务可视化模块 | agent.module.monitor.enabled=true | 基于Spring Boot Admin生成任务依赖图，暴露任务执行进度监控端点            | 自定义`@Endpoint`端点，生成DOT格式依赖图，支持转换为PNG图片              |
| 降级兜底模块   | agent.module.fallback.enabled=true | 从Agent执行失败时自动触发降级逻辑，支持自定义兜底策略                    | 基于SlaveAgent接口的fallback默认方法，支持子类重写                       |

### 5.3 热更新配置实现示例
```java
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;

@Configuration
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.log", name = "enabled", havingValue = "true")
public class LogModuleAutoConfig {
    @Bean
    public AgentLogListener agentLogListener() {
        return new AgentLogListener();
    }
}
```

## 六、 部署与运维设计
### 6.1 打包部署方案
1.  **打包方式**：所有Agent模块打包为**单一Spring Boot Jar包**
2.  **运行模式切换**：通过启动参数指定运行模式
    ```bash
    # 主模式
    java -jar agent-app.jar --agent.mode=MASTER
    # 从模式
    java -jar agent-app.jar --agent.mode=SLAVE
    # 混合模式
    java -jar agent-app.jar --agent.mode=MIXED
    ```
3.  **资源控制**：默认JVM堆内存512M，可通过启动参数调整
    ```bash
    java -Xmx512m -jar agent-app.jar
    ```

### 6.2 运维监控设计
1.  **健康检查端点**：暴露`/actuator/agent-health`端点，展示主从Agent状态、熔断状态、任务执行统计
    ```java
    @Endpoint(id = "agent-health")
    @Component
    public class AgentHealthEndpoint {
        // ... 实现健康状态收集逻辑 ...
    }
    ```
2.  **链路追踪**：集成Spring Cloud Sleuth，生成全链路traceId，支持日志关联分析
3.  **配置热更新**：对接Nacos Lite轻量化配置中心，支持灰度比例、熔断阈值等参数动态调整

## 七、 可靠性保障清单
| 保障能力               | 实现方案                                                                 |
|------------------------|--------------------------------------------------------------------------|
| 消息幂等性             | 基于messageId全局唯一标识，接收方去重                                   |
| 故障隔离               | 基于Resilience4j实现从Agent级别的熔断和限流                             |
| 数据完整性             | 基于MD5哈希值校验任务结果                                               |
| 任务执行时序可控       | 基于依赖解析器实现子任务批次执行，支持并行/串行                          |
| 崩溃恢复能力           | 基于H2数据库持久化任务状态，应用重启后自动恢复未完成任务                 |
| 风险可控发布           | 基于灰度执行器实现能力链的比例/白名单灰度发布                           |
| 可观测性               | 内置日志审计、健康监控、链路追踪能力                                     |