package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.beans.factory.annotation.Autowired;

/**
 * 任务可视化模块自动配置
 * 基于@ConditionalOnProperty注解实现模块开关控制
 * 结合@RefreshScope支持配置热更新
 * 通过自定义@Endpoint暴露任务监控端点
 */
@Slf4j
@Configuration
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.monitor", name = "enabled", havingValue = "true", matchIfMissing = false)
public class MonitorModuleAutoConfig {

    /**
     * 注册任务监控端点
     */
    @Bean
    public AgentMonitorEndpoint agentMonitorEndpoint() {
        return new AgentMonitorEndpoint();
    }
}

/**
 * 任务监控端点，提供任务执行进度和依赖图
 */
@Slf4j
@Component
@Endpoint(id = "agent-monitor")
class AgentMonitorEndpoint {
    
    @Autowired
    private ModuleProperties moduleProperties;
    
    // 任务执行状态缓存
    private final Map<String, TaskExecutionStatus> taskStatusCache = new ConcurrentHashMap<>();
    
    public AgentMonitorEndpoint() {
    }
    
    /**
     * 获取任务依赖图（DOT格式）
     */
    @ReadOperation
    public Map<String, Object> getTaskDependencyGraph() {
        Map<String, Object> result = new java.util.HashMap<>();
        
        try {
            // 这里生成任务依赖图的DOT格式表示
            String dotGraph = generateTaskDependencyDotGraph();
            result.put("graphType", "dot");
            result.put("graph", dotGraph);
            result.put("timestamp", System.currentTimeMillis());
            result.put("status", "success");
        } catch (Exception e) {
            log.error("生成任务依赖图失败", e);
            result.put("status", "error");
            result.put("error", e.getMessage());
        }
        
        return result;
    }
    
    /**
     * 获取任务执行统计信息
     */
    @ReadOperation
    public Map<String, Object> getTaskStatistics() {
        Map<String, Object> result = new java.util.HashMap<>();
        
        try {
            // 统计任务执行情况
            Map<String, Long> stats = calculateTaskStatistics();
            result.put("statistics", stats);
            result.put("timestamp", System.currentTimeMillis());
            result.put("status", "success");
        } catch (Exception e) {
            log.error("获取任务统计信息失败", e);
            result.put("status", "error");
            result.put("error", e.getMessage());
        }
        
        return result;
    }
    
    /**
     * 生成任务依赖图的DOT格式表示
     */
    private String generateTaskDependencyDotGraph() {
        StringBuilder dot = new StringBuilder();
        dot.append("digraph TaskDependencies {\n");
        dot.append("  rankdir=TB;\n"); // 从上到下布局
        dot.append("  node [shape=box, style=filled, color=lightblue];\n");
        
        // 这里应该根据实际的任务依赖关系生成图
        // 示例：添加节点和边
        dot.append("  MainTask [label=\"主任务\\n(进行中)\"];\n");
        dot.append("  SubTask1 [label=\"子任务1\\n(已完成)\"];\n");
        dot.append("  SubTask2 [label=\"子任务2\\n(等待中)\"];\n");
        dot.append("  SubTask3 [label=\"子任务3\\n(失败)\"];\n");
        
        dot.append("  MainTask -> SubTask1;\n");
        dot.append("  MainTask -> SubTask2;\n");
        dot.append("  MainTask -> SubTask3;\n");
        
        dot.append("}\n");
        
        return dot.toString();
    }
    
    /**
     * 计算任务统计信息
     */
    private Map<String, Long> calculateTaskStatistics() {
        Map<String, Long> stats = new java.util.HashMap<>();
        
        // 这里应该从实际的任务状态存储中获取统计信息
        // 示例数据
        stats.put("totalTasks", 100L);
        stats.put("completedTasks", 85L);
        stats.put("runningTasks", 10L);
        stats.put("failedTasks", 5L);
        stats.put("successRate", 85L); // 成功率百分比
        
        return stats;
    }
    
    /**
     * 更新任务执行状态
     */
    public void updateTaskStatus(String taskId, String status, String details) {
        TaskExecutionStatus taskStatus = new TaskExecutionStatus(taskId, status, details, System.currentTimeMillis());
        taskStatusCache.put(taskId, taskStatus);
    }
    
    /**
     * 获取所有任务状态
     */
    public Map<String, TaskExecutionStatus> getAllTaskStatuses() {
        return new java.util.HashMap<>(taskStatusCache);
    }
}

/**
 * 任务执行状态类
 */
class TaskExecutionStatus {
    private final String taskId;
    private final String status; // RUNNING, COMPLETED, FAILED, WAITING
    private final String details;
    private final long timestamp;
    
    public TaskExecutionStatus(String taskId, String status, String details, long timestamp) {
        this.taskId = taskId;
        this.status = status;
        this.details = details;
        this.timestamp = timestamp;
    }
    
    // Getters
    public String getTaskId() { return taskId; }
    public String getStatus() { return status; }
    public String getDetails() { return details; }
    public long getTimestamp() { return timestamp; }
}