# 功能扩展模块（插拔式 + 热更新）系统

## 概述

本系统实现了轻量级、高可扩展的模块化架构，支持插拔式扩展和配置热更新。通过`@ConditionalOnProperty`和`@RefreshScope`注解实现模块的动态开关控制和配置热更新，所有模块遵循"高内聚低耦合"原则，不侵入核心逻辑。

## 架构设计

### 1. 模块实现原则

- **基于`@ConditionalOnProperty`注解**：实现模块开关控制
- **结合`@RefreshScope`**：支持配置热更新，无需重启应用
- **高内聚低耦合**：所有模块不侵入核心逻辑，独立运行

### 2. 内置核心扩展模块

| 模块名称 | 开关配置项 | 核心功能 | 实现方式 |
|---------|-----------|----------|----------|
| 日志审计模块 | `agent.module.log.enabled=true` | 记录Agent全生命周期日志，支持按taskId追溯链路，持久化到H2数据库 | 基于Spring事件监听机制，监听AgentTaskEvent事件 |
| 结果校验模块 | `agent.module.check.enabled=true` | 对比结果哈希值，校验数据传输和计算过程中的完整性，防止数据篡改 | 主Agent聚合结果后，对比实际哈希与预期哈希 |
| 任务可视化模块 | `agent.module.monitor.enabled=true` | 基于Spring Boot Admin生成任务依赖图，暴露任务执行进度监控端点 | 自定义`@Endpoint`端点，生成DOT格式依赖图，支持转换为PNG图片 |
| 降级兜底模块 | `agent.module.fallback.enabled=true` | 从Agent执行失败时自动触发降级逻辑，支持自定义兜底策略 | 基于SlaveAgent接口的fallback默认方法，支持子类重写 |

## 核心组件

### 1. 配置属性类

`ModuleProperties`：支持配置热更新的属性类，包含所有模块的配置项。

```java
@Data
@Component
@RefreshScope
@ConfigurationProperties(prefix = "agent.module")
public class ModuleProperties {
    private boolean enabled = true;
    private LogModuleProperties log = new LogModuleProperties();
    private CheckModuleProperties check = new CheckModuleProperties();
    private MonitorModuleProperties monitor = new MonitorModuleProperties();
    private FallbackModuleProperties fallback = new FallbackModuleProperties();
}
```

### 2. 事件机制

系统基于Spring事件机制实现模块间通信：

- `AgentTaskEvent`：定义任务生命周期事件
- 各模块通过`@EventListener`监听任务事件并执行相应逻辑

### 3. 模块自动配置

每个模块都有对应的自动配置类，使用`@ConditionalOnProperty`控制是否加载：

```java
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.check", name = "enabled", havingValue = "true", matchIfMissing = false)
public class CheckModuleAutoConfig {
    // 模块实现逻辑
}
```

## 配置说明

### application.yml 配置项

```yaml
agent:
  module:
    enabled: true  # 模块总开关
    log:
      enabled: true  # 日志审计模块开关
      retentionDays: 30  # 日志保留天数
      storagePath: "./logs"  # 日志存储路径
    check:
      enabled: true  # 结果校验模块开关
      strictMode: false  # 严格模式
      algorithm: "SHA-256"  # 校验算法
    monitor:
      enabled: true  # 任务可视化模块开关
      refreshInterval: 5000  # 监控刷新间隔（毫秒）
      enableGraph: true  # 是否启用依赖图
    fallback:
      enabled: true  # 降级兜底模块开关
      maxRetries: 3  # 最大重试次数
      retryDelay: 1000  # 重试延迟（毫秒）
      defaultStrategy: "cache"  # 默认降级策略
```

## 热更新实现

### 1. 依赖配置

在`pom.xml`中添加必要的依赖：

```xml
<!-- Spring Boot Actuator for monitoring endpoints -->
<dependency>
    <groupId>org.springframework.boot</groupId>
    <artifactId>spring-boot-starter-actuator</artifactId>
</dependency>

<!-- Spring Cloud Context for @RefreshScope -->
<dependency>
    <groupId>org.springframework.cloud</groupId>
    <artifactId>spring-cloud-context</artifactId>
</dependency>
```

### 2. 监控端点

通过Spring Boot Actuator的`/actuator/refresh`端点实现配置热更新。

## 模块管理

### 1. 模块生命周期

- **启动时**：根据配置决定是否加载模块
- **运行时**：通过事件机制处理任务状态变化
- **配置更新时**：通过`@RefreshScope`重新加载配置

### 2. 事件驱动机制

主Agent在任务执行过程中发布以下事件：

- `STARTED`：任务开始
- `COMPLETED`：任务完成
- `FAILED`：任务失败
- `SUBTASK_STARTED`：子任务开始
- `SUBTASK_COMPLETED`：子任务完成
- `SUBTASK_FAILED`：子任务失败

各模块监听相关事件并执行相应处理逻辑。

## 扩展指南

### 1. 添加新模块

要添加新模块，需要：

1. 创建模块配置属性类（可选）
2. 创建模块自动配置类
3. 使用`@ConditionalOnProperty`控制模块开关
4. 使用`@RefreshScope`支持热更新
5. 实现模块业务逻辑

### 2. 模块间通信

推荐使用Spring事件机制进行模块间通信，避免模块间的直接依赖。

## 测试验证

系统包含`ModuleTestService`用于验证所有模块功能和热更新能力。

## 总结

本模块化扩展系统实现了：

1. **插拔式架构**：模块可独立开启/关闭
2. **热更新能力**：配置修改无需重启应用
3. **事件驱动**：模块间松耦合通信
4. **统一配置管理**：集中管理所有模块配置
5. **高可扩展性**：易于添加新模块