package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.service.AgentChatService;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.web.dto.ChatRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;

/**
 * Agent 对话控制器
 * 职责：接收HTTP请求并转发给AgentChatService处理
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
public class AgentChatController {
    
    private final AgentChatService agentChatService;
    
    public AgentChatController(AgentChatService agentChatService) {
        this.agentChatService = agentChatService;
    }
    
    /**
     * 流式对话接口
     * 
     * @param agentId Agent ID
     * @param chatRequest 对话请求
     * @param response HTTP响应
     * @return SSE emitter
     */
    @PostMapping("/chat-stream")
    public SseEmitter chatStream(
            @RequestParam @NotBlank(message = "Agent ID不能为空") String agentId,
            @RequestBody @Valid ChatRequest chatRequest,
            HttpServletResponse response) {
        log.info("接收到流式对话请求，AgentId: {}", agentId);
        
        // 检查用户权限
        String userId = UserUtils.getCurrentUserId();
        if (userId == null) {
            log.warn("用户未认证，无法执行Agent对话");
            throw new org.springframework.security.access.AccessDeniedException("用户未认证");
        }
        
        return agentChatService.handleChatStream(agentId, chatRequest, response);
    }
}