package pangea.hiagent.web.service;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Tool;
import pangea.hiagent.web.repository.ToolRepository;

import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;

/**
 * 工具服务类
 * 提供工具管理相关业务逻辑
 */
@Slf4j
@Service
public class ToolService extends ServiceImpl<ToolRepository, Tool> {
    
    private final ToolRepository toolRepository;
    
    public ToolService(ToolRepository toolRepository) {
        this.toolRepository = toolRepository;
    }
    
    /**
     * 创建工具
     * @param tool 工具对象
     * @param userId 用户ID
     * @return 创建后的工具对象
     */
    public Tool createTool(Tool tool, String userId) {
        // 设置ID
        if (tool.getId() == null || tool.getId().isEmpty()) {
            tool.setId(UUID.randomUUID().toString());
        }
        
        // 设置所有者
        tool.setOwner(userId);
        
        // 设置创建和更新信息
        tool.setCreatedBy(userId);
        tool.setUpdatedBy(userId);
        tool.setCreatedAt(LocalDateTime.now());
        tool.setUpdatedAt(LocalDateTime.now());
        
        // 设置默认状态
        if (tool.getStatus() == null || tool.getStatus().isEmpty()) {
            tool.setStatus("active");
        }
        
        // 保存到数据库
        toolRepository.insert(tool);
        log.info("创建工具成功，工具ID: {}", tool.getId());
        
        return tool;
    }
    
    /**
     * 更新工具
     * @param tool 工具对象
     * @param userId 用户ID
     * @return 更新后的工具对象
     */
    public Tool updateTool(Tool tool, String userId) {
        // 检查工具是否存在
        Tool existingTool = toolRepository.selectById(tool.getId());
        if (existingTool == null) {
            throw new RuntimeException("工具不存在");
        }
        
        // 检查权限
        if (!existingTool.getOwner().equals(userId)) {
            throw new RuntimeException("无权限修改此工具");
        }
        
        // 更新信息
        tool.setUpdatedBy(userId);
        tool.setUpdatedAt(LocalDateTime.now());
        
        // 更新数据库
        toolRepository.updateById(tool);
        log.info("更新工具成功，工具ID: {}", tool.getId());
        
        return tool;
    }
    
    /**
     * 删除工具
     * @param toolId 工具ID
     * @param userId 用户ID
     */
    public void deleteTool(String toolId, String userId) {
        // 检查工具是否存在
        Tool existingTool = toolRepository.selectById(toolId);
        if (existingTool == null) {
            throw new RuntimeException("工具不存在");
        }
        
        // 检查权限
        if (!existingTool.getOwner().equals(userId)) {
            throw new RuntimeException("无权限删除此工具");
        }
        
        // 逻辑删除
        Tool tool = new Tool();
        tool.setId(toolId);
        tool.setDeleted(1);
        tool.setUpdatedBy(userId);
        tool.setUpdatedAt(LocalDateTime.now());
        
        toolRepository.updateById(tool);
        log.info("删除工具成功，工具ID: {}", toolId);
    }
    
    /**
     * 根据ID获取工具
     * @param toolId 工具ID
     * @param userId 用户ID
     * @return 工具对象
     */
    public Tool getToolById(String toolId, String userId) {
        Tool tool = toolRepository.selectById(toolId);
        if (tool == null || tool.getDeleted() == 1) {
            return null;
        }
        
        // 检查权限
        if (!tool.getOwner().equals(userId)) {
            throw new RuntimeException("无权限访问此工具");
        }
        
        return tool;
    }
    
    /**
     * 获取用户的所有工具
     * @param userId 用户ID
     * @return 工具列表
     */
    public List<Tool> getUserTools(String userId) {
        List<Tool> userTools = toolRepository.findByOwner(userId);
        
        // 确保displayName字段正确设置
        for (Tool tool : userTools) {
            if (tool.getDisplayName() == null || tool.getDisplayName().trim().isEmpty()) {
                tool.setDisplayName(tool.getName());
            }
        }
        
        // 如果用户没有工具，返回所有公共工具
        if (userTools.isEmpty()) {
            return getAllTools(); // 使用增强版的getAllTools方法
        }
        
        return userTools;
    }
    
    /**
     * 获取所有工具
     * @return 工具列表
     */
    public List<Tool> getAllTools() {
        List<Tool> tools = toolRepository.findAllActive();
        
        // 确保displayName字段正确设置，如果没有则使用name作为displayName
        for (Tool tool : tools) {
            if (tool.getDisplayName() == null || tool.getDisplayName().trim().isEmpty()) {
                tool.setDisplayName(tool.getName());
            }
        }
        
        return tools;
    }
    
    /**
     * 根据状态获取用户工具
     * @param userId 用户ID
     * @param status 工具状态
     * @return 工具列表
     */
    public List<Tool> getUserToolsByStatus(String userId, String status) {
        return toolRepository.findByOwnerAndStatus(userId, status);
    }
    
    /**
     * 根据名称获取用户工具
     * @param name 工具名称
     * @param userId 用户ID
     * @return 工具对象
     */
    public Tool getToolByName(String name, String userId) {
        return toolRepository.findByNameAndOwner(name, userId);
    }
}