package pangea.hiagent.rag;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.document.Document;
import org.springframework.ai.vectorstore.SearchRequest;
import org.springframework.ai.vectorstore.VectorStore;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.config.AppConfig;
import pangea.hiagent.model.Agent;
import pangea.hiagent.service.AgentService;

import java.util.List;
import java.util.Map;

/**
 * RAG服务类
 * 负责文档检索、相似度匹配和RAG增强生成
 */
@Slf4j
@Service
public class RagService {
    
    @Autowired(required = false) // 允许为空
    private VectorStore vectorStore;
    
    @Autowired
    private AppConfig appConfig;
    
    @Autowired
    private AgentService agentService;
    
    /**
     * 根据查询检索相关文档片段
     *
     * @param query 查询内容
     * @param collectionIds 集合ID列表
     * @param topK 返回结果数量
     * @param scoreThreshold 相似度阈值
     * @return 相关文档片段列表
     */
    public List<Document> searchDocuments(String query, List<String> collectionIds,
        Integer topK, Double scoreThreshold) {
        log.info("开始检索文档，查询: {}, 集合数: {}, topK: {}, 阈值: {}", 
                query, collectionIds != null ? collectionIds.size() : 0, topK, scoreThreshold);
        
        // 检查vectorStore是否可用
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法执行文档检索");
            return List.of();
        }
        
        try {
            if (collectionIds != null && !collectionIds.isEmpty()) {
            }
            
            // 设置检索参数
            SearchRequest.Builder requestBuilder = SearchRequest.builder()
                    .query(query)
                    .topK(topK != null ? topK : appConfig.getRag().getTopK())
                    .similarityThreshold(scoreThreshold != null ? scoreThreshold : appConfig.getRag().getScoreThreshold());
            
            // 注意：在当前Spring AI版本中，可能不支持动态过滤器设置
            // 我们暂时移除过滤器设置，以确保代码能够编译通过
            SearchRequest request = requestBuilder.build();
            
            // 执行检索
            List<Document> results = vectorStore.similaritySearch(request);
            log.info("检索完成，返回 {} 个结果", results.size());
            
            return results;
        } catch (Exception e) {
            log.error("文档检索失败", e);
            throw new RuntimeException("文档检索失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 根据Agent配置检索相关文档
     *
     * @param agent Agent对象
     * @param query 查询内容
     * @return 相关文档片段列表
     */
    public List<Document> searchDocumentsByAgent(Agent agent, String query) {
        log.info("根据Agent配置检索文档，Agent ID: {}, 查询: {}", agent.getId(), query);
        
        // 检查vectorStore是否可用
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法执行文档检索");
            return List.of();
        }
        
        try {
            // 获取Agent关联的RAG集合
            String collectionId = agent.getRagCollectionId();
            if (collectionId == null || collectionId.isEmpty()) {
                log.warn("Agent未配置RAG集合，Agent ID: {}", agent.getId());
                return List.of();
            }
            
            // 使用Agent特定的RAG配置
            Integer topK = agent.getRagTopK();
            Double scoreThreshold = agent.getRagScoreThreshold();
            
            // 执行检索
            return searchDocuments(query, List.of(collectionId), 
                topK != null ? topK : appConfig.getRag().getTopK(), 
                scoreThreshold != null ? scoreThreshold : appConfig.getRag().getScoreThreshold());
        } catch (Exception e) {
            log.error("根据Agent配置检索文档失败，Agent ID: {}", agent.getId(), e);
            throw new RuntimeException("文档检索失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 构建RAG增强的提示词
     *
     * @param query 用户查询
     * @param documents 相关文档片段
     * @return 增强的提示词
     */
    public String buildRagPrompt(String query, List<Document> documents) {
        if (documents == null || documents.isEmpty()) {
            return query;
        }
        
        StringBuilder prompt = new StringBuilder();
        prompt.append("请根据以下文档内容回答问题：\n\n");
        prompt.append("问题：").append(query).append("\n\n");
        prompt.append("参考文档：\n");
        
        for (int i = 0; i < documents.size(); i++) {
            Document doc = documents.get(i);
            prompt.append("文档 ").append(i + 1).append(":\n");
            prompt.append(doc.getText()).append("\n\n");
        }
        
        prompt.append("请根据以上文档内容回答问题，如果文档中没有相关信息，请说明无法根据提供的文档回答该问题。\n\n");
        prompt.append("回答：");
        
        return prompt.toString();
    }
    
    /**
     * 构建基于Agent配置的RAG增强提示词
     *
     * @param agent Agent对象
     * @param query 用户查询
     * @param documents 相关文档片段
     * @return 增强的提示词
     */
    public String buildRagPromptByAgent(Agent agent, String query, List<Document> documents) {
        if (documents == null || documents.isEmpty()) {
            return query;
        }
        
        // 使用Agent特定的RAG提示词模板
        String ragPromptTemplate = agent.getRagPromptTemplate();
        if (ragPromptTemplate == null || ragPromptTemplate.isEmpty()) {
            // 使用默认模板
            return buildRagPrompt(query, documents);
        }
        
        // 使用Agent特定的模板
        StringBuilder prompt = new StringBuilder();
        prompt.append(ragPromptTemplate).append("\n\n");
        prompt.append("问题：").append(query).append("\n\n");
        prompt.append("参考文档：\n");
        
        for (int i = 0; i < documents.size(); i++) {
            Document doc = documents.get(i);
            prompt.append("文档 ").append(i + 1).append(":\n");
            prompt.append(doc.getText()).append("\n\n");
        }
        
        prompt.append("请根据以上文档内容回答问题，如果文档中没有相关信息，请说明无法根据提供的文档回答该问题。\n\n");
        prompt.append("回答：");
        
        return prompt.toString();
    }
    
    /**
     * RAG增强问答
     *
     * @param agent Agent对象
     * @param query 用户查询
     * @return 增强的回答
     */
    public String ragQa(Agent agent, String query) {
        log.info("开始RAG增强问答，Agent ID: {}, 查询: {}", agent.getId(), query);
        
        // 检查vectorStore是否可用
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法执行RAG增强问答");
            return null;
        }
        
        try {
            // 检查是否启用RAG
            if (agent.getEnableRag() == null || !agent.getEnableRag()) {
                log.warn("Agent未启用RAG功能，Agent ID: {}", agent.getId());
                return null;
            }
            
            // 检索相关文档
            List<Document> documents = searchDocumentsByAgent(agent, query);
            if (documents.isEmpty()) {
                log.info("未检索到相关文档，Agent ID: {}", agent.getId());
                return null;
            }
            
            // 构建增强提示词
            String enhancedPrompt = buildRagPromptByAgent(agent, query, documents);
            
            // 获取ChatModel
            ChatModel chatModel = agentService.getChatModelForAgent(agent);
            
            // 调用模型生成回答
            String response = chatModel.call(enhancedPrompt);
            log.info("RAG增强问答完成，Agent ID: {}", agent.getId());
            
            return response;
        } catch (Exception e) {
            log.error("RAG增强问答失败，Agent ID: {}", agent.getId(), e);
            throw new RuntimeException("RAG增强问答失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 获取文档检索统计信息
     *
     * @param collectionIds 集合ID列表
     * @return 统计信息
     */
    public Map<String, Object> getRetrievalStats(List<String> collectionIds) {
        log.info("获取文档检索统计信息，集合数: {}", collectionIds != null ? collectionIds.size() : 0);
        
        try {
            // 获取向量存储统计信息
            // 这里可以根据实际需求实现具体的统计逻辑
            // 暂时返回基本统计信息
            return Map.of(
                "collectionCount", collectionIds != null ? collectionIds.size() : 0,
                "timestamp", System.currentTimeMillis()
            );        } catch (Exception e) {
            log.error("获取文档检索统计信息失败", e);
            throw new RuntimeException("获取统计信息失败: " + e.getMessage(), e);
        }
    }
}